/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"
#include <commctrl.h>
#include <windowsx.h>


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:ダイアログボックスのコントロールを有効化または無効化する
       hDlg,ダイアログボックス
        nID,コントロールID
    fEnable,TRUE:有効,FALSE:無効
        RET,TRUE:無効だった,FALSE:有効だった                                */
BOOL WINAPI
EnableDlgItem (HWND hDlg,
               int  nID,
               BOOL fEnable)
{
  HWND hWnd;

  hWnd = GetDlgItem (hDlg, nID);
  return !hWnd || EnableWindow (hWnd, fEnable);
}


/*  ja:ダイアログボックスのコントロールの文字数を取得する
    hDlg,ダイアログボックス
     nID,コントロールID
     RET,文字数,0:エラー                                                    */
int WINAPI
GetDlgItemTextLengthW (HWND hDlg,
                       int  nID)
{
  HWND hWnd;

  hWnd = GetDlgItem (hDlg, nID);
  return hWnd ? GetWindowTextLengthW (hWnd) : 0;
}


/*  ja:ダイアログボックスのコントロールの文字数を取得する
    hDlg,ダイアログボックス
     nID,コントロールID
     RET,文字数,0:エラー                                                    */
int WINAPI
GetDlgItemTextLengthA (HWND hDlg,
                       int  nID)
{
  HWND hWnd;

  hWnd = GetDlgItem (hDlg, nID);
  return hWnd ? GetWindowTextLengthA (hWnd) : 0;
}


/*  ja:ダイアログボックスのコントロールの文字を取得する
    hDlg,ダイアログボックス
     nID,コントロールID
     RET,文字数,0:エラー                                                    */
LPWSTR WINAPI
GetDlgItemTextNewW (HWND hDlg,
                    int  nID)
{
  HWND hWnd;

  hWnd = GetDlgItem (hDlg, nID);
  return hWnd ? GetWindowTextNewW (hWnd) : NULL;
}


/*  ja:ダイアログボックスのコントロールの文字を取得する
    hDlg,ダイアログボックス
     nID,コントロールID
     RET,文字数,0:エラー                                                    */
LPSTR WINAPI
GetDlgItemTextNewA (HWND hDlg,
                    int  nID)
{
  HWND hWnd;

  hWnd = GetDlgItem (hDlg, nID);
  return hWnd ? GetWindowTextNewA (hWnd) : NULL;
}


/*  ja:ダイアログボックスのコントロールの表示状態を設定する
        hDlg,ダイアログボックス
         nID,コントロールID
    nCmdShow,表示方法
         RET,TRUE:表示されていた,FALSE:表示されていなかった                 */
BOOL WINAPI
ShowDlgItem (HWND hDlg,
             int  nID,
             int  nCmdShow)
{
  HWND hWnd;

  hWnd = GetDlgItem (hDlg, nID);
  return hWnd && ShowWindow (hWnd, nCmdShow);
}


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:ダイアログボックスのコントロールの位置情報を初期化する
            hDlg,ダイアログボックス
    lpDlgItemPos,ダイアログ情報
          nCount,コントロールの数
             RET,TRUE:正常終了,FALSE:エラー                                 */
BOOL WINAPI
InitDlgItemPosition (HWND              hDlg,
                     LPDLGITEMPOSITION lpDlgItemPos,
                     int               nCount)
{
  int i;
  BOOL fResult;
  RECT rcClient;

  /* ja:クライアント領域のサイズ */
  fResult = GetClientRect (hDlg, &rcClient);
  for (i = 0; fResult && i < nCount; i++)
    {
      HWND hWnd;
      RECT rc;

      hWnd = GetDlgItem (hDlg, lpDlgItemPos[i].nID);
      fResult = hWnd && GetWindowRect (hWnd, &rc)
                     && ScreenToClient (hDlg, (LPPOINT)&rc)
                     && ScreenToClient (hDlg, (LPPOINT)&rc + 1);
      if (fResult)
        {
          lpDlgItemPos[i].nLeft = rc.left;
          lpDlgItemPos[i].nTop = rc.top;
          lpDlgItemPos[i].nRight = rcClient.right - rc.right;
          lpDlgItemPos[i].nBottom = rcClient.bottom - rc.bottom;
          lpDlgItemPos[i].nWidth = rc.right - rc.left;
          lpDlgItemPos[i].nHeight = rc.bottom - rc.top;
        }
    }
  return fResult;
}


/*  ja:ダイアログボックスのコントロールの位置を設定する
            hDlg,ダイアログボックス
    lpDlgItemPos,ダイアログ情報
          nCount,コントロールの数
             RET,TRUE:正常終了,FALSE:エラー                                 */
BOOL WINAPI
SetDlgItemPosition (HWND              hDlg,
                    LPDLGITEMPOSITION lpDlgItemPos,
                    int               nCount)
{
  int i;
  BOOL fResult;
  HWND hWnd;
  RECT rcClient;

  /* ja:クライアント領域のサイズ */
  fResult = GetClientRect (hDlg, &rcClient);
  for (i = 0; fResult && i < nCount; i++)
    fResult = (hWnd = GetDlgItem (hDlg, lpDlgItemPos[i].nID))
      && MoveWindow (hWnd,
      lpDlgItemPos[i].fLeft ? lpDlgItemPos[i].nLeft
        : rcClient.right - lpDlgItemPos[i].nWidth - lpDlgItemPos[i].nRight,
      lpDlgItemPos[i].fTop ? lpDlgItemPos[i].nTop
        : rcClient.bottom - lpDlgItemPos[i].nHeight - lpDlgItemPos[i].nBottom,
      lpDlgItemPos[i].fLeft && lpDlgItemPos[i].fRight
        ? rcClient.right - lpDlgItemPos[i].nLeft - lpDlgItemPos[i].nRight
                                                    : lpDlgItemPos[i].nWidth,
      lpDlgItemPos[i].fTop && lpDlgItemPos[i].fBottom
        ? rcClient.bottom - lpDlgItemPos[i].nTop - lpDlgItemPos[i].nBottom
                                                    : lpDlgItemPos[i].nHeight,
                                                                        TRUE);
  return fResult;
}


static LPDLGTEMPLATE WINAPI
MakeDlgGUIFontW (HINSTANCE hInstance,
                 LPCWSTR   lpTemplate)
{
  HDC hDC;
  LPDLGTEMPLATE lpDialogTemplate = NULL;

  /* ja:フォント */
  hDC = CreateCompatibleDC (NULL);
  if (hDC)
    {
      int i, nLogPixel;
      SHORT sPoint = 0;
      LPWSTR lpszFont = NULL;
      const static int nObject[] = {DEFAULT_GUI_FONT, ANSI_VAR_FONT,
            SYSTEM_FONT, ANSI_FIXED_FONT, SYSTEM_FIXED_FONT, OEM_FIXED_FONT};

      nLogPixel = GetDeviceCaps (hDC, LOGPIXELSY);
      for (i = 0; i < n_elements (nObject) && sPoint <= 0; i++)
        {
          HFONT hFont;

          hFont = GetStockObject (nObject[i]);
          if (hFont)
            {
              TEXTMETRICW tm;

              hFont = SelectObject (hDC, hFont);
              if (GetTextMetricsW (hDC, &tm))
                {
                  int n;

                  n = GetTextFaceW (hDC, 0, NULL);
                  if (n > 0)
                    {
                      lpszFont = MemoryReAlloc (lpszFont, n * sizeof (WCHAR));
                      if (GetTextFaceW (hDC, n, lpszFont) > 0)
                        sPoint = MulDiv (tm.tmHeight, 72, nLogPixel);
                    }
                }
              hFont = SelectObject (hDC, hFont);
            }
        }
      if (DeleteDC (hDC))
        {
          DWORD dwSize;
          HGLOBAL hGlobal;
          HRSRC hRes;
          LPDLGTEMPLATE lpMemory;

          if ((hRes = FindResourceW (hInstance, lpTemplate,
                                                        (LPCWSTR)RT_DIALOG))
            && (dwSize = SizeofResource (hInstance, hRes)) > 0
            && (hGlobal = LoadResource (hInstance, hRes))
            && (lpMemory = LockResource (hGlobal))
            && (lpDialogTemplate = MemoryAlloc (dwSize
                +(lstrlenW (lpszFont) + 1) * sizeof (WCHAR) + sizeof (SHORT))))
            {
              LPVOID p, q;

              p = lpDialogTemplate;
              q = lpMemory;
              /* ja:構造体をコピー */
              *lpDialogTemplate = *lpMemory;
              dwSize -= sizeof (DLGTEMPLATE);
              ((LPDLGTEMPLATE)p)++;
              ((LPDLGTEMPLATE)q)++;
              for (i = 0; i < 3; i++)
                if (((LPBYTE)q)[0] == 0xff && ((LPBYTE)q)[1] == 0xff)
                  {
                    *((LPDWORD)p) = *((LPDWORD)q);
                    ((LPDWORD)p)++;
                    ((LPDWORD)q)++;
                    dwSize -= sizeof (DWORD);
                  }
                else
                  {
                    int nLength;

                    nLength = (lstrlenW (q) + 1) * sizeof (WCHAR);
                    MemoryCopy (p, q, nLength);
                    (LPBYTE)p += nLength;
                    (LPBYTE)q += nLength;
                    dwSize -= nLength;
                  }
              if (sPoint > 0)
                {
                  /* ja:フォントの調整 */
                  int nLength;

                  sPoint = min (GetSystemMetrics (SM_CXSCREEN) * 576
                                / (lpDialogTemplate->cx * nLogPixel), sPoint);
                  sPoint = min (GetSystemMetrics (SM_CYSCREEN) * 576
                                / (lpDialogTemplate->cy * nLogPixel), sPoint);
                  if (sPoint < 1)
                    sPoint = 1;
                  /* ja:フォントを設定する */
                  lpDialogTemplate->style |= DS_SETFONT;
                  *(SHORT *)p = sPoint;
                  ((SHORT *)p)++;
                  nLength = (lstrlenW (lpszFont) + 1) * sizeof (WCHAR);
                  MemoryCopy (p, lpszFont, nLength);
                  (LPBYTE)p += nLength;
                  /* ja:元のフォントを破棄する */
                  if (((LPDLGTEMPLATE)lpMemory)->style & DS_SETFONT)
                    {
                      ((SHORT *)q)++;
                      dwSize -= sizeof (SHORT);
                      if (((LPBYTE)q)[0] == 0xff && ((LPBYTE)q)[1] == 0xff)
                        {
                          ((LPDWORD)q)++;
                          dwSize -= sizeof (DWORD);
                        }
                      else
                        {
                          nLength = (lstrlenW (q) + 1) * sizeof (WCHAR);
                          (LPBYTE)q += nLength;
                          dwSize -= nLength;
                        }
                    }
                }
              /* ja:残りをコピーする */
              MemoryCopy (p, q, dwSize);
            }
        }
      MemoryFree (lpszFont);
    }
  return lpDialogTemplate;
}


static LPDLGTEMPLATE WINAPI
MakeDlgGUIFontA (HINSTANCE hInstance,
                 LPCSTR    lpTemplate)
{
  HDC hDC;
  LPDLGTEMPLATE lpDialogTemplate = NULL;

  /* ja:フォント */
  hDC = CreateCompatibleDC (NULL);
  if (hDC)
    {
      int i, nLogPixel;
      SHORT sPoint = 0;
      LPWSTR lpszFont = NULL;
      const static int nObject[] = {DEFAULT_GUI_FONT, ANSI_VAR_FONT,
            SYSTEM_FONT, ANSI_FIXED_FONT, SYSTEM_FIXED_FONT, OEM_FIXED_FONT};

      nLogPixel = GetDeviceCaps (hDC, LOGPIXELSY);
      for (i = 0; i < n_elements (nObject) && sPoint <= 0; i++)
        {
          HFONT hFont;

          hFont = GetStockObject (nObject[i]);
          if (hFont)
            {
              TEXTMETRICW tm;

              hFont = SelectObject (hDC, hFont);
              if (GetTextMetricsW (hDC, &tm))
                {
                  int n;

                  n = GetTextFaceW (hDC, 0, NULL);
                  if (n > 0)
                    {
                      lpszFont = MemoryReAlloc (lpszFont, n * sizeof (WCHAR));
                      if (GetTextFaceW (hDC, n, lpszFont) > 0)
                        sPoint = MulDiv (tm.tmHeight, 72, nLogPixel);
                    }
                }
              hFont = SelectObject (hDC, hFont);
            }
        }
      if (DeleteDC (hDC))
        {
          DWORD dwSize;
          HGLOBAL hGlobal;
          HRSRC hRes;
          LPDLGTEMPLATE lpMemory;
          if ((hRes = FindResourceA (hInstance, lpTemplate, (LPCSTR)RT_DIALOG))
            && (dwSize = SizeofResource (hInstance, hRes)) > 0
            && (hGlobal = LoadResource (hInstance, hRes))
            && (lpMemory = LockResource (hGlobal))
            && (lpDialogTemplate = MemoryAlloc (dwSize
                +(lstrlenW (lpszFont) + 1) * sizeof (WCHAR) + sizeof (SHORT))))
            {
              LPVOID p, q;

              p = lpDialogTemplate;
              q = lpMemory;
              /* ja:構造体をコピー */
              *lpDialogTemplate = *lpMemory;
              dwSize -= sizeof (DLGTEMPLATE);
              ((LPDLGTEMPLATE)p)++;
              ((LPDLGTEMPLATE)q)++;
              for (i = 0; i < 3; i++)
                if (((LPBYTE)q)[0] == 0xff && ((LPBYTE)q)[1] == 0xff)
                  {
                    *((LPDWORD)p) = *((LPDWORD)q);
                    ((LPDWORD)p)++;
                    ((LPDWORD)q)++;
                    dwSize -= sizeof (DWORD);
                  }
                else
                  {
                    int nLength;

                    nLength = (lstrlenW (q) + 1) * sizeof (WCHAR);
                    MemoryCopy (p, q, nLength);
                    (LPBYTE)p += nLength;
                    (LPBYTE)q += nLength;
                    dwSize -= nLength;
                  }
              if (sPoint > 0)
                {
                  /* ja:フォントの調整 */
                  int nLength;

                  sPoint = min (GetSystemMetrics (SM_CXSCREEN) * 576
                                / (lpDialogTemplate->cx * nLogPixel), sPoint);
                  sPoint = min (GetSystemMetrics (SM_CYSCREEN) * 576
                                / (lpDialogTemplate->cy * nLogPixel), sPoint);
                  if (sPoint < 1)
                    sPoint = 1;
                  /* ja:フォントを設定する */
                  lpDialogTemplate->style |= DS_SETFONT;
                  *(SHORT *)p = sPoint;
                  ((SHORT *)p)++;
                  nLength = (lstrlenW (lpszFont) + 1) * sizeof (WCHAR);
                  MemoryCopy (p, lpszFont, nLength);
                  (LPBYTE)p += nLength;
                  /* ja:元のフォントを破棄する */
                  if (((LPDLGTEMPLATE)lpMemory)->style & DS_SETFONT)
                    {
                      ((SHORT *)q)++;
                      dwSize -= sizeof (SHORT);
                      if (((LPBYTE)q)[0] == 0xff && ((LPBYTE)q)[1] == 0xff)
                        {
                          ((LPDWORD)q)++;
                          dwSize -= sizeof (DWORD);
                        }
                      else
                        {
                          nLength = (lstrlenW (q) + 1) * sizeof (WCHAR);
                          (LPBYTE)q += nLength;
                          dwSize -= nLength;
                        }
                    }
                }
              /* ja:残りをコピーする */
              MemoryCopy (p, q, dwSize);
            }
        }
      MemoryFree (lpszFont);
    }
  return lpDialogTemplate;
}


/*  ja:モードレスダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
          lParam,初期化値
             RET,ダイアログボックスのハンドル                               */
HWND WINAPI
CreateDialogParamGUIW (HINSTANCE hInstance,
                       LPCWSTR   lpTemplate,
                       HWND      hWndParent,
                       DLGPROC   lpDialogFunc,
                       LPARAM    lParam)
{
  HWND hDlg;
  LPDLGTEMPLATE lpDialogTemplate;

  lpDialogTemplate = MakeDlgGUIFontW (hInstance, lpTemplate);
  if (lpDialogTemplate)
    {
      hDlg = CreateDialogIndirectParamW (hInstance, lpDialogTemplate,
                                            hWndParent, lpDialogFunc, lParam);
      MemoryFree (lpDialogTemplate);
    }
  else
    {
      hDlg = NULL;
    }
  return hDlg;
}


/*  ja:モードレスダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
          lParam,初期化値
             RET,ダイアログボックスのハンドル                               */
HWND WINAPI
CreateDialogParamGUIA (HINSTANCE hInstance,
                       LPCSTR    lpTemplate,
                       HWND      hWndParent,
                       DLGPROC   lpDialogFunc,
                       LPARAM    lParam)
{
  HWND hDlg;
  LPDLGTEMPLATE lpDialogTemplate;

  lpDialogTemplate = MakeDlgGUIFontA (hInstance, lpTemplate);
  if (lpDialogTemplate)
    {
      hDlg = CreateDialogIndirectParamA (hInstance, lpDialogTemplate,
                                            hWndParent, lpDialogFunc, lParam);
      MemoryFree (lpDialogTemplate);
    }
  else
    {
      hDlg = NULL;
    }
  return hDlg;
}


/*  ja:モードレスダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
             RET,ダイアログボックスのハンドル                               */
HWND WINAPI
CreateDialogGUIW (HINSTANCE hInstance,
                  LPCWSTR   lpTemplate,
                  HWND      hWndParent,
                  DLGPROC   lpDialogFunc)
{
  return CreateDialogParamGUIW (hInstance, lpTemplate,
                                                hWndParent, lpDialogFunc, 0);
}


/*  ja:モードレスダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
             RET,ダイアログボックスのハンドル                               */
HWND WINAPI
CreateDialogGUIA (HINSTANCE hInstance,
                  LPCSTR    lpTemplate,
                  HWND      hWndParent,
                  DLGPROC   lpDialogFunc)
{
  return CreateDialogParamGUIA (hInstance, lpTemplate,
                                                hWndParent, lpDialogFunc, 0);
}


/*  ja:モーダルダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
          lParam,初期化値
             RET,ダイアログボックスのハンドル                               */
INT_PTR WINAPI
DialogBoxParamGUIW (HINSTANCE hInstance,
                    LPCWSTR   lpTemplate,
                    HWND      hWndParent,
                    DLGPROC   lpDialogFunc,
                    LPARAM    lParam)
{
  INT_PTR nResult;
  LPDLGTEMPLATE lpDialogTemplate;

  lpDialogTemplate = MakeDlgGUIFontW (hInstance, lpTemplate);
  if (lpDialogTemplate)
    {
      nResult = DialogBoxIndirectParamW (hInstance, lpDialogTemplate,
                                            hWndParent, lpDialogFunc, lParam);
      MemoryFree (lpDialogTemplate);
    }
  else
    {
      nResult = -1;
    }
  return nResult;
}


/*  ja:モーダルダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
          lParam,初期化値
             RET,ダイアログボックスのハンドル                               */
INT_PTR WINAPI
DialogBoxParamGUIA (HINSTANCE hInstance,
                    LPCSTR    lpTemplate,
                    HWND      hWndParent,
                    DLGPROC   lpDialogFunc,
                    LPARAM    lParam)
{
  INT_PTR nResult;
  LPDLGTEMPLATE lpDialogTemplate;

  lpDialogTemplate = MakeDlgGUIFontA (hInstance, lpTemplate);
  if (lpDialogTemplate)
    {
      nResult = DialogBoxIndirectParamA (hInstance, lpDialogTemplate,
                                            hWndParent, lpDialogFunc, lParam);
      MemoryFree (lpDialogTemplate);
    }
  else
    {
      nResult = -1;
    }
  return nResult;
}


/*  ja:モーダルダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
             RET,ダイアログボックスのハンドル                               */
INT_PTR WINAPI
DialogBoxGUIW (HINSTANCE hInstance,
               LPCWSTR   lpTemplate,
               HWND      hWndParent,
               DLGPROC   lpDialogFunc)
{
  return DialogBoxParamGUIW (hInstance, lpTemplate,
                                                hWndParent, lpDialogFunc, 0);
}


/*  ja:モーダルダイアログボックスを作成する
       hInstance,モジュールのハンドル
      lpTemplate,ダイアログボックステンプレートの名前
      hWndParent,オーナーウィンドウのハンドル
    lpDialogFunc,ダイアログボックスプロシージャ
             RET,ダイアログボックスのハンドル                               */
INT_PTR WINAPI
DialogBoxGUIA (HINSTANCE hInstance,
               LPCSTR    lpTemplate,
               HWND      hWndParent,
               DLGPROC   lpDialogFunc)
{
  return DialogBoxParamGUIA (hInstance, lpTemplate,
                                                hWndParent, lpDialogFunc, 0);
}
