/*
    wcommon
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wcommon.h"


/******************************************************************************
*                                                                             *
******************************************************************************/
/*  ja:レジストリのキーの値を取得する
             hKey,キーのハンドル
    lpszValueName,キーの名前
           lpData,データを受け取るバッファ
           dwData,データのサイズ
              RET,ERROR_SUCCESS:正常終了                                    */
LONG WINAPI
RegGetValueExW (HKEY   hKey,
                LPWSTR lpszValueName,
                LPVOID lpData,
                DWORD  dwData)
{
  return RegQueryValueExW (hKey, lpszValueName, NULL, NULL, lpData, &dwData);
}


/*  ja:レジストリのキーの値を取得する
             hKey,キーのハンドル
    lpszValueName,キーの名前
           lpData,データを受け取るバッファ
           dwData,データのサイズ
              RET,ERROR_SUCCESS:正常終了                                    */
LONG WINAPI
RegGetValueExA (HKEY   hKey,
                LPSTR  lpszValueName,
                LPVOID lpData,
                DWORD  dwData)
{
  return RegQueryValueExA (hKey, lpszValueName, NULL, NULL, lpData, &dwData);
}


/*  ja:レジストリのDWORD値を取得する
             hKey,キーのハンドル
    lpszValueName,キーの名前
        lpdwValue,DWORD値
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegGetDwordW (HKEY    hKey,
              LPCWSTR lpszValueName,
              LPDWORD lpdwValue)
{
  BOOL fResult;

  fResult = (BOOL)lpdwValue;
  if (lpdwValue)
    {
      DWORD dwData, cbData = sizeof (DWORD);

      fResult = RegQueryValueExW (hKey, lpszValueName, NULL, NULL,
                                    (LPBYTE)&dwData, &cbData) == ERROR_SUCCESS
                                                && cbData >= sizeof (DWORD);
      if (fResult)
        *lpdwValue = dwData;
    }
  return fResult;
}


/*  ja:レジストリのDWORD値を取得する
             hKey,キーのハンドル
    lpszValueName,キーの名前
        lpdwValue,DWORD値
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegGetDwordA (HKEY    hKey,
              LPCSTR  lpszValueName,
              LPDWORD lpdwValue)
{
  BOOL fResult;

  fResult = (BOOL)lpdwValue;
  if (lpdwValue)
    {
      DWORD dwData, cbData = sizeof (DWORD);

      fResult = RegQueryValueExA (hKey, lpszValueName, NULL, NULL,
                                    (LPBYTE)&dwData, &cbData) == ERROR_SUCCESS
                                                && cbData >= sizeof (DWORD);
      if (fResult)
        *lpdwValue = dwData;
    }
  return fResult;
}


/*  ja:レジストリの文字列を取得する
             hKey,キーのハンドル
    lpszValueName,キーの名前
              RET,文字列,NULL:エラー                                        */
LPWSTR WINAPI
RegGetStringW (HKEY    hKey,
               LPCWSTR lpszValueName)
{
  DWORD dwType, cbData;
  LPWSTR lpszString = NULL;

  if (RegQueryValueExW (hKey, lpszValueName, NULL, &dwType,
            NULL, &cbData) == ERROR_SUCCESS && dwType == REG_SZ && cbData > 0)
    {
      lpszString = MemoryAlloc (cbData);
      if (RegQueryValueExW (hKey, lpszValueName, NULL, NULL,
                                (LPBYTE)lpszString, &cbData) != ERROR_SUCCESS)
        {
          MemoryFree (lpszString);
          lpszString = NULL;
        }
    }
  return lpszString;
}


/*  ja:レジストリの文字列を取得する
             hKey,キーのハンドル
    lpszValueName,キーの名前
              RET,文字列,NULL:エラー                                        */
LPSTR WINAPI
RegGetStringA (HKEY   hKey,
               LPCSTR lpszValueName)
{
  DWORD dwType, cbData;
  LPSTR lpszString = NULL;

  if (RegQueryValueExA (hKey, lpszValueName, NULL, &dwType,
            NULL, &cbData) == ERROR_SUCCESS && dwType == REG_SZ && cbData > 0)
    {
      lpszString = MemoryAlloc (cbData);
      if (RegQueryValueExA (hKey, lpszValueName, NULL, NULL,
                                (LPBYTE)lpszString, &cbData) != ERROR_SUCCESS)
        {
          MemoryFree (lpszString);
          lpszString = NULL;
        }
    }
  return lpszString;
}


/*  ja:レジストリにバイナリを設定する
             hKey,キーのハンドル
    lpszValueName,キーの名前
          lpcData,バイナリ
            nData,バイト数
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegSetBinaryW (HKEY    hKey,
               LPCWSTR lpszValueName,
               LPCVOID lpcData,
               SSIZE_T nData)
{
  return RegSetValueExW (hKey, lpszValueName, 0, REG_BINARY,
                                            lpcData, nData) == ERROR_SUCCESS;
}


/*  ja:レジストリにバイナリを設定する
             hKey,キーのハンドル
    lpszValueName,キーの名前
          lpcData,バイナリ
            nData,バイト数
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegSetBinaryA (HKEY    hKey,
               LPCSTR  lpszValueName,
               LPCVOID lpcData,
               SSIZE_T nData)
{
  return RegSetValueExA (hKey, lpszValueName, 0, REG_BINARY,
                                            lpcData, nData) == ERROR_SUCCESS;
}


/*  ja:レジストリにDWORD値を設定する
             hKey,キーのハンドル
    lpszValueName,キーの名前
          dwValue,DWORD値
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegSetDwordW (HKEY    hKey,
              LPCWSTR lpszValueName,
              DWORD   dwValue)
{
  return RegSetValueExW (hKey, lpszValueName, 0, REG_DWORD,
                        (LPCBYTE)&dwValue, sizeof (DWORD)) == ERROR_SUCCESS;
}


/*  ja:レジストリにDWORD値を設定する
             hKey,キーのハンドル
    lpszValueName,キーの名前
          dwValue,DWORD値
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegSetDwordA (HKEY   hKey,
              LPCSTR lpszValueName,
              DWORD  dwValue)
{
  return RegSetValueExA (hKey, lpszValueName, 0, REG_DWORD,
                        (LPCBYTE)&dwValue, sizeof (DWORD)) == ERROR_SUCCESS;
}


/*  ja:レジストリに文字列を設定する
             hKey,キーのハンドル
    lpszValueName,キーの名前
       lpszString,文字列
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegSetStringW (HKEY    hKey,
               LPCWSTR lpszValueName,
               LPCWSTR lpszString)
{
  if (!lpszString)
    lpszString = L"";
  return RegSetValueExW (hKey, lpszValueName, 0, REG_SZ, (LPCBYTE)lpszString,
                (lstrlenW (lpszString) + 1) * sizeof (WCHAR)) == ERROR_SUCCESS;
}


/*  ja:レジストリに文字列を設定する
             hKey,キーのハンドル
    lpszValueName,キーの名前
       lpszString,文字列
              RET,TRUE:正常終了,FALSE:エラー                                */
BOOL WINAPI
RegSetStringA (HKEY   hKey,
               LPCSTR lpszValueName,
               LPCSTR lpszString)
{
  if (!lpszString)
    lpszString = "";
  return RegSetValueExA (hKey, lpszValueName, 0, REG_SZ, (LPCBYTE)lpszString,
                (lstrlenA (lpszString) + 1) * sizeof (CHAR)) == ERROR_SUCCESS;
}
