/*
    Text maid for Windows
    copyright (c) 1998-2018 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "find.h"
#include <windowsx.h>
#include "abort.h"
#include "edit.h"
#include "general.h"
#include "resource.h"
#include "wcommon.h"


int nFind = 0;                  /* ja:登録されている検索文字列 */
BOOL bFindArrow = TRUE;         /* ja:TRUE:下へ検索 */
BOOL bFindCase = FALSE;         /* ja:TRUE:大文字小文字を区別する */
BOOL bFindWidth = FALSE;        /* ja:TRUE:全角半角を区別する */
LPTSTR ppszFind[FINDNUM];       /* ja:検索する文字列 */


/******************************************************************************
*                                                                             *
* ja:検索関数群                                                               *
*                                                                             *
******************************************************************************/
/*  ja:検索
         hWnd,ウインドウ
    lpptStart,検索範囲
      lpptEnd,検索範囲(x=-1ならば選択範囲なし)
     lpszFind,検索文字列
       bArrow,TRUE:下へ,FALSE:上へ
        bCase,TRUE:大文字小文字を区別する,FALSE:大文字小文字を区別しない
       bWidth,TRUE:全角半角を区別する,FALSE:全角半角を区別しない
          RET,TRUE:正常終了,FALSE:中断                                      */
BOOL
FindOperation (HWND    hWnd,
               LPPOINT lpptStart,
               LPPOINT lpptEnd,
               LPCTSTR lpszFind,
               BOOL    bArrow,
               BOOL    bCase,
               BOOL    bWidth)
{
  HINSTANCE hInstance;
  LPSTR lpszText0;
  LPTSTR lpszAbort;
  ABORTDLG AbortDlg;

  hInstance = GetModuleHandle (NULL);
  AbortDlg.fSignal = FALSE;
  AbortDlg.lpszText = lpszAbort = LoadText (hInstance, IDS_JOB_FIND);
  lpszText0 = StringToMultiByte (lpszFind);
  if (lpszText0)
    {
      int i = 0;
      int nBytes0;      /* ja:文字数(バイト単位) */
      int nLeng0 = 0;   /* ja:文字数(文字単位) */
      LPTEXTWND ptw;
      POINT ptStart, ptEnd;

      ptStart = *lpptStart;
      ptEnd = *lpptEnd;
      ptw = (LPTEXTWND)GetWindowLongPtr (hWnd, GWLP_USERDATA);
      ptw->ptCursor.x = GetAlignPos (&ptw->lpStart, &ptw->nOff,
                        ptw->ptCursor.x, ptw->ptCursor.y, ptw->nTab, FALSE);
      /* ja:同じ文字数で比較するために文字数を数える */
      nBytes0 = lstrlenA (lpszText0);
      while (i < nBytes0)
        {
          i += IsDBCSLeadByteEx (CP_ACP, lpszText0[i]) ? 2 : 1;
          nLeng0++;
        }
      HideCaret (hWnd);
      if (ptw->ptSelect.x >= 0)
        {
          /* ja:選択範囲を解除 */
          ClearSel (hWnd, &ptw->ptSelect, &ptw->ptCursor);
          ptw->ptSelect.x = -1;
        }
      if (ptEnd.x >= 0)
        {
          int nBytes1;
          BOOL fResult;/* ja:比較の結果,TRUE:同じ,FALSE:異なる */
          LPSTR lpszText1;

          /* ja:検索結果による選択のときには選択を無効 */
          nBytes1 = GetSelByte (&ptw->lpStart, &ptw->nOff,
                                                &ptStart, &ptEnd, ptw->nTab);
          lpszText1 = MemoryAlloc ((nBytes1 + 1) * sizeof (CHAR));
          CpySelMem (&ptw->lpStart, &ptw->nOff,
                                    &ptStart, &ptEnd, ptw->nTab, lpszText1);
          if (!bWidth)/* ja:全角/半角の区別なし */
            fResult = CompareStringA (LOCALE_USER_DEFAULT,
                            (bCase ? 0 : NORM_IGNORECASE) | NORM_IGNOREWIDTH,
                                            lpszText0, -1, lpszText1, -1) == 2;
          else if (bCase)/* ja:大文字/小文字の区別あり */
            fResult = lstrcmpA (lpszText0, lpszText1) == 0;
          else
            fResult = lstrcmpiA (lpszText0, lpszText1) == 0;
          MemoryFree (lpszText1);
          if (fResult)
            ptEnd.x = -1;
        }
      if (ptEnd.x < 0)
        if (bArrow)
          {
            ptEnd.x = GetWidth (&ptw->lpStart, &ptw->nOff,
                                                    ptw->nMax - 1, ptw->nTab);
            ptEnd.y = ptw->nMax - 1;/* ja:下へ */
          }
        else
          {
            ptEnd.x = ptEnd.y = 0;/* ja:上へ */
          }
      if (bArrow && (ptStart.y < ptEnd.y
                            || ptStart.y == ptEnd.y && ptStart.x < ptEnd.x)
            || !bArrow && (ptEnd.y < ptStart.y
                            || ptStart.y == ptEnd.y && ptEnd.x < ptStart.x))
        {
          /* ja:方向と範囲が矛盾しない */
          int nBytes1 = 0;  /* ja:文字数(バイト単位) */
          int nLeng1 = 0;   /* ja:文字数(文字単位) */
          int nMax1, nCount = 0;
          HWND hDlg;
          LPSTR lpszText1;
          LPLINEBUF p, q;
          POINT ptData;

          EnableWindow (hWndMain, FALSE);
          hDlg = CreateDialogParamGUI (hInstance,
                                        MAKEINTRESOURCE (DIALOG_G), hWndClient,
                                        AbortDlgProc, (LPARAM)&AbortDlg);

          ptEnd.x = GetDataPos (&ptw->lpStart, &ptw->nOff,
                                    ptEnd.x, ptEnd.y, ptw->nTab, FALSE);
          ptStart.x = GetDataPos (&ptw->lpStart, &ptw->nOff,
                                    ptStart.x, ptStart.y, ptw->nTab, FALSE);
          p = q = GetLineBuffer (&ptw->lpStart, &ptw->nOff, ptStart.y);
          ptData = ptStart;
          nMax1 = (nBytes0 * 2 + 1);
          lpszText1 = MemoryAlloc (nMax1 * sizeof (CHAR));
          if (ptStart.y < ptEnd.y
                            || (ptStart.y == ptEnd.y && ptStart.x < ptEnd.x))
            {
              /* ja:上から下 */
              while (ptStart.y < ptEnd.y
                                || ptStart.y == ptEnd.y && ptStart.x < ptEnd.x)
                {
                  BOOL fResult;/* ja:比較の結果,TRUE:同じ,FALSE:異なる */

                  if (nCount++ % WCOMMON_BUFFER_SIZE == 0)
                    {
                      MSG msg;

                      while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
                        if (!IsDialogMessage (hDlg, &msg))
                          {
                            TranslateMessage (&msg);
                            DispatchMessage (&msg);
                          }
                      if (AbortDlg.fSignal)
                        break;
                    }
                  /* ja:バッファの最後に文字を加える */
                  while (nLeng1 < nLeng0 && (ptData.y < ptEnd.y
                                || ptData.y == ptEnd.y && ptData.x < ptEnd.x))
                    {
                      if (ptData.x < q->nLength)
                        {
                          /* ja:文字 */
                          if (ptData.x < q->nLength - 1
                                && IsDBCSLeadByteEx (CP_ACP,
                                                        q->lpszText[ptData.x]))
                            {
                              MemoryCopy (lpszText1 + nBytes1,
                                    q->lpszText + ptData.x, sizeof (CHAR) * 2);
                              nBytes1 += 2;
                              ptData.x += 2;
                            }
                          else
                            {
                              lpszText1[nBytes1] = q->lpszText[ptData.x];
                              nBytes1++;
                              ptData.x++;
                            }
                          nLeng1++;
                        }
                      else
                        {
                          /* ja:改行 */
                          if (!q->fMargin)
                            {
                              lpszText1[nBytes1] = '\n';
                              nBytes1++;
                              nLeng1++;
                            }
                          ptData.x = 0;
                          ptData.y++;
                          q = q->next;
                        }
                    }
                  lpszText1[nBytes1] = '\0';
                  /* ja:比較 */
                  if (!bWidth)/* ja:全角/半角の区別なし */
                    fResult = CompareStringA (LOCALE_USER_DEFAULT,
                            (bCase ? 0 : NORM_IGNORECASE) | NORM_IGNOREWIDTH,
                                lpszText0, -1, lpszText1, -1) == CSTR_EQUAL;
                  else if (bCase)/* ja:大文字/小文字の区別あり */
                    fResult = lstrcmpA (lpszText0, lpszText1) == 0;
                  else
                    fResult = lstrcmpiA (lpszText0, lpszText1) == 0;
                  if (fResult)
                    {
                      /* ja:一致するとき */
                      if (p->fMargin && p->nLength <= ptStart.x)
                        {
                          /* ja:右マージンのときには次の行の先頭 */
                          ptStart.x = 0;
                          ptStart.y++;
                        }
                      ptw->ptSelect.x = GetScreenPos (&ptw->lpStart,
                                &ptw->nOff, ptStart.x, ptStart.y, ptw->nTab);
                      ptw->ptSelect.y = ptStart.y;
                      if (q->fMargin && q->nLength <= ptData.x)
                        {
                          /* ja:右マージンのときには次の行の先頭 */
                          ptData.x = 0;
                          ptData.y++;
                        }
                      ptw->ptCursor.x = GetScreenPos (&ptw->lpStart,
                                    &ptw->nOff, ptData.x, ptData.y, ptw->nTab);
                      ptw->ptCursor.y = ptData.y;
                      break;
                    }
                  /* ja:バッファの先頭の文字を破棄する */
                  if (IsDBCSLeadByteEx (CP_ACP, lpszText1[0]))
                    {
                      MemoryCopy (lpszText1, lpszText1 + 2,
                                                (nMax1 - 2) * sizeof (CHAR));
                      nBytes1 -= 2;
                      if (ptStart.x < p->nLength)
                        {
                          ptStart.x += 2;
                        }
                      else
                        {
                          ptStart.x = 2;
                          ptStart.y++;
                          p = p->next;
                        }
                    }
                  else
                    {
                      MemoryCopy (lpszText1, lpszText1 + 1,
                                                (nMax1 - 1) * sizeof (CHAR));
                      nBytes1--;
                      if (ptStart.x < p->nLength)
                        {
                          ptStart.x++;
                        }
                      else
                        {
                          ptStart.x = p->fMargin ? 1 : 0;
                          ptStart.y++;
                          p = p->next;
                        }
                    }
                  nLeng1--;
                }
            }
          else
            {
              /* ja:下から上 */
              LPBYTE lpbType0, lpbType1;

              lpbType0 = MemoryAlloc (p->nLength * sizeof (BYTE));
              lpbType1 = MemoryAlloc (nMax1 * sizeof (BYTE));
              mCharType (lpbType0, p->lpszText, p->nLength);
              while (ptEnd.y < ptStart.y
                                || ptStart.y == ptEnd.y && ptEnd.x < ptStart.x)
                {
                  BOOL fResult;/* ja:比較の結果,TRUE:同じ,FALSE:異なる */

                  if (nCount++ % WCOMMON_BUFFER_SIZE == 0)
                    {
                      MSG msg;

                      while (PeekMessage (&msg, NULL, 0, 0, PM_REMOVE))
                        if (!IsDialogMessage (hDlg, &msg))
                          {
                            TranslateMessage (&msg);
                            DispatchMessage (&msg);
                          }
                      if (AbortDlg.fSignal)
                        break;
                    }
                  /* ja:バッファの最初に文字加える */
                  while (nLeng1 < nLeng0 && (ptEnd.y < ptData.y
                                || ptData.y == ptEnd.y && ptEnd.x < ptData.x))
                    {
                      if (0 < ptData.x)
                        {
                          /* ja:文字 */
                          if (lpbType0[ptData.x - 1] == 2)
                            {
                              MemoryCopy (lpszText1 + 2, lpszText1,
                                                (nMax1 - 2) * sizeof (CHAR));
                              MemoryCopy (lpbType1 + 2, lpbType1,
                                                (nMax1 - 2) * sizeof (BYTE));
                              MemoryCopy (lpszText1,
                                q->lpszText + ptData.x - 2, sizeof (CHAR) * 2);
                              lpbType1[0] = 1;
                              lpbType1[1] = 2;
                              nBytes1 += 2;
                              ptData.x -= 2;
                            }
                          else
                            {
                              MemoryCopy (lpszText1 + 1, lpszText1,
                                                (nMax1 - 1) * sizeof (CHAR));
                              MemoryCopy (lpbType1 + 1, lpbType1,
                                                (nMax1 - 1) * sizeof (BYTE));
                              lpszText1[0] = q->lpszText[ptData.x - 1];
                              lpbType1[0] = 0;
                              nBytes1++;
                              ptData.x--;
                            }
                          nLeng1++;
                        }
                      else
                        {
                          /* ja:改行 */
                          q = q->prev;
                          lpbType0 = MemoryReAlloc (lpbType0,
                                                q->nLength * sizeof (BYTE));
                          mCharType (lpbType0, q->lpszText, q->nLength);
                          if (!q->fMargin)
                            {
                              MemoryCopy (lpszText1 + 1, lpszText1,
                                                (nMax1 - 1) * sizeof (CHAR));
                              MemoryCopy (lpbType1 + 1, lpbType1,
                                                (nMax1 - 1) * sizeof (BYTE));
                              lpszText1[0] = '\n';
                              lpbType1[0] = 0;
                              nBytes1++;
                              nLeng1++;
                            }
                          ptData.x = q->nLength;
                          ptData.y--;
                        }
                    }
                  lpszText1[nBytes1] = '\0';
                  /* ja:比較 */
                  if (!bWidth)/* ja:全角/半角の区別なし */
                    fResult = CompareStringA (LOCALE_USER_DEFAULT,
                            (bCase ? 0 : NORM_IGNORECASE) | NORM_IGNOREWIDTH,
                                            lpszText0, -1, lpszText1, -1) == 2;
                  else if (bCase)/* ja:大文字/小文字の区別あり */
                    fResult = lstrcmpA (lpszText0, lpszText1) == 0;
                  else
                    fResult = lstrcmpiA (lpszText0, lpszText1) == 0;
                  if (fResult)
                    {
                      /* ja:一致するとき */
                      if (p->fMargin && p->nLength <= ptStart.x)
                        {
                          /* ja:右マージンのときには次の行の先頭 */
                          ptStart.x = 0;
                          ptStart.y++;
                        }
                      ptw->ptSelect.x = GetScreenPos (&ptw->lpStart,
                                &ptw->nOff, ptStart.x, ptStart.y, ptw->nTab);
                      ptw->ptSelect.y = ptStart.y;
                      if (q->fMargin && q->nLength <= ptData.x)
                        {
                          /* ja:右マージンのときには次の行の先頭 */
                          ptData.x = 0;
                          ptData.y++;
                        }
                      ptw->ptCursor.x = GetScreenPos (&ptw->lpStart,
                                    &ptw->nOff, ptData.x, ptData.y, ptw->nTab);
                      ptw->ptCursor.y = ptData.y;
                      break;
                    }
                  /* ja:バッファの最後の文字を破棄する */
                  if (lpbType1[nBytes1 - 1] == 2)
                    {
                      nBytes1 -= 2;
                      if (ptStart.x > 0)
                        {
                          ptStart.x -= 2;
                        }
                      else
                        {
                          p = p->prev;
                          ptStart.x = p->nLength - 2;
                          ptStart.y--;
                        }
                    }
                  else
                    {
                      nBytes1--;
                      if (ptStart.x > 0)
                        {
                          ptStart.x--;
                        }
                      else
                        {
                          p = p->prev;
                          ptStart.x = p->nLength - (p->fMargin ? 1 : 0);
                          ptStart.y--;
                        }
                    }
                  nLeng1--;
                }
              MemoryFree (lpbType0);
              MemoryFree (lpbType1);
            }
          MemoryFree (lpszText1);
          EnableWindow (hWndMain, TRUE);
          DestroyWindow (hDlg);
          if (ptw->ptSelect.x >= 0)
            {
              int sx, sy;
              POINT ptTop;

              ptTop = ptw->ptTop;
              sx = max (ptw->siWnd.cx / ptw->nFontSize, 1);
              sy = max (ptw->siWnd.cy / (ptw->nFontSize * 2), 1);
              ClearSel (hWnd, &ptw->ptSelect, &ptw->ptCursor);
              if (ptw->ptCursor.x < ptw->ptTop.x)
                ptw->ptTop.x = ptw->ptCursor.x;
              else if (ptw->ptCursor.x - sx + 1 > ptw->ptTop.x)
                ptw->ptTop.x = ptw->ptCursor.x - sx + 1;
              if (ptw->ptCursor.y < ptw->ptTop.y)
                ptw->ptTop.y = ptw->ptCursor.y;
              else if (ptw->ptCursor.y - sy + 1 > ptw->ptTop.y)
                ptw->ptTop.y = ptw->ptCursor.y - sy + 1;
              DrawCaret (hWnd);
              MoveTextWindow (hWnd, &ptTop);
            }
        }
      MemoryFree (lpszText0);
      ShowCaret (hWnd);
    }
  MemoryFree (lpszAbort);
  return !AbortDlg.fSignal;
}


/******************************************************************************
*                                                                             *
* ja:検索ダイアログ群                                                         *
*                                                                             *
******************************************************************************/
BOOL CALLBACK
FindDlgProc (HWND   hDlg,
             UINT   uMsg,
             WPARAM wParam,
             LPARAM lParam)
{
  switch (uMsg)
    {
      case WM_INITDIALOG:
        {
          int i;
          LPFINDDLG lpDlg;

          SetWindowLongPtr (hDlg, DWLP_USER, lParam);
          lpDlg = (LPFINDDLG)lParam;
          /* ja:チェックボックスの設定 */
          CheckDlgButton (hDlg, IDC_CHECKBOX81,
                                lpDlg->bCase ? BST_CHECKED : BST_UNCHECKED);
          CheckDlgButton (hDlg, IDC_CHECKBOX82,
                                lpDlg->bWidth ? BST_CHECKED : BST_UNCHECKED);
          /* ja:コンボボックスの設定 */
          for (i = 0; i < nFind; i++)
            ComboBoxItem_AddString (hDlg, IDC_COMBOBOX81, ppszFind[i]);
          if (nFind > 0)
            {
              ComboBoxItem_SetCurSel (hDlg, IDC_COMBOBOX81, 0);
              /* ja:コントロールの表示状態 */
              EnableDlgItem (hDlg, IDOK, TRUE);
            }
          /* ja:ラジオボタンの設定 */
          CheckRadioButton (hDlg, IDC_RADIOBUTTON81, IDC_RADIOBUTTON82,
                        lpDlg->bArrow ? IDC_RADIOBUTTON82 : IDC_RADIOBUTTON81);
        }
        return TRUE;
      case WM_COMMAND:
        switch (LOWORD (wParam))
          {
            case IDOK:
              {
                LPFINDDLG lpDlg;

                lpDlg = (LPFINDDLG)GetWindowLongPtr (hDlg, DWLP_USER);
                lpDlg->bCase = IsDlgButtonChecked (hDlg, IDC_CHECKBOX81)
                                                                == BST_CHECKED;
                lpDlg->bWidth = IsDlgButtonChecked (hDlg, IDC_CHECKBOX82)
                                                                == BST_CHECKED;
                lpDlg->bArrow = IsDlgButtonChecked (hDlg, IDC_RADIOBUTTON82)
                                                                == BST_CHECKED;
                lpDlg->lpszText = ComboBoxItem_GetTextNew (hDlg,
                                                            IDC_COMBOBOX81);
              }
            case IDCANCEL:
              EndDialog (hDlg, LOWORD (wParam));
              return TRUE;
            case IDC_COMBOBOX81:
              {
                int nLength;

                switch (HIWORD (wParam))
                  {
                    case CBN_EDITCHANGE:
                      nLength = ComboBox_GetTextLength ((HWND)lParam);
                      break;
                    case CBN_SELCHANGE:
                      nLength = 1;
                      break;
                    default: return TRUE;
                  }
                /* ja:コントロールの表示状態 */
                EnableDlgItem (hDlg, IDOK, nLength > 0);
              }
          }
        return TRUE;
    }
  return FALSE;
}
