/*
    Text maid for Windows
    copyright (c) 1998-2019 Kazuki Iwamoto https://www.maid.org/ iwm@maid.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "wndmain.h"
#include <commctrl.h>
#include <shlobj.h>
#include <shlwapi.h>
#include "file.h"
#include "find.h"
#include "format.h"
#include "general.h"
#include "history.h"
#include "jump.h"
#include "replace.h"
#include "resource.h"
#include "valchr.h"
#include "version.h"
#include "wcommon.h"


/******************************************************************************
*                                                                             *
* ja:ウインドウ関数群                                                         *
*                                                                             *
******************************************************************************/
static BOOL CALLBACK
CloseEnumProc (HWND   hWnd,
               LPARAM lParam)
{
  BOOL fResult = TRUE;
  LPTSTR lpszClass;

  lpszClass = GetClassNameNew (hWnd);
  if (lstrcmp (lpszClass, TEXTCLASS) == 0)
    {
      if (IsIconic (hWnd))
        SendMessage (hWndClient, WM_MDIRESTORE, (WPARAM)hWnd, 0);
      fResult = SendMessage (hWnd, WM_QUERYENDSESSION, 0, 0);
      if (fResult)
        SendMessage (hWndClient, WM_MDIDESTROY, (WPARAM)hWnd, 0);
    }
  MemoryFree (lpszClass);
  return fResult;
}


LRESULT CALLBACK
WndProc (HWND   hWnd,
         UINT   uMsg,
         WPARAM wParam,
         LPARAM lParam)
{
  static HWND hWndClip;     /* ja:元のクリップボードオーナー */

  switch (uMsg)
    {
      case WM_CREATE:/* ja:ウインドウが作られた時 */
        {
          int i;
          CLIENTCREATESTRUCT ccs;
          HMENU hMenu;
          const static TBBUTTON TBBbutton[] = {
{STD_FILENEW,    CM_NEW,      TBSTATE_ENABLED, TBSTYLE_BUTTON, {0, 0}, 0, 0},
{STD_FILEOPEN,   CM_OPEN,     TBSTATE_ENABLED, TBSTYLE_BUTTON, {0, 0}, 0, 0},
{STD_FILESAVE,   CM_SAVE,     0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{0,              0,           TBSTATE_ENABLED, TBSTYLE_SEP,    {0, 0}, 0, 0},
{STD_PROPERTIES, CM_PROPERTY, 0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{0,              0,           TBSTATE_ENABLED, TBSTYLE_SEP,    {0, 0}, 0, 0},
{STD_PRINT,      CM_PRINT,    0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{0,              0,           TBSTATE_ENABLED, TBSTYLE_SEP,    {0, 0}, 0, 0},
{STD_CUT,        CM_CUT,      0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{STD_COPY,       CM_COPY,     0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{STD_PASTE,      CM_PASTE,    0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{STD_DELETE,     CM_DELETE,   0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{0,              0,           TBSTATE_ENABLED, TBSTYLE_SEP,    {0, 0}, 0, 0},
{STD_UNDO,       CM_UNDO,     0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{STD_REDOW,      CM_REDO,     0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{0,              0,           TBSTATE_ENABLED, TBSTYLE_SEP,    {0, 0}, 0, 0},
{STD_FIND,       CM_FIND,     0,               TBSTYLE_BUTTON, {0, 0}, 0, 0},
{STD_REPLACE,    CM_REPLACE,  0,               TBSTYLE_BUTTON, {0, 0}, 0, 0}};
          const static UINT uID[] = {CM_UNDO, 0, CM_CUT, CM_COPY,
                                            CM_PASTE, CM_DELETE, 0, CM_ALL};

          InitIme ();
          hWndClip = SetClipboardViewer (hWnd);
          hWndTool = CreateToolbarEx (hWnd, WS_CHILD | WS_VISIBLE
                                    | CCS_NODIVIDER | TBSTYLE_TOOLTIPS,
                                    1, 0, HINST_COMMCTRL, IDB_STD_SMALL_COLOR,
                                    TBBbutton, n_elements (TBBbutton),
                                    0, 0, 0, 0, sizeof (TBBUTTON));
          hWndStat = CreateStatusWindow (WS_CHILD | WS_VISIBLE
                                            | SBARS_SIZEGRIP, _T(""), hWnd, 2);
          ccs.hWindowMenu = GetSubMenu (GetMenu (hWnd), MENUTOP - 2);
          ccs.idFirstChild = 300;
          hWndClient = CreateWindowEx (0, _T("MDICLIENT"), NULL,
            WS_HSCROLL | WS_VSCROLL | WS_CHILD | WS_CLIPCHILDREN | WS_VISIBLE,
            0, 0, 0, 0, hWnd, (HMENU)3, GetModuleHandle (NULL), (LPVOID)&ccs);
          uDlMsg = RegisterWindowMessage (DRAGLISTMSGSTRING);
          if (uDlMsg == 0)
            {
              MessageBox (hWnd, _T("RegisterWindowMessage"),
                                APPLICATION, MB_OK | MB_ICONSTOP);
              DestroyWindow (hWnd);
              return 0;
            }
          /* ja:空のペン */
          hPenNull = GetStockObject (NULL_PEN);
          /* ja:ポップアップメニューを作る */
          hMenu = GetMenu (hWnd);
          hMenuPopup = CreatePopupMenu ();
          for (i = 0; i < n_elements (uID); i++)
            if (uID[i] > 0)
              {
                LPTSTR p, lpszText;

                lpszText = GetMenuStringNew (hMenu, uID[i], MF_BYCOMMAND);
                p = StrChr (lpszText, '\t');
                if (p)
                  *p = '\0';
                AppendMenu (hMenuPopup, 0, uID[i], lpszText);
                MemoryFree (lpszText);
              }
            else
              {
                AppendMenu (hMenuPopup, MF_SEPARATOR, 0, NULL);
              }
        }
      case WM_SYSCOLORCHANGE:
        if (hBrushWindow)
          DeleteObject (hBrushWindow);
        if (hBrushGray)
          DeleteObject (hBrushGray);
        if (hBrushHighLight)
          DeleteObject (hBrushHighLight);
        if (hPenGray)
          DeleteObject (hPenGray);
        if (hPenGrid)
          DeleteObject (hPenGrid);
        hBrushWindow = hBrushGray = hBrushHighLight = NULL;
        hPenGray = hPenGrid = NULL;
        crWindowText = GetSysColor (COLOR_WINDOWTEXT);
        crGrayText = GetSysColor (COLOR_GRAYTEXT);
        crHighLightText = GetSysColor (COLOR_HIGHLIGHTTEXT);
        hBrushWindow = CreateSolidBrush (GetSysColor (COLOR_WINDOW));
        hBrushGray = CreateSolidBrush (crGrayText);
        hBrushHighLight = CreateSolidBrush (GetSysColor (COLOR_HIGHLIGHT));
        hPenGray = CreatePen (PS_SOLID, 0, crGrayText);
        hPenGrid = CreatePen (PS_SOLID, 0, GetSysColor (COLOR_BTNFACE));
        break;
      case WM_SIZE:
        {
          int y;
          RECT rc;

          ToolBar_AutoSize (hWndTool);
          GetWindowRect (hWndTool, &rc);
          y = rc.bottom - rc.top;
          GetWindowRect (hWndStat, &rc);
          MoveWindow (hWndStat, 0, HIWORD (lParam) - rc.bottom + rc.top,
                                    LOWORD(lParam), rc.bottom - rc.top, TRUE);
          MoveWindow (hWndClient, 0, y, LOWORD (lParam),
                            HIWORD (lParam) - y - rc.bottom + rc.top, TRUE);
        }
        return 0;
      case WM_NOTIFY:
        switch (((LPNMHDR)lParam)->code)
          {
            case TTN_NEEDTEXT:
              ((LPTOOLTIPTEXT)lParam)->lpszText
                            = (LPTSTR)(((LPNMHDR)lParam)->idFrom + TIPBASE);
              ((LPTOOLTIPTEXT)lParam)->hinst = GetModuleHandle (NULL);
              break;
            case TTN_SHOW:
              {
                LPTSTR lpszText;

                lpszText = LoadText (GetModuleHandle (NULL),
                                                    ((LPNMHDR)lParam)->idFrom);
                if (lpszText)
                  {
                    StatusBar_Simple (hWndStat, TRUE);
                    StatusBar_SetText (hWndStat, 255, SBT_NOBORDERS, lpszText);
                    MemoryFree (lpszText);
                  }
              }
              break;
            case TTN_POP:
              StatusBar_SetText (hWndStat, 255, 0, _T(""));
              StatusBar_Simple (hWndStat, FALSE);
          }
        break;
      case WM_MENUSELECT:
        {
          int nMenu;
          HINSTANCE hInstance;
          HMENU hMenu;
          UINT ids[2] = {0, 0};

          hInstance = GetModuleHandle (NULL);
          hMenu = GetMenu (hWnd);
          if (hMenu)
          nMenu = (GetMenuItemCount (hMenu) != MENUTOP);
          if (300 <= LOWORD (wParam) && LOWORD (wParam) < 400)
            MenuHelp (uMsg, MAKEWPARAM (IDS_WND, HIWORD (wParam)), lParam,
                                            hMenu, hInstance, hWndStat, ids);
          else if (200 <= LOWORD (wParam) && LOWORD (wParam) < 300)
            MenuHelp (uMsg, MAKEWPARAM (IDS_THIS, HIWORD (wParam)), lParam,
                                            hMenu, hInstance, hWndStat, ids);
          else if (HIWORD (wParam) & MF_SYSMENU
                                || LOWORD (wParam) >= 100
                                || nMenu != 0 && LOWORD (wParam) == 0
                                || HIWORD (wParam) == 0xffff && !(HMENU)lParam)
            MenuHelp (uMsg, wParam, lParam, hMenu, hInstance, hWndStat, ids);
          else
            MenuHelp (uMsg, MAKEWPARAM (IDS_FILE + LOWORD (wParam) - nMenu,
                    HIWORD (wParam)), lParam, hMenu, hInstance, hWndStat, ids);
        }
        return 0;
      case WM_CHANGECBCHAIN:
        if (hWndClip == (HWND)wParam)
          hWndClip = (HWND)lParam;
        else if (hWndClip)
          SendMessage (hWndClip, uMsg, wParam, lParam);
        return 0;
      case WM_DRAWCLIPBOARD:
        {
          int nResult;
          BOOL f;
          UINT uPriority[2] = {CF_TEXT, CF_UNICODETEXT};

          nResult = GetPriorityClipboardFormat (uPriority, 2);
          f = nResult && nResult != -1;
          if (fClipBoard != f)
            {
              HMENU hMenu;

              fClipBoard = f;
              hMenu = GetMenu (hWnd);
              EnableMenuItem (hMenu, CM_PASTE, fClipBoard
                                                    ? MF_ENABLED : MF_GRAYED);
              DrawMenuBar (hWnd);
              if (IsWindow (hWndTool))
                ToolBar_EnableButton (hWndTool, CM_PASTE, fClipBoard
                                    && IsWindow ((HWND)SendMessage (hWndClient,
                                                    WM_MDIGETACTIVE, 0, 0)));
            }
          if (hWndClip)
            SendMessage (hWndClip, uMsg, wParam, lParam);
        }
        return 0;
      case WM_DROPFILES:
        {
          int i, nFile;

          SetForegroundWindow (hWnd);
          nFile = DragQueryFile ((HDROP)wParam, 0xffffffff, NULL, 0);
          for (i = 0; i < nFile; i++)
            {
              LPTSTR lpszFile;

              lpszFile = DragQueryFileNew ((HDROP)wParam, i);
              OpenEditFile (lpszFile);
              MemoryFree (lpszFile);
            }
          DragFinish ((HDROP)wParam);
        }
        return 0;
      case WM_USER:
        {
          LPTSTR lpszArgv;

          lpszArgv = GlobalGetAtomNameNew ((ATOM)wParam);
          if (lpszArgv)
            {
              HWND hWndChild;
              static int nInitLine = 0;

              if (lpszArgv[0] == '-' || lpszArgv[0] == '/')
                switch ((int)CharUpper ((LPTSTR)lpszArgv[1]))
                  {
                    case 'C':
                      if (lpszArgv[2] == '\0')
                        nCodeDef=' ';
                      else
                        nCodeDef = StringToInteger (lpszArgv + 2, NULL, 10);
                      break;
                    case 'F':
                      if (lpszArgv[2] == '\0')
                        dwFilter = 0;
                      else
                        dwFilter = StringToInteger (lpszArgv + 2, NULL, 10);
                      break;
                    case 'H':
                      nHistory = IsCharNumeric (lpszArgv[2])
                                && lpszArgv[3] == '\0' ? lpszArgv[2] - '0' : 4;
                      ModifyHistory (hWnd, nHistory, MENUTOP, MENUFILE);
                      break;
                    case 'L':
                      nInitLine = StringToInteger (lpszArgv + 2, NULL, 10);
                      break;
                    case 'S':
                      if (lpszArgv[2] == '\0')
                        {
                          bFindArrow = TRUE;
                          bFindCase = FALSE;
                          bFindWidth = FALSE;
                          bReplaceArrow = TRUE;
                          bReplaceCase = FALSE;
                          bReplaceWidth = FALSE;
                        }
                      else
                        {
                          DWORD dwStat;

                          dwStat = StringToInteger (lpszArgv + 2, NULL, 10);
                          bFindArrow = !(dwStat & 4);
                          bFindCase = dwStat & 1;
                          bFindWidth = dwStat & 2;
                          bReplaceArrow = !(dwStat & 32);
                          bReplaceCase = dwStat & 8;
                          bReplaceWidth = dwStat & 16;
                        }
                      break;
                    case 'Z':
                      fZoomed = lpszArgv[2] - '0';
                  }
              else if ((hWndChild = OpenEditFile (lpszArgv)) && nInitLine != 0)
                JumpOperation (hWndChild, nInitLine);
              MemoryFree (lpszArgv);
            }
        }
        return 0;
      case WM_COMMAND:
        switch (LOWORD (wParam))
          {
            case CM_NEW:
              OpenEditFile (NULL);
              return 0;
            case CM_OPEN:
              {
                int i,nPos = 0;
                LPTSTR lpszFilter = NULL;
                OPENFILENAME ofn;
                TCHAR szFile[MAX_PATH];

                /* ja:フィルタの設定 */
                for (i = 0; i < nFileType; i++)
                  {
                    int nLength;

                    nLength = lstrlen (lpFileType[i].lpszText) + 1;
                    lpszFilter = MemoryReAlloc (lpszFilter,
                                        (nPos + nLength + 1) * sizeof (TCHAR));
                    MemoryCopy (lpszFilter + nPos, lpFileType[i].lpszText,
                                                    nLength * sizeof (TCHAR));
                    nPos += nLength;
                    nLength = lstrlen (lpFileType[i].lpszExt) + 1;
                    lpszFilter = MemoryReAlloc (lpszFilter,
                                        (nPos + nLength + 1) * sizeof (TCHAR));
                    MemoryCopy (lpszFilter + nPos, lpFileType[i].lpszExt,
                                                    nLength * sizeof (TCHAR));
                    nPos += nLength;
                  }
                szFile[0] = '\0';
                ofn.lStructSize = sizeof (OPENFILENAME);
                ofn.hwndOwner = hWndClient;
                ofn.lpstrFilter = lpszFilter;
                ofn.nFilterIndex = dwFilter + 1;
                ofn.lpstrCustomFilter = NULL;
                ofn.lpstrFile = szFile;
                ofn.nMaxFile = MAX_PATH;
                ofn.lpstrFileTitle = NULL;
                ofn.lpstrInitialDir = lpszOpenPath;
                ofn.lpstrTitle = NULL;
                ofn.Flags = OFN_HIDEREADONLY | OFN_NOCHANGEDIR;
                ofn.lpstrDefExt = NULL;
                if (GetOpenFileName (&ofn))
                  {
                    dwFilter = ofn.nFilterIndex - 1;
                    MemoryFree (lpszOpenPath);
                    lpszOpenPath = StringDuplicateEx (szFile, ofn.nFileOffset);
                    OpenEditFile (szFile);
                  }
                MemoryFree (lpszFilter);
              }
              return 0;
            case CM_CLOSE:
              {
                HWND hWndChild;

                hWndChild = (HWND)SendMessage (hWndClient, WM_MDIGETACTIVE,
                                                                        0, 0);
                if (SendMessage (hWndChild, WM_QUERYENDSESSION, 0, 0))
                  SendMessage (hWndClient, WM_MDIDESTROY,
                                                        (WPARAM)hWndChild, 0);
              }
              return 0;
            case CM_EXIT:
              SendMessage (hWnd, WM_CLOSE, 0, 0);
              return 0;
            case CM_HISTORY:
              nHistory = ModifyHistoryDlg (GetModuleHandle (NULL),
                MAKEINTRESOURCE (DIALOG_B), hWnd, nHistory, MENUTOP, MENUFILE);
              return 0;
            case CM_FORMAT:
              DialogBoxGUI (GetModuleHandle (NULL), MAKEINTRESOURCE (DIALOG_E),
                                                        hWnd, FormatDlgProc);
              return 0;
            case CM_HTILE:
              {
                HWND hWndChild;

                SendMessage (hWndClient, WM_MDITILE, MDITILE_HORIZONTAL, 0);
                hWndChild = (HWND)SendMessage (hWndClient,
                                                        WM_MDIGETACTIVE, 0, 0);
                if (hWndChild)
                  DrawCaret (hWndChild);
              }
              return 0;
            case CM_VTILE:
              {
                HWND hWndChild;

                SendMessage (hWndClient, WM_MDITILE, MDITILE_VERTICAL, 0);
                hWndChild = (HWND)SendMessage (hWndClient,
                                                        WM_MDIGETACTIVE, 0, 0);
                if (hWndChild)
                  DrawCaret (hWndChild);
              }
              return 0;
            case CM_CASCADE:
              {
                HWND hWndChild;

                SendMessage (hWndClient, WM_MDICASCADE, 0, 0);
                hWndChild = (HWND)SendMessage (hWndClient,
                                                        WM_MDIGETACTIVE, 0, 0);
                if (hWndChild)
                  DrawCaret (hWndChild);
              }
              return 0;
            case CM_VERSION:
              DialogBoxGUI (GetModuleHandle (NULL), MAKEINTRESOURCE (DIALOG_F),
                                                            hWnd, HelpDlgProc);
              return 0;
            case 200: case 201: case 202:
            case 203: case 204: case 205:
            case 206: case 207: case 208:
              {
                LPTSTR lpszFile;

                lpszFile = GetHistory (hWnd, LOWORD (wParam));
                OpenEditFile (lpszFile);
                MemoryFree (lpszFile);
              }
              return 0;
            default:
              {
                HWND hWndChild;

                hWndChild = (HWND)SendMessage (hWndClient, WM_MDIGETACTIVE,
                                                                        0, 0);
                if (IsWindow (hWndChild))
                  SendMessage (hWndChild, WM_COMMAND, wParam, lParam);
              }
          }
        break;
      case WM_QUERYENDSESSION:
      case WM_CLOSE:
        EnumChildWindows (hWndClient, CloseEnumProc, TRUE);
        if (IsWindow ((HWND)SendMessage (hWndClient, WM_MDIGETACTIVE, 0, 0)))
          return 0;
        break;
      case WM_DESTROY:
        if (nRegStat & REG_STAT_WRITE)
          RegWrite ();
        if (hWndClip)
          ChangeClipboardChain (hWnd, hWndClip);
        if (hBrushWindow)
          DeleteObject (hBrushWindow);
        if (hBrushGray)
          DeleteObject (hBrushGray);
        if (hBrushHighLight)
          DeleteObject (hBrushHighLight);
        if (hPenGray)
          DeleteObject (hPenGray);
        if (hPenGrid)
          DeleteObject (hPenGrid);
        if (hMenuPopup)
          DestroyMenu (hMenuPopup);
        ExitIme ();
        PostQuitMessage (0);
        return 0;
    }
  return DefFrameProc (hWnd, hWndClient, uMsg, wParam, lParam);
}
