/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2005 Takuro Ashie
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>

#include "tomoe-char-table.h"
#include "tomoe-canvas.h"
#include "tomoe-scrollable.h"

#define DEFAULT_FONT_SCALE 2
/*#define DEFAULT_FONT_SCALE PANGO_SCALE_XX_LARGE*/

enum {
    SELECTED_SIGNAL,
    LAST_SIGNAL
};

enum {
    PROP_0
};

typedef struct _TomoeCharTablePriv TomoeCharTablePriv;
struct _TomoeCharTablePriv
{
    TomoeCharTableLayout layout;

    TomoeCanvas   *canvas;

    GdkPixmap     *pixmap;
    guint          padding;
    gint           selected;
    gint           prelighted;
    GList         *layout_list;

    GtkAdjustment *h_adj;
    GtkAdjustment *v_adj;

    GtkTreeModel  *model;
};

#define TOMOE_CHAR_TABLE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_CHAR_TABLE, TomoeCharTablePriv))

static void   tomoe_char_table_scrollable_iface_init(TomoeScrollableIFace *iface);

G_DEFINE_TYPE_EXTENDED (TomoeCharTable, tomoe_char_table, GTK_TYPE_WIDGET, 0,\
                        G_IMPLEMENT_INTERFACE (TOMOE_TYPE_SCROLLABLE, tomoe_char_table_scrollable_iface_init))

/* virtual functions for GtkObject class */
static void   dispose              (GObject              *object);
static void   set_property         (GObject              *object,
                                    guint                 prop_id,
                                    const GValue         *value,
                                    GParamSpec           *pspec);
static void   get_property         (GObject              *object,
                                    guint                 prop_id,
                                    GValue               *value,
                                    GParamSpec           *pspec);

/* virtual functions for GtkWidget class */
static void   realize              (GtkWidget            *widget);
static void   size_allocate        (GtkWidget            *widget,
                                    GtkAllocation        *allocation);
static void   size_request         (GtkWidget            *widget,
                                    GtkRequisition       *requisition);
static gint   expose_event         (GtkWidget            *widget,
                                    GdkEventExpose       *event);
static gint   button_press_event   (GtkWidget            *widget,
                                    GdkEventButton       *event);
static gint   button_release_event (GtkWidget            *widget,
                                    GdkEventButton       *event);
static gint   motion_notify_event  (GtkWidget            *widget,
                                    GdkEventMotion       *event);
static gint   leave_notify_event   (GtkWidget            *widget,
                                    GdkEventCrossing     *event);

static void   tomoe_char_table_draw                 (TomoeCharTable       *view);

/* virtual functions for TomoeScrollable interface */
static void   tomoe_char_table_set_scroll_adjustments (TomoeScrollable    *scrollable,
                                                       GtkAdjustment      *h_adj,
                                                       GtkAdjustment      *v_adj);
static void   tomoe_char_table_get_scroll_adjustments (TomoeScrollable    *scrollable,
                                                       GtkAdjustment     **h_adj,
                                                       GtkAdjustment     **v_adj);

/* callback functions for related tomoe canvas */
static void   on_canvas_find                        (TomoeCanvas          *canvas,
                                                     gpointer              user_data);
static void   on_canvas_clear                       (TomoeCanvas          *canvas,
                                                     gpointer              user_data);
static void   on_h_adjustment_value_changed         (GtkAdjustment        *adj,
                                                     gpointer              data);
static void   on_v_adjustment_value_changed         (GtkAdjustment        *adj,
                                                     gpointer              data);

/* utility functions */
static void   get_char_frame_size                   (TomoeCharTable       *view,
                                                     gint                 *inner_width,
                                                     gint                 *inner_hegiht,
                                                     gint                 *outer_width,
                                                     gint                 *outer_height);
static gint   get_char_id_from_coordinate           (TomoeCharTable       *view,
                                                     gint                  x,
                                                     gint                  y);
static void   adjust_adjustments                    (TomoeCharTable       *view);

static guint view_signals[LAST_SIGNAL] = { 0 };

static void
tomoe_char_table_class_init (TomoeCharTableClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
    GtkWidgetClass *widget_class     = GTK_WIDGET_CLASS (klass);

    view_signals[SELECTED_SIGNAL] =
      g_signal_new ("selected",
  		  G_TYPE_FROM_CLASS (klass),
  		  G_SIGNAL_RUN_LAST,
  		  G_STRUCT_OFFSET (TomoeCharTableClass, selected),
  		  NULL, NULL,
  		  g_cclosure_marshal_VOID__VOID,
  		  G_TYPE_NONE, 0);

    gobject_class->dispose             = dispose;
    gobject_class->set_property        = set_property;
    gobject_class->get_property        = get_property;
    widget_class->realize              = realize;
    widget_class->size_allocate        = size_allocate;
    widget_class->size_request         = size_request;
    widget_class->expose_event         = expose_event;
    widget_class->button_press_event   = button_press_event;
    widget_class->button_release_event = button_release_event;
    widget_class->motion_notify_event  = motion_notify_event;
    widget_class->leave_notify_event   = leave_notify_event;

    klass->selected                    = NULL;

    g_type_class_add_private (gobject_class, sizeof(TomoeCharTablePriv));

    tomoe_scrollable_setup_widget_class (GTK_WIDGET_CLASS (klass));
}

static void
tomoe_char_table_scrollable_iface_init (TomoeScrollableIFace *iface)
{
    iface->set_adjustments = tomoe_char_table_set_scroll_adjustments;
    iface->get_adjustments = tomoe_char_table_get_scroll_adjustments;
}

static void
tomoe_char_table_init (TomoeCharTable *view)
{
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    GtkWidget *widget = GTK_WIDGET (view);
    PangoFontDescription *font_desc;
    gint size;

    priv->layout      = TOMOE_CHAR_TABLE_LAYOUT_SINGLE_HORIZONTAL;
    priv->canvas      = NULL;
    priv->pixmap      = NULL;
    priv->padding     = 2;
    priv->selected    = -1;
    priv->prelighted  = -1;
    priv->layout_list = NULL;
    priv->h_adj       = NULL;
    priv->v_adj       = NULL;

    /* Set default font with large scale */
    font_desc = pango_font_description_copy (widget->style->font_desc);
    size = pango_font_description_get_size(font_desc);
    pango_font_description_set_size(font_desc, size * DEFAULT_FONT_SCALE);
    gtk_widget_modify_font (widget, font_desc);
    pango_font_description_free (font_desc);
}

static void
dispose (GObject *object)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (object);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    tomoe_char_table_set_canvas (view, NULL);

    if (priv->pixmap) {
        g_object_unref(G_OBJECT (priv->pixmap));
        priv->pixmap = NULL;
    }

    if (priv->layout_list) {
        g_list_foreach (priv->layout_list, (GFunc) g_object_unref, NULL);
        g_list_free (priv->layout_list);
        priv->layout_list = NULL;
    }

    if (priv->h_adj) {
        /* FIXME: disconnect callbacks */
        g_object_unref (priv->h_adj);
        priv->h_adj = NULL;
    }

    if (priv->v_adj) {
        /* FIXME: disconnect callbacks */
        g_object_unref (priv->v_adj);
        priv->v_adj = NULL;
    }

    if (G_OBJECT_CLASS(tomoe_char_table_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_char_table_parent_class)->dispose(object);
}

static void
set_property (GObject *object,
              guint prop_id,
              const GValue *value,
              GParamSpec *pspec)
{
    switch (prop_id) {
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
get_property (GObject *object,
              guint prop_id,
              GValue *value,
              GParamSpec *pspec)
{
    switch (prop_id) {
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}

static void
realize (GtkWidget *widget)
{
    GdkWindowAttr attributes;

    GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

    attributes.window_type = GDK_WINDOW_CHILD;
    attributes.wclass = GDK_INPUT_OUTPUT;
    attributes.visual = gtk_widget_get_visual (widget);
    attributes.colormap = gtk_widget_get_colormap (widget);

    attributes.x = widget->allocation.x;
    attributes.y = widget->allocation.y;
    attributes.width = widget->allocation.width;
    attributes.height = widget->allocation.height;
    attributes.event_mask = GDK_EXPOSURE_MASK |
	    GDK_BUTTON_PRESS_MASK |
	    GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK |
	    GDK_POINTER_MOTION_HINT_MASK |
	    GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK;

    widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
		    &attributes,
		    GDK_WA_X | GDK_WA_Y |
		    GDK_WA_COLORMAP |
		    GDK_WA_VISUAL);
    gdk_window_set_user_data (widget->window, widget);
    widget->style = gtk_style_attach (widget->style, widget->window);

    gdk_window_set_background (widget->window, &widget->style->bg [GTK_STATE_NORMAL]);
}

static void
size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (widget);

    if (GTK_WIDGET_CLASS (tomoe_char_table_parent_class)->size_allocate)
        GTK_WIDGET_CLASS (tomoe_char_table_parent_class)->size_allocate (widget, allocation);

    if (GTK_WIDGET_REALIZED (widget)) {
        if (priv->pixmap)
            g_object_unref(priv->pixmap);

        priv->pixmap = gdk_pixmap_new(widget->window,
                                      allocation->width,
                                      allocation->height,
                                      -1);

        adjust_adjustments (TOMOE_CHAR_TABLE (widget));
        tomoe_char_table_draw (TOMOE_CHAR_TABLE (widget));
    }
}

static void
size_request (GtkWidget *widget, GtkRequisition *requisition)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (widget);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    PangoFontMetrics *metrics;
    PangoContext *context;
    gint ascent, descent, char_width, digit_width, char_pixels;
  
    gtk_widget_ensure_style (widget);
    context = gtk_widget_get_pango_context (widget);
    metrics = pango_context_get_metrics (context,
                                         widget->style->font_desc,
                                         pango_context_get_language (context));

    /* width */
    char_width = pango_font_metrics_get_approximate_char_width (metrics);
    digit_width = pango_font_metrics_get_approximate_digit_width (metrics);
    char_pixels = PANGO_PIXELS (MAX (char_width, digit_width) * PANGO_SCALE_XX_LARGE);
    requisition->width = char_pixels + priv->padding * 2;

    /* height */
    ascent  = pango_font_metrics_get_ascent (metrics);
    descent = pango_font_metrics_get_descent (metrics);
    requisition->height = PANGO_PIXELS (ascent + descent) + priv->padding * 2;

    pango_font_metrics_unref (metrics);
}

GtkWidget *
tomoe_char_table_new (void)
{
    return GTK_WIDGET(g_object_new (TOMOE_TYPE_CHAR_TABLE,
                                    NULL));
}

void
tomoe_char_table_set_canvas (TomoeCharTable *view, TomoeCanvas *canvas)
{
    TomoeCharTablePriv *priv;

    g_return_if_fail (TOMOE_IS_CHAR_TABLE (view));

    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    if (priv->canvas) {
        g_signal_handlers_disconnect_by_func (
            G_OBJECT (priv->canvas),
            (gpointer) on_canvas_find,
            (gpointer) view);
        g_object_remove_weak_pointer (G_OBJECT (canvas),
                                      (gpointer *) ((gpointer )&(priv->canvas)));
    }

    priv->canvas = canvas;

    if (canvas)
    {
        g_object_add_weak_pointer (G_OBJECT (canvas),
                                   (gpointer *) ((gpointer )&priv->canvas));

        g_signal_connect_after (G_OBJECT (canvas), "find",
                                G_CALLBACK (on_canvas_find), (gpointer) view);
        g_signal_connect_after (G_OBJECT (canvas), "clear",
                                G_CALLBACK (on_canvas_clear), (gpointer) view);
    }
}

TomoeChar *
tomoe_char_table_get_selected (TomoeCharTable *view)
{
    GtkWidget *widget;
    TomoeCharTablePriv *priv;

    g_return_val_if_fail (TOMOE_IS_CHAR_TABLE (view), NULL);

    widget = GTK_WIDGET (view);
    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    if (!priv->canvas)
        return NULL;

    if (priv->selected < 0)
        return NULL;

    return tomoe_canvas_get_nth_candidate (priv->canvas, priv->selected);
}

TomoeCharTableLayout
tomoe_char_table_get_layout (TomoeCharTable *view)
{
    TomoeCharTablePriv *priv;

    g_return_val_if_fail (TOMOE_IS_CHAR_TABLE (view), 0);

    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    return priv->layout;
}

void
tomoe_char_table_set_layout (TomoeCharTable *view, TomoeCharTableLayout layout)
{
    TomoeCharTablePriv *priv;

    g_return_if_fail (TOMOE_IS_CHAR_TABLE (view));

    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    /*
     * FIXME! check the value!
     * TomoeCharTableLayout should be implemented as GObject's property.
     */
    priv->layout = layout;
}

static gint
expose_event (GtkWidget *widget, GdkEventExpose *event)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (widget);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    gboolean retval = FALSE;

    if (GTK_WIDGET_CLASS(tomoe_char_table_parent_class)->expose_event)
        retval = GTK_WIDGET_CLASS(tomoe_char_table_parent_class)->expose_event (widget, event);
 
    if (GTK_WIDGET_REALIZED (widget) && !priv->pixmap) {
        priv->pixmap = gdk_pixmap_new(widget->window,
                                      widget->allocation.width,
                                      widget->allocation.height,
                                      -1);

        adjust_adjustments (TOMOE_CHAR_TABLE (widget));
        tomoe_char_table_draw (TOMOE_CHAR_TABLE (widget));
    }

    if (priv->pixmap)
        gdk_draw_drawable(widget->window,
                          widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                          priv->pixmap,
                          event->area.x, event->area.y,
                          event->area.x, event->area.y,
                          event->area.width, event->area.height);

    return retval;
}

static gint
button_press_event (GtkWidget *widget, GdkEventButton *event)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (widget);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    gint prev_selected;

    prev_selected = priv->selected;
    priv->selected = get_char_id_from_coordinate (view,
                                                  (gint) event->x,
                                                  (gint) event->y);

    if (prev_selected != priv->selected)
        tomoe_char_table_draw (view);
    if (priv->selected >= 0)
        g_signal_emit (G_OBJECT (widget),
                       view_signals[SELECTED_SIGNAL], 0);

    return FALSE;
}

static gint
button_release_event (GtkWidget *widget, GdkEventButton *event)
{
    return FALSE;
}

static gint
motion_notify_event (GtkWidget *widget, GdkEventMotion *event)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (widget);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    GdkModifierType state;
    gint x, y, prev_prelighted;

    if (event->is_hint) {
        gdk_window_get_pointer(event->window, &x, &y, &state);
    } else {
        x = (int) event->x;
        y = (int) event->y;
        state = (GdkModifierType) event->state;
    }

    prev_prelighted = priv->prelighted;
    priv->prelighted = get_char_id_from_coordinate (view, x, y);

    if (prev_prelighted != priv->prelighted) {
        tomoe_char_table_draw (view);
    }

    return FALSE;
}

static gint
leave_notify_event (GtkWidget *widget, GdkEventCrossing *event)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (widget);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    gint prev_prelighted;

    prev_prelighted = priv->prelighted;
    priv->prelighted = -1;

    if (prev_prelighted != priv->prelighted) {
        tomoe_char_table_draw (view);
    }

    return FALSE;
}

static void
tomoe_char_table_draw (TomoeCharTable *view)
{
    GtkWidget *widget = GTK_WIDGET (view);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    gint inner_width, inner_height, outer_width, outer_height;
    gint h_offset = 0, v_offset = 0;
    gint x_pos, y_pos, cols, rows;
    guint i;
    GList *node;

    if (!priv->pixmap) return;

    get_char_frame_size (view,
                         &inner_width, &inner_height,
                         &outer_width, &outer_height);

    /* Calculate y position of characters for single horizontal layout */
    y_pos = (GTK_WIDGET (view)->allocation.height - inner_height) / 2;

    /* Calculate x position of characters for single vertical layout */
    x_pos = (GTK_WIDGET (view)->allocation.width  - inner_width) / 2;

    /* Calculate columns for horizontal layout */
    cols = GTK_WIDGET (view)->allocation.width  / outer_width;
    cols = cols <= 0 ? 1 : cols;

    /* Calculate rows for vertical layout */
    rows = GTK_WIDGET (view)->allocation.height / outer_height;
    rows = rows <= 0 ? 1 : rows;

    /* offset */
    if (priv->h_adj)
        h_offset = priv->h_adj->value;
    if (priv->v_adj)
        v_offset = priv->v_adj->value;

    /* Fill background */
    gdk_draw_rectangle (priv->pixmap,
                        widget->style->white_gc,
                        TRUE,
                        0, 0,
                        widget->allocation.width,
                        widget->allocation.height);

    /* Draw characters */
    /* FIXME: stupid linear search */
    for (i = 0, node = priv->layout_list;
         node;
         i++, node = g_list_next (node))
    {
        PangoLayout *layout = PANGO_LAYOUT (node->data);
        gint char_width = 20, char_height = 20;
        gint outer_x, outer_y;
        gint inner_x, inner_y;
        gboolean selected = (gint) i == priv->selected;

        pango_layout_get_pixel_size (layout, &char_width, &char_height);

        if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_SINGLE_HORIZONTAL) {
            outer_x = outer_width * i - h_offset;
            outer_y = 0;
            outer_height = widget->allocation.height;
            inner_x = outer_x + (outer_width  - char_width)  / 2;
            inner_y = y_pos;

            if (outer_x + outer_width < 0)
                continue;
            if (outer_x + outer_width > widget->allocation.width)
                break;

        } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_SINGLE_VERTICAL) {
            outer_x = 0;
            outer_y = outer_height * i - v_offset;
            outer_width = widget->allocation.width;
            inner_x = x_pos;
            inner_y = outer_y + (outer_height - char_height) / 2;

            if (outer_y + outer_height < 0)
                continue;
            if (outer_y + outer_height > widget->allocation.height)
                break;

        } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_HORIZONTAL) {
            outer_x      = outer_width  * (i % cols) - h_offset;
            outer_y      = outer_height * (i / cols) - v_offset;
            inner_x      = outer_x + (outer_width  - char_width)  / 2;
            inner_y      = outer_y + (outer_height - char_height) / 2;

            if (outer_y + outer_height < 0)
                continue;
            if (outer_y + outer_height > widget->allocation.height)
                break;

        } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_VERTICAL) {
            outer_x      = outer_width  * (i / rows) - h_offset;
            outer_y      = outer_height * (i % rows) - v_offset;
            inner_x      = outer_x + (outer_width  - char_width)  / 2;
            inner_y      = outer_y + (outer_height - char_height) / 2;

            if (outer_x + outer_width < 0)
                continue;
            if (outer_x + outer_width > widget->allocation.width)
                break;

        } else {
            /* Shouldn't reach */
            break;
        }

        gdk_draw_rectangle (priv->pixmap,
                            selected ? widget->style->bg_gc[GTK_STATE_SELECTED]
                                     : widget->style->white_gc,
                            TRUE,
                            outer_x, outer_y,
                            outer_width, outer_height);

        gdk_draw_layout (priv->pixmap,
                         selected ? widget->style->white_gc
                                  : widget->style->black_gc,
                         inner_x, inner_y,
                         layout);

        if ((gint) i == priv->prelighted)
            gtk_paint_shadow (widget->style, priv->pixmap,
                              GTK_STATE_PRELIGHT, GTK_SHADOW_OUT,
                              NULL, NULL, NULL,
                              outer_x, outer_y,
                              outer_width, outer_height);
    }

    /* Draw off-screen pixmap to foreground */
    gdk_draw_drawable(widget->window,
                      widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
                      priv->pixmap,
                      0, 0,
                      0, 0,
                      widget->allocation.width, widget->allocation.height);
}


/*
 *  Virtual functions for TomoeScrollable interface.
 */
static void
tomoe_char_table_set_scroll_adjustments (TomoeScrollable *scrollable,
                                         GtkAdjustment *h_adj,
                                         GtkAdjustment *v_adj)
{
    TomoeCharTable *view;
    TomoeCharTablePriv *priv;

    g_return_if_fail (TOMOE_IS_CHAR_TABLE (scrollable));

    view = TOMOE_CHAR_TABLE (scrollable);
    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (scrollable);

    if (priv->h_adj) {
        /* FIXME: disconnect callbacks */
        g_object_unref (priv->h_adj);
        priv->h_adj = NULL;
    }

    if (priv->v_adj) {
        /* FIXME: disconnect callbacks */
        g_object_unref (priv->v_adj);
        priv->v_adj = NULL;
    }

    if (h_adj) {
        g_object_ref (h_adj);
        priv->h_adj = h_adj;
        g_signal_connect (G_OBJECT (h_adj), "value-changed",
                          G_CALLBACK (on_h_adjustment_value_changed),
                          (gpointer) scrollable);
    }

    if (v_adj) {
        g_object_ref (v_adj);
        priv->v_adj = v_adj;
        g_signal_connect (G_OBJECT (v_adj), "value-changed",
                          G_CALLBACK (on_v_adjustment_value_changed),
                          (gpointer) scrollable);
    }

    adjust_adjustments (view);
}

static void
tomoe_char_table_get_scroll_adjustments (TomoeScrollable *scrollable,
                                         GtkAdjustment **h_adj,
                                         GtkAdjustment **v_adj)
{
    TomoeCharTable *view;
    TomoeCharTablePriv *priv;

    g_return_if_fail (TOMOE_IS_CHAR_TABLE (scrollable));

    view = TOMOE_CHAR_TABLE (scrollable);
    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (scrollable);

    if (h_adj)
        *h_adj = priv->h_adj;
    if (v_adj)
        *v_adj = priv->v_adj;
}


void
tomoe_char_table_set_model (TomoeCharTable *view, GtkTreeModel *model)
{
    TomoeCharTablePriv *priv;

    g_return_if_fail (TOMOE_IS_CHAR_TABLE (view));

    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    if (priv->model)
        g_object_unref (priv->model);

    if (model)
        g_object_ref (model);
    priv->model = model;
}

GtkTreeModel *
tomoe_char_table_get_model (TomoeCharTable *view)
{
    TomoeCharTablePriv *priv;

    g_return_val_if_fail (TOMOE_IS_CHAR_TABLE (view), NULL);

    priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    return priv->model;
}



/*
 *  Callback functions for related TomoeCanvas.
 */
static void
on_canvas_find (TomoeCanvas *canvas, gpointer user_data)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (user_data);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    GtkWidget *widget = GTK_WIDGET (view);
    guint n_candidates = 0;
    unsigned int i = 0;

    if (priv->layout_list) {
        g_list_foreach (priv->layout_list, (GFunc) g_object_unref, NULL);
        g_list_free (priv->layout_list);
        priv->layout_list = NULL;
    }
    priv->selected = -1;
    priv->prelighted = -1;

    if (priv->canvas)
        n_candidates = tomoe_canvas_get_n_candidates (priv->canvas);

    for (i = 0; i < n_candidates; i++) {
        PangoLayout *layout;
        TomoeChar* chr;

        chr = tomoe_canvas_get_nth_candidate (priv->canvas, i);
        if (!chr)
            continue;
        layout = gtk_widget_create_pango_layout (widget, tomoe_char_get_utf8 (chr));

        priv->layout_list = g_list_append (priv->layout_list, layout);
    }

    adjust_adjustments (view);
    tomoe_char_table_draw (view);
}

static void
on_canvas_clear (TomoeCanvas *canvas, gpointer user_data)
{
    TomoeCharTable *view = TOMOE_CHAR_TABLE (user_data);
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    if (priv->layout_list) {
        g_list_foreach (priv->layout_list, (GFunc) g_object_unref, NULL);
        g_list_free (priv->layout_list);
        priv->layout_list = NULL;
    }
    priv->selected = -1;
    priv->prelighted = -1;

    adjust_adjustments (view);
    tomoe_char_table_draw (view);
}

static void
on_h_adjustment_value_changed (GtkAdjustment *h_adj, gpointer data)
{
    TomoeCharTable *table = TOMOE_CHAR_TABLE (data);

    tomoe_char_table_draw (table);
}

static void
on_v_adjustment_value_changed (GtkAdjustment *v_adj, gpointer data)
{
    TomoeCharTable *table = TOMOE_CHAR_TABLE (data);

    tomoe_char_table_draw (table);
}


/*
 *  Utility functions
 */
static void
get_char_frame_size (TomoeCharTable *view,
                     gint *inner_width,
                     gint *inner_height,
                     gint *outer_width,
                     gint *outer_height)
{
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    PangoContext *context;
    PangoFontMetrics *metrics;
    gint char_width, digit_width, ascent, descent;
    gint max_char_width, max_char_height;

    context = gtk_widget_get_pango_context (GTK_WIDGET (view));
    metrics = pango_context_get_metrics (context,
                                         GTK_WIDGET (view)->style->font_desc,
                                         pango_context_get_language (context));

    /* Get max char size */
    char_width = pango_font_metrics_get_approximate_char_width (metrics);
    digit_width = pango_font_metrics_get_approximate_digit_width (metrics);
    max_char_width = PANGO_PIXELS (MAX (char_width, digit_width) * PANGO_SCALE_XX_LARGE);

    ascent  = pango_font_metrics_get_ascent (metrics);
    descent = pango_font_metrics_get_descent (metrics);
    max_char_height = PANGO_PIXELS (ascent + descent);

    if (inner_width)
        *inner_width  = max_char_width;
    if (inner_height)
        *inner_height = max_char_height;
    if (outer_width)
        *outer_width  = max_char_width  + priv->padding * 2;
    if (outer_height)
        *outer_height = max_char_height + priv->padding * 2;

    /* clean */
    pango_font_metrics_unref (metrics);
}

static gint
get_char_id_from_coordinate (TomoeCharTable *view, gint x, gint y)
{
    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);
    gint cols, rows, i;
    gint inner_width, inner_height, outer_width, outer_height;
    gint h_offset = 0, v_offset = 0;

    get_char_frame_size (view,
                         &inner_width, &inner_height,
                         &outer_width, &outer_height);
    if (priv->h_adj)
        h_offset = priv->h_adj->value;
    if (priv->v_adj)
        v_offset = priv->v_adj->value;

    /* Calculate columns for horizontal layout */
    cols = GTK_WIDGET (view)->allocation.width  / outer_width;
    cols = cols <= 0 ? 1 : cols;

    /* Calculate rows for vertical layout */
    rows = GTK_WIDGET (view)->allocation.height / outer_height;
    rows = rows <= 0 ? 1 : rows;

    /* FIXME: stupid linear search */
    for (i = 0; i < (gint) g_list_length (priv->layout_list); i++) {
        if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_SINGLE_HORIZONTAL) {
            gint area_x;

            area_x = outer_width * i - h_offset;

            if (x >= area_x && x < area_x + outer_width)
                return i;

        } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_SINGLE_VERTICAL) {
            gint area_y;

            area_y = outer_height * i - v_offset;

            if (y >= area_y && y < area_y + outer_height)
                return i;

        } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_HORIZONTAL) {
            gint area_x, area_y;

            area_x = outer_width  * (i % cols) - h_offset;
            area_y = outer_height * (i / cols) - v_offset;

            if (x >= area_x && x < area_x + outer_width &&
                y >= area_y && y < area_y + outer_height)
            {
                return i;
            }

        } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_VERTICAL) {
            gint area_x, area_y;

            area_x = outer_width  * (i / rows) - h_offset;
            area_y = outer_height * (i % rows) - v_offset;

            if (x >= area_x && x < (gint) area_x + outer_width &&
                y >= area_y && y < (gint) area_y + outer_height)
            {
                return i;
            }

        } else {
            /* Shoudn't reach */;
        }
    }

    return -1;
}

static void
adjust_adjustments (TomoeCharTable *view)
{
    gint inner_width, inner_height, outer_width, outer_height;
    gint n_chars;
    gfloat upper;
    gboolean vertical_scroll = FALSE;

    TomoeCharTablePriv *priv = TOMOE_CHAR_TABLE_GET_PRIVATE (view);

    if (!priv->h_adj && !priv->v_adj)
        return;

    get_char_frame_size (view,
                         &inner_width, &inner_height,
                         &outer_width, &outer_height);
    n_chars = g_list_length (priv->layout_list);

    if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_SINGLE_HORIZONTAL) {
        upper = outer_width  * n_chars;
    } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_SINGLE_VERTICAL) {
        upper = outer_height * n_chars;
        vertical_scroll = TRUE;
    } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_HORIZONTAL) {
        gint cols, rows;

        cols = GTK_WIDGET (view)->allocation.width / outer_width;
        cols = cols <= 0 ? 1 : cols;
        rows = n_chars / cols;
        if (cols * rows < n_chars) rows++;
        upper = outer_height * rows;
        vertical_scroll = TRUE;
    } else if (priv->layout == TOMOE_CHAR_TABLE_LAYOUT_VERTICAL) {
        gint cols, rows;

        rows = GTK_WIDGET (view)->allocation.height / outer_height;
        rows = rows <= 0 ? 1 : rows;
        cols = n_chars / rows;
        if (cols * rows < n_chars) cols++;
        upper = outer_width * cols;
    } else {
        upper = 0.0;
    }

    if (priv->h_adj && !vertical_scroll) {
        GtkAdjustment *adj = priv->h_adj;
        gint cols_in_page = GTK_WIDGET(view)->allocation.width / outer_width;

        adj->value          = 0.0; /* FIXME */
        adj->lower          = 0.0;
        adj->upper          = upper;
        adj->step_increment = outer_width;
        adj->page_increment = adj->page_size;
        adj->page_size      = cols_in_page * outer_width;
        gtk_adjustment_changed (GTK_ADJUSTMENT (adj));
    }

    if (priv->v_adj && vertical_scroll) {
        GtkAdjustment *adj = priv->v_adj;
        gint rows_in_page = GTK_WIDGET(view)->allocation.height / outer_height;

        adj->value          = 0.0; /* FIXME */
        adj->lower          = 0.0;
        adj->upper          = upper;
        adj->step_increment = outer_height;
        adj->page_increment = adj->page_size;
        adj->page_size      = rows_in_page * outer_height;
        gtk_adjustment_changed (GTK_ADJUSTMENT (adj));
    }
}
/*
 * vi:ts=4:nowrap:ai:expandtab
 */
