/*
 * policy.h
 *
 * Common functions for handling TOMOYO Linux's domain policy.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.1.1   2006/05/15
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define ROOT_NAME "<kernel>"

#define MAXBUFSIZE  8192

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(1);
}

#define PAGE_SIZE  4096

static const char *SaveName(const char *name) {
	static char *buf = NULL;
	static int buf_used_len = PAGE_SIZE;
	int i, len;
	char *saved_name = NULL;
	char **new_ptr = NULL;
	static char **search_list = NULL;
	static int search_list_count = 0;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		printf("ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	for (i = 0; i < search_list_count; i++) {
		if (strcmp(name, search_list[i]) == 0) return search_list[i];
	}
	if (buf_used_len + len > PAGE_SIZE) {
		if ((buf = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = 0;
	}
	saved_name = buf + buf_used_len;
	memmove(saved_name, name, len);
	if ((new_ptr = (char **) realloc(search_list, (search_list_count + 1) * sizeof(char *))) == NULL) OutOfMemory();
	search_list = new_ptr;
	search_list[search_list_count++] = saved_name;
	buf_used_len += len;
	return (const char *) saved_name;
}

static int IsCorrectDomain(const unsigned char *domainname) {
	unsigned char c, d, e;
	if (!domainname || strncmp(domainname, ROOT_NAME, strlen(ROOT_NAME))) goto out;
	domainname += strlen(ROOT_NAME);
	if (!*domainname) return 1;
	do {
		if (*domainname++ != ' ') goto out;
		if (*domainname++ != '/') goto out;
		while ((c = *domainname) != '\0' && c != ' ') {
			domainname++;
			if (c == '\\') {
				switch ((c = *domainname++)) {
				case '\\':  /* "\\" */
					continue;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *domainname++) >= '0' && d <= '7' && (e = *domainname++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				goto out;
			} else if (c < ' ' || c >= 127) {
				goto out;
			}
		}
	} while (*domainname);
	return 1;
 out:
	return 0;
}

typedef struct domain_info {
	const char *domainname;
	const char **string_ptr;
	int string_count;
} DOMAIN_INFO;

static DOMAIN_INFO *domain_list[2] = { NULL, NULL };
static int domain_list_count[2] = { 0, 0 };

static int AddStringEntry(const char *entry, const int index, const int type) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count[type]) {
		printf("AddStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	acl_ptr = domain_list[type][index].string_ptr;
	acl_count = domain_list[type][index].string_count;

	// Check for the same entry.
	for (i = 0; i < acl_count; i++) {
		if (strcmp(entry, acl_ptr[i]) == 0) return 0;
	}

	if ((cp = SaveName(entry)) == NULL) OutOfMemory();
	if ((acl_ptr = (const char **) realloc(acl_ptr, (acl_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
	acl_ptr[acl_count++] = cp;
	domain_list[type][index].string_ptr = acl_ptr;
	domain_list[type][index].string_count = acl_count;
	return 0;
}

static int FindDomain(const char *domainname, const int type) {
	int i;
	for (i = 0; i < domain_list_count[type]; i++) {
		if (strcmp(domainname, domain_list[type][i].domainname) == 0) {
			return i;
		}
	}
	return EOF;
}

static int FindOrAssignNewDomain(const char *domainname, const int type) {
	const char *saved_domainname;
	int index;
	if ((index = FindDomain(domainname, type)) == EOF) {
		if (IsCorrectDomain(domainname)) {
			if ((domain_list[type] = (DOMAIN_INFO *) realloc(domain_list[type], (domain_list_count[type] + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
			memset(&domain_list[type][domain_list_count[type]], 0, sizeof(DOMAIN_INFO));
			if ((saved_domainname = SaveName(domainname)) == NULL) OutOfMemory();
			domain_list[type][domain_list_count[type]].domainname = saved_domainname;
			index = domain_list_count[type]++;
		} else {
			fprintf(stderr, "FindOrAssignNewDomain: Invalid domainname '%s'\n", domainname);
		}
	}
	return index;
}

static int IsDomainDef(const unsigned char *buffer) {
	while (*buffer && (*buffer <= 32 || 127 <= *buffer)) buffer++;
	return strncmp(buffer, ROOT_NAME, strlen(ROOT_NAME)) == 0;
}

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static void SortPolicy(const int type) {
	int i, j, k;
	for (i = 0; i < domain_list_count[type]; i++) {
		for (j = i + 1; j < domain_list_count[type]; j++) {
			if (strcmp(domain_list[type][i].domainname, domain_list[type][j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[type][i]; domain_list[type][i] = domain_list[type][j]; domain_list[type][j] = tmp;
			}
		}
	}
	for (i = 0; i < domain_list_count[type]; i++) {
		const char **string_ptr = domain_list[type][i].string_ptr;
		const int string_count = domain_list[type][i].string_count;
		for (j = 0; j < string_count; j++) {
			for (k = j + 1; k < string_count; k++) {
				const char *a = string_ptr[j];
				const char *b = string_ptr[k];
				if (*a && *b && strcmp(a + 1, b + 1) > 0) {
					string_ptr[j] = b; string_ptr[k] = a;
				}
			}
		}
	}
}

static char buffer[MAXBUFSIZE];

static void ReadDomainPolicy(const char *filename, const int type) {
	FILE *fp;
	int index;
	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Can't open %s\n", filename);
		return;
	}
	index = EOF;
	while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
		char *cp = strchr(buffer, '\n');
		int perm;
		if (cp) *cp = '\0';
		else if (!feof(fp)) break;
		NormalizeLine(buffer);
		if (IsDomainDef(buffer)) {
			index = FindOrAssignNewDomain(buffer, type);
		} else if (index >= 0) {
			if (sscanf(buffer, "%d", &perm) == 1 || strncmp(buffer, "allow_", 6) == 0) AddStringEntry(buffer, index, type);
		}
	}
	fclose(fp);
	SortPolicy(type);
}
