/*
 * sortpolicy.c
 *
 * Sort domain policy.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.1.1   2006/05/15
 *
 * This program reads domain policy from stdin and sorts and writes to stdout.
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(1);
}

/* Copied from kernel source. */
static inline unsigned long partial_name_hash(unsigned long c, unsigned long prevhash) {
	return (prevhash + (c << 4) + (c >> 4)) * 11;
}

/* Copied from kernel source. */
static inline unsigned int full_name_hash(const unsigned char *name, unsigned int len) {
	unsigned long hash = 0;
	while (len--) hash = partial_name_hash(*name++, hash);
	return (unsigned int) hash;
}

#define PAGE_SIZE  4096

static char *alloc_element(const unsigned int size) {
	static char *buf = NULL;
	static unsigned int buf_used_len = PAGE_SIZE;
	char *ptr = NULL;
	if (size > PAGE_SIZE) return NULL;
	if (buf_used_len + size > PAGE_SIZE) {
		if ((ptr = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		buf = ptr;
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = size;
		ptr = buf;
	} else if (size) {
		ptr = buf + buf_used_len;
		buf_used_len += size;
	}
	return ptr;
}

#define MAX_HASH 256

typedef struct name_entry {
	struct name_entry *next; /* Pointer to next record. NULL if none.             */
	unsigned int hash;       /* hash and length                                   */
	const char *name;        /* Text form of filename and domainname. Never NULL. */
} NAME_ENTRY;

typedef struct free_memory_block_list {
	struct free_memory_block_list *next; /* Pointer to next record. NULL if none. */
	char *ptr;                           /* Pointer to a free area.               */
	int len;                             /* Length of the area.                   */
} FREE_MEMORY_BLOCK_LIST;

static const char *SaveName(const char *name) {
	static FREE_MEMORY_BLOCK_LIST fmb_list = { NULL, NULL, 0 };
	static NAME_ENTRY name_list[MAX_HASH]; /* The list of names. */
	NAME_ENTRY *ptr, *prev = NULL;
	unsigned int hash;
	FREE_MEMORY_BLOCK_LIST *fmb = &fmb_list;
	int len;
	static int first_call = 1;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		fprintf(stderr, "ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	hash = full_name_hash((const unsigned char *) name, len - 1);
	if (first_call) {
		int i;
		first_call = 0;
		memset(&name_list, 0, sizeof(name_list));
		for (i = 0; i < MAX_HASH; i++) name_list[i].name = "/";
	}
	ptr = &name_list[hash % MAX_HASH];
	hash ^= len; /* The hash % MAX_HASH are always same for ptr->hash, so embed length into the hash value. */
	while (ptr) {
		if (hash == ptr->hash && strcmp(name, ptr->name) == 0) goto out;
		prev = ptr; ptr = ptr->next;
	}
	while (len > fmb->len) {
		if (fmb->next) {
			fmb = fmb->next;
		} else {
			char *cp;
			if ((cp = (char *) malloc(PAGE_SIZE)) == NULL || (fmb->next = (FREE_MEMORY_BLOCK_LIST *) alloc_element(sizeof(FREE_MEMORY_BLOCK_LIST))) == NULL) OutOfMemory();
			memset(cp, 0, PAGE_SIZE);
			fmb = fmb->next;
			fmb->ptr = cp;
			fmb->len = PAGE_SIZE;
			fmb->next = NULL;
		}
	}
	if ((ptr = (NAME_ENTRY *) alloc_element(sizeof(NAME_ENTRY))) == NULL) OutOfMemory();
	memset(ptr, 0, sizeof(NAME_ENTRY));
	ptr->next = NULL;
	ptr->hash = hash;
	ptr->name = fmb->ptr;
	memmove(fmb->ptr, name, len);
	fmb->ptr += len;
	fmb->len -= len;
	prev->next = ptr; /* prev != NULL because name_list is not empty. */
	if (fmb->len == 0) {
		FREE_MEMORY_BLOCK_LIST *ptr = &fmb_list;
		while (ptr->next != fmb) ptr = ptr->next; ptr->next = fmb->next;
	}
 out:
	return ptr ? (const char *) ptr->name : NULL;
}

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}


typedef struct domain_info {
	const char *domainname;
	const char **string_ptr;
	int string_count;
} DOMAIN_INFO;

static DOMAIN_INFO *domain_list = NULL;
static int domain_list_count = 0;

static int AddStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	const char *saved_entry;
	int i;
	if (index < 0 || index >= domain_list_count) {
		fprintf(stderr, "AddStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	if ((saved_entry = SaveName(entry)) == NULL) OutOfMemory();
	
	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;

	// Check for the same entry.
	for (i = 0; i < acl_count; i++) {
		if (saved_entry == acl_ptr[i]) return 0;
	}

	if ((acl_ptr = (const char **) realloc(acl_ptr, (acl_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
	acl_ptr[acl_count++] = saved_entry;
	domain_list[index].string_ptr = acl_ptr;
	domain_list[index].string_count = acl_count;
	return 0;
}

static int FindOrAssignNewDomain(const char *domainname) {
	const char *saved_domainname;
	int index;
	if ((saved_domainname = SaveName(domainname)) == NULL) OutOfMemory();
	for (index = 0; index < domain_list_count; index++) {
		if (saved_domainname == domain_list[index].domainname) return index;
	}
	if ((domain_list = (DOMAIN_INFO *) realloc(domain_list, (domain_list_count + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
	memset(&domain_list[domain_list_count], 0, sizeof(DOMAIN_INFO));
	domain_list[domain_list_count].domainname = saved_domainname;
	return domain_list_count++;
}

int main(int argc, char *argv[]) {
	static char buffer[8192];
	int i, j, k;
	int index = EOF;
	while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, stdin) != NULL) {
		char *cp = strchr(buffer, '\n');
		if (cp) *cp = '\0';
		else if (!feof(stdin)) break;
		NormalizeLine(buffer);
		if (buffer[0] == '#' || buffer[0] == '\0') continue;
		if (strncmp(buffer, "<kernel>", 8) == 0) {
			index = FindOrAssignNewDomain(buffer);
		} else if (index >= 0) {
			AddStringEntry(buffer, index);
		}
	}
	for (i = 0; i < domain_list_count; i++) {
		for (j = i + 1; j < domain_list_count; j++) {
			if (strcmp(domain_list[i].domainname, domain_list[j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[i]; domain_list[i] = domain_list[j]; domain_list[j] = tmp;
			}
		}
	}
	for (i = 0; i < domain_list_count; i++) {
		const char **string_ptr = domain_list[i].string_ptr;
		const int string_count = domain_list[i].string_count;
		for (j = 0; j < string_count; j++) {
			for (k = j + 1; k < string_count; k++) {
				const char *a = string_ptr[j];
				const char *b = string_ptr[k];
				if (strcmp(a, b) > 0) {
					string_ptr[j] = b; string_ptr[k] = a;
				}
			}
		}
	}
	for (i = 0; i < domain_list_count; i++) {
		const char **string_ptr = domain_list[i].string_ptr;
		const int string_count = domain_list[i].string_count;
		printf("%s\n\n", domain_list[i].domainname);
		for (j = 0; j < string_count; j++) printf("%s\n", string_ptr[j]);
		printf("\n");
	}
	return 0;
}
