/*
 * editpolicy.c
 *
 * An editor for editing TOMOYO Linux's policy.
 *
 * Copyright (C) 2005-2006  NTT DATA CORPORATION
 *
 * Version: 1.3.1     2007/01/07
 *
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <curses.h>
#include <sys/socket.h>
#include <sys/un.h>

#define ROOT_NAME "<kernel>"

static void OutOfMemory(void) {
	fprintf(stderr, "Out of memory. Aborted.\n");
	exit(127);
}

/*
 * Check whether the given filename follows the naming rules.
 * Returns nonzero if follows, zero otherwise.
 */
static int IsCorrectPath(const char *filename, const int may_contain_pattern) {
	if (filename && *filename == '/') {
		char c, d, e;
		while ((c = *filename++) != '\0') {
			if (c == '\\') {
				switch ((c = *filename++)) {
				case '\\':  /* "\\" */
					continue;
				case '$':   /* "\$" */
				case '+':   /* "\+" */
				case '?':   /* "\?" */
				case '*':   /* "\*" */
				case '@':   /* "\@" */
				case 'x':   /* "\x" */
				case 'X':   /* "\X" */
				case 'a':   /* "\a" */
				case 'A':   /* "\A" */
					if (may_contain_pattern) continue;
					break;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *filename++) >= '0' && d <= '7' && (e = *filename++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				return 0;
			} else if (c <= ' ' || c >= 127) {
				return 0;
			}
		}
		return 1;
	}
	return 0;
}

static int IsCorrectDomain(const unsigned char *domainname) {
	unsigned char c, d, e;
	if (!domainname || strncmp(domainname, ROOT_NAME, strlen(ROOT_NAME))) goto out;
	domainname += strlen(ROOT_NAME);
	if (!*domainname) return 1;
	do {
		if (*domainname++ != ' ') goto out;
		if (*domainname++ != '/') goto out;
		while ((c = *domainname) != '\0' && c != ' ') {
			domainname++;
			if (c == '\\') {
				switch ((c = *domainname++)) {
				case '\\':  /* "\\" */
					continue;
				case '0':   /* "\ooo" */
				case '1':
				case '2':
				case '3':
					if ((d = *domainname++) >= '0' && d <= '7' && (e = *domainname++) >= '0' && e <= '7') {
						const unsigned char f =
							(((unsigned char) (c - '0')) << 6) +
							(((unsigned char) (d - '0')) << 3) +
							(((unsigned char) (e - '0')));
						if (f && (f <= ' ' || f >= 127)) continue; /* pattern is not \000 */
					}
				}
				goto out;
			} else if (c < ' ' || c >= 127) {
				goto out;
			}
		}
	} while (*domainname);
	return 1;
 out:
	return 0;
}

static inline int strendswith(const char *name, const char *tail) {
	int len;
	if (!name || !tail) return 0;
	len = strlen(name) - strlen(tail);
	return len >= 0 && strcmp(name + len, tail) == 0;
}

static int getch2(void) {
	static int c0 = 0, c1 = 0, c2 = 0, c3 = 0, len = 0;
	if (len > 0) { c0 = c1; c1 = c2; c2 = c3; len--; return c0; }
	c0 = getch(); if (c0 != 0x1B) return c0;
	c1 = getch(); if (c1 != '[') { len = 1; return c0; }
	c2 = getch(); if (c2 != '5' && c2 != '6') { len = 2; return c0; }
	c3 = getch(); if (c3 != '~') { len = 3; return c0; }
	return c2 == '5' ? KEY_PPAGE : KEY_NPAGE;
}

static void ReadGenericPolicy(void);
static void ReadDomainPolicy(void);
static int FindOrAssignNewDomain(const char *domainname);
static int FindDomain(const char *domainname);
static void NormalizeLine(unsigned char *buffer);
static char *DomainName(const int index);
static int IsDomainKeeper(const int index);
static int IsInitializer(const char *filename);
static int IsInitializerDomain(const int index);
static int IsVirtualDomain(const int index);

#define SYSTEM_POLICY_FILE    "system_policy"
#define EXCEPTION_POLICY_FILE "exception_policy"
#define DOMAIN_POLICY_FILE    "domain_policy"

static const char *policy_file = DOMAIN_POLICY_FILE;
static const char *list_caption = NULL;
static char *current_domain = NULL;

#define SCREEN_SYSTEM_LIST    0
#define SCREEN_EXCEPTION_LIST 1
#define SCREEN_DOMAIN_LIST    2
#define SCREEN_ACL_LIST       3
#define MAXSCREEN             4

static int current_screen = SCREEN_DOMAIN_LIST;

// List for generic policy.
static char **generic_acl_list = NULL;
static int generic_acl_list_count = 0;

// List for domain policy.
typedef struct domain_info {
	char *domainname;
	const char **string_ptr; /***** Used by offline mode. *****/
	int string_count; /***** Used by offline mode. *****/
	unsigned char is_virtual;
	unsigned char profile; /***** Used by offline mode. *****/
} DOMAIN_INFO;
static DOMAIN_INFO *domain_list = NULL;
static int domain_list_count = 0;
static char **domain_keepers = NULL;
static int domain_keepers_count = 0;
static char **initializers = NULL;
static int initializers_count = 0;

///////////////////////////  ACL HANDLER  //////////////////////////////

static int IsDomainKeeper(const int index) {
	int i;
	char *domainname = domain_list[index].domainname;
	for (i = 0; i < domain_keepers_count; i++) {
		if (strcmp(domainname, domain_keepers[i]) == 0) return 1;
	}
	return 0;
}

static int IsInitializer(const char *filename) {
	if (filename) {
		int i;
		for (i = 0; i < initializers_count; i++) {
			if (strcmp(filename, initializers[i]) == 0) return 1;
		}
	}
	return 0;
}

static int IsInitializerDomain(const int index) {
	char *cp = strchr(domain_list[index].domainname, ' ');
	if (cp) return IsInitializer(cp + 1);
	return 0;
}

static int IsVirtualDomain(const int index) {
	return domain_list[index].is_virtual;
}

////////////////////////////  DOMAIN HANDLER  ////////////////////////////

static int FindDomain(const char *domainname) {
	int i;
	for (i = 0; i < domain_list_count; i++) {
		if (strcmp(domainname, domain_list[i].domainname) == 0) {
			return i;
		}
	}
	return EOF;
}

static int FindOrAssignNewDomain(const char *domainname) {
	int index;
	if ((index = FindDomain(domainname)) == EOF) {
		if (IsCorrectDomain(domainname)) {
			char *saved_domainname = strdup(domainname);
			if (!saved_domainname || (domain_list = (DOMAIN_INFO *) realloc(domain_list, (domain_list_count + 1) * sizeof(DOMAIN_INFO))) == NULL) OutOfMemory();
			memset(&domain_list[domain_list_count], 0, sizeof(DOMAIN_INFO));
			domain_list[domain_list_count].domainname = saved_domainname;
			index = domain_list_count++;
		} else {
			fprintf(stderr, "%s: Invalid domainname '%s'\n",  __FUNCTION__, domainname);
		}
	}
	return index;
}

static char *DomainName(const int index) {
	return domain_list[index].domainname;
}

///////////////////////////  UTILITY FUNCTIONS  //////////////////////////////

static int IsDomainDef(const unsigned char *buffer) {
	return strncmp(buffer, ROOT_NAME, strlen(ROOT_NAME)) == 0;
}

static void NormalizeLine(unsigned char *buffer) {
	unsigned char *sp = buffer, *dp = buffer;
	int first = 1;
	while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	while (*sp) {
		if (!first) *dp++ = ' ';
		first = 0;
		while (32 < *sp && *sp < 127) *dp++ = *sp++;
		while (*sp && (*sp <= 32 || 127 <= *sp)) sp++;
	}
	*dp = '\0';
}

static int buffer_lock = 0;
static void get(void) {
	if (buffer_lock) OutOfMemory();
	buffer_lock++;
}
static void put(void) {
	if (buffer_lock != 1) OutOfMemory();
	buffer_lock--;
}

#define MAXBUFSIZE  8192
static char buffer[MAXBUFSIZE];

static int offline_mode = 0;
static int persistent_fd = EOF;

static void SendFD(char *buffer, int *fd) {
	struct msghdr msg;
	struct iovec iov = { buffer, strlen(buffer) };
	char cmsg_buf[CMSG_SPACE(sizeof(int))];
	struct cmsghdr *cmsg = (struct cmsghdr *) cmsg_buf; 
	memset(&msg, 0, sizeof(msg));
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = cmsg_buf;
	msg.msg_controllen = sizeof(cmsg_buf);
	cmsg->cmsg_level = SOL_SOCKET;
	cmsg->cmsg_type = SCM_RIGHTS;
	msg.msg_controllen = cmsg->cmsg_len = CMSG_LEN(sizeof(int));
	memmove(CMSG_DATA(cmsg), fd, sizeof(int));
	sendmsg(persistent_fd, &msg, 0);
	close(*fd);
}

static FILE *open_read(const char *filename) {
	if (offline_mode) {
		char buffer[1024];
		int fd[2];
		FILE *fp;
		if (socketpair(PF_UNIX, SOCK_STREAM, 0, fd)) {
			fprintf(stderr, "socketpair()\n");
			exit(1);
		}
		if (shutdown(fd[0], SHUT_WR) || (fp = fdopen(fd[0], "r")) == NULL) {
			close(fd[1]); close(fd[0]);
			exit(1);
		}
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "GET %s", filename);
		SendFD(buffer, &fd[1]);
		return fp;
	} else {
		return fopen(filename, "r");
	}
}

static int open_write(const char *filename) {
	if (offline_mode) {
		char buffer[1024];
		int fd[2];
		if (socketpair(PF_UNIX, SOCK_STREAM, 0, fd)) {
			fprintf(stderr, "socketpair()\n");
			exit(1);
		}
		if (shutdown(fd[0], SHUT_RD)) {
			close(fd[1]); close(fd[0]);
			exit(1);
		}
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "POST %s", filename);
		SendFD(buffer, &fd[1]);
		return fd[0];
	} else {
		return open(filename, O_WRONLY);
	}
}

static void ReadGenericPolicy(void) {
	FILE *fp;
	int i, j;
	while (generic_acl_list_count) free(generic_acl_list[--generic_acl_list_count]);
	if ((fp = open_read(policy_file)) != NULL) {
		if (current_screen == SCREEN_ACL_LIST) {
			int flag = 0;
			get();
			while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
				char *cp = strchr(buffer, '\n');
				if (cp) *cp = '\0';
				else if (!feof(fp)) break;
				NormalizeLine(buffer);
				if (IsDomainDef(buffer)) {
					flag = strcmp(buffer, current_domain) == 0 ? 1 : 0;
				} else if (flag && buffer[0]) {
					if ((generic_acl_list = (char **) realloc(generic_acl_list, (generic_acl_list_count + 1) * sizeof(char *))) == NULL
						|| (generic_acl_list[generic_acl_list_count++] = strdup(buffer)) == NULL) OutOfMemory();
				}
			}
			put();
			for (i = 0; i < generic_acl_list_count; i++) {
				for (j = i + 1; j < generic_acl_list_count; j++) {
					char *a = generic_acl_list[i], *b = generic_acl_list[j];
					if (*a && *b && strcmp(a + 1, b + 1) > 0) {
						generic_acl_list[i] = b; generic_acl_list[j] = a;
					}
				}
			}
		} else {
			get();
			while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
				char *cp = strchr(buffer, '\n');
				if (cp) *cp = '\0';
				else if (!feof(fp)) break;
				NormalizeLine(buffer);
				if ((generic_acl_list = (char **) realloc(generic_acl_list, (generic_acl_list_count + 1) * sizeof(char *))) == NULL
					|| (generic_acl_list[generic_acl_list_count++] = strdup(buffer)) == NULL) OutOfMemory();
			}
			put();
			for (i = 0; i < generic_acl_list_count; i++) {
				for (j = i + 1; j < generic_acl_list_count; j++) {
					char *a = generic_acl_list[i], *b = generic_acl_list[j];
					if (strcmp(a, b) > 0) {
						generic_acl_list[i] = b; generic_acl_list[j] = a;
					}
				}
			}
		}
		fclose(fp);
	}
}

static void ReadDomainPolicy(void) {
	FILE *fp;
	int i, j;
	while (domain_list_count) free(domain_list[--domain_list_count].domainname);
	while (initializers_count) free(initializers[--initializers_count]);
	while (domain_keepers_count) free(domain_keepers[--domain_keepers_count]);
	FindOrAssignNewDomain(ROOT_NAME);

	// Load initializer list, domain_keeper list.
	if ((fp = open_read(EXCEPTION_POLICY_FILE)) != NULL) {
		get();
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (strncmp(buffer, "initializer ", 12) == 0) {
				memmove(buffer, buffer + 12, strlen(buffer + 12) + 1);
				if (!IsCorrectPath(buffer, 0)) continue;
				if ((initializers = (char **) realloc(initializers, (initializers_count + 1) * sizeof(char *))) == NULL
					|| (initializers[initializers_count++] = strdup(buffer)) == NULL) OutOfMemory();
			} else if (strncmp(buffer, "keep_domain ", 12) == 0) {
				memmove(buffer, buffer + 12, strlen(buffer + 12) + 1);
				if (!IsCorrectDomain(buffer)) continue;
				if ((domain_keepers = (char **) realloc(domain_keepers, (domain_keepers_count + 1) * sizeof(char *))) == NULL
					|| (domain_keepers[domain_keepers_count++] = strdup(buffer)) == NULL) OutOfMemory();
			}
		}
		put();
		fclose(fp);
	}

	// Load all domain list.
	if ((fp = open_read(DOMAIN_POLICY_FILE)) != NULL) {
		int index = EOF;
		get();
		while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
			char *cp = strchr(buffer, '\n');
			if (cp) *cp = '\0';
			else if (!feof(fp)) break;
			NormalizeLine(buffer);
			if (IsDomainDef(buffer)) {
				index = FindOrAssignNewDomain(buffer);
			} else if (index >= 0 && (atoi(buffer) & 1) == 1 && (cp = strchr(buffer, ' ')) != NULL) {
				char *cp2 = strstr(cp + 1, " if ");
				if (cp2) *cp2 = '\0';
				if (IsCorrectPath(cp + 1, 0) && !strendswith(cp + 1, "/") && IsInitializer(cp + 1)) {
					int virtual_index;
					static char buffer2[MAXBUFSIZE];
					memset(buffer2, 0, sizeof(buffer2));
					snprintf(buffer2, sizeof(buffer2) - 1, "%s %s", domain_list[index].domainname, cp + 1);
					NormalizeLine(buffer2);
					if ((virtual_index = FindOrAssignNewDomain(buffer2)) == EOF) {
						fprintf(stderr, "ERROR: Can't create domain for '%s'.\n", buffer2);
						exit(127);
					}
					domain_list[virtual_index].is_virtual = 1;
				}
			}
		}
		put();
		fclose(fp);
	}
	// Clear is_virtual flag for initialized domains.
	for (i = 0; i < domain_list_count; i++) {
		char *cp;
		if (!domain_list[i].is_virtual) continue;
		if ((cp = strchr(domain_list[i].domainname, ' ')) != NULL && strchr(cp + 1, ' ') == NULL) domain_list[i].is_virtual = 0;
	}
	// Sort by domain name.
	for (i = 0; i < domain_list_count; i++) {
		for (j = i + 1; j < domain_list_count; j++) {
			if (strcmp(domain_list[i].domainname, domain_list[j].domainname) > 0) {
				DOMAIN_INFO tmp = domain_list[i]; domain_list[i] = domain_list[j]; domain_list[j] = tmp;
			}
		}
	}
}

////////////////////////////  UI HANDLER  ////////////////////////////

static int DomainListLoop(void);
static int GenericListLoop(void);
static void ShowList(void);

static void ResizeWindow(void);
static void UpArrowKey(void);
static void DownArrowKey(void);
static void PageUpKey(void);
static void PageDownKey(void);
static void ShowCurrent(void);
static int GetCurrent(void);

static int window_width = 0, window_height = 0;
static int current_y[MAXSCREEN], current_item_index[MAXSCREEN], list_item_count[MAXSCREEN];

static struct {
	int current_y[MAXSCREEN];
	int current_item_index[MAXSCREEN];
	char search_buffer[4096];
} history;

static const int header_lines = 3;
static int body_lines = 0;

static void ShowList(void) {
	const int offset = current_item_index[current_screen];
	int i;
	if (current_screen == SCREEN_DOMAIN_LIST) list_item_count[SCREEN_DOMAIN_LIST] = domain_list_count;
	else list_item_count[current_screen] = generic_acl_list_count;
	clear();
	if (window_height < header_lines + 1) {
		mvprintw(0, 0, "Please resize window. This program needs at least %d lines.\n", header_lines + 1);
		refresh();
		return;
	}
	if (current_screen == SCREEN_DOMAIN_LIST) mvprintw(0, 0, "<<< Domain Transition Editor >>>      %d domain%c", list_item_count[SCREEN_DOMAIN_LIST], list_item_count[SCREEN_DOMAIN_LIST] > 1 ? 's' : ' ');
	else mvprintw(0, 0, "<<< %s Editor >>>      %d entr%s", list_caption, list_item_count[current_screen], list_item_count[current_screen] > 1 ? "ies" : "y");
	if (current_screen == SCREEN_DOMAIN_LIST) mvprintw(1, 0, "Commands = Q:quit A:append D:delete R:refresh Enter:edit @:search Tab:switch");
	else if (current_screen == SCREEN_ACL_LIST) mvprintw(1, 0, "Commands = Q:quit A:append D:delete R:refresh Enter:list");
	else mvprintw(1, 0, "Commands = Q:quit A:append D:delete R:refresh Tab:switch");
	if (current_screen == SCREEN_ACL_LIST) mvprintw(2, 0, "%s", current_domain);
	for (i = 0; i < body_lines; i++) {
		const int index = offset + i;
		if (index >= list_item_count[current_screen]) break;
		if (current_screen == SCREEN_DOMAIN_LIST) {
			char *sp, *cp;
			mvprintw(header_lines + i, 0, "%5d: %c%c ", index, IsDomainKeeper(index) ? '#' : ' ', IsInitializerDomain(index) ? '*' : ' ');
			sp = DomainName(index);
			while ((cp = strchr(sp, ' ')) != NULL) { printw("    "); sp = cp + 1; }
			printw("%s", sp);
			if (IsVirtualDomain(index)) {
				get();
				memset(buffer, 0, sizeof(buffer));
				snprintf(buffer, sizeof(buffer) - 1, ROOT_NAME "%s", strrchr(DomainName(index), ' '));
				printw(" ( -> %d )", FindDomain(buffer));
				put();
			}
		} else {
			mvprintw(header_lines + i, 0, "%5d: %s", index, generic_acl_list[index]);
		}
		clrtoeol();
	}
	ShowCurrent();
}

static void ResizeWindow(void) {
	getmaxyx(stdscr, window_height, window_width);
	body_lines = window_height - header_lines;
	if (body_lines <= current_y[current_screen]) current_y[current_screen] = body_lines - 1;
	if (current_y[current_screen] < 0) current_y[current_screen] = 0;
}

static void UpArrowKey(void) {
	if (current_y[current_screen] > 0) {
		current_y[current_screen]--;
		ShowCurrent();
	} else if (current_item_index[current_screen] > 0) {
		current_item_index[current_screen]--;
		ShowList();
	}
}

static void DownArrowKey(void) {
	if (current_y[current_screen] < body_lines - 1) {
		if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
			current_y[current_screen]++;
			ShowCurrent();
		}
	} else if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
		current_item_index[current_screen]++;
		ShowList();
	}
}

static void PageUpKey(void) {
	if (current_item_index[current_screen] + current_y[current_screen] > body_lines) {
		current_item_index[current_screen] -= body_lines;
		if (current_item_index[current_screen] < 0) current_item_index[current_screen] = 0;
		ShowList();
	} else if (current_item_index[current_screen] + current_y[current_screen] > 0) {
		current_item_index[current_screen] = 0;
		current_y[current_screen] = 0;
		ShowList();
	}
}

static void PageDownKey(void) {
	if (list_item_count[current_screen] - current_item_index[current_screen] > body_lines) {
		current_item_index[current_screen] += body_lines;
		if (current_item_index[current_screen] + current_y[current_screen] > list_item_count[current_screen] - 1) current_y[current_screen] = list_item_count[current_screen] - 1 - current_item_index[current_screen];
		ShowList();
	} else if (current_item_index[current_screen] + current_y[current_screen] < list_item_count[current_screen] - 1) {
		current_y[current_screen] = list_item_count[current_screen] - current_item_index[current_screen] - 1;
		ShowCurrent();
	}
}

static int GetCurrent(void) {
	if (list_item_count[current_screen] == 0) return EOF;
	if (current_item_index[current_screen] + current_y[current_screen] < 0 || current_item_index[current_screen] + current_y[current_screen] >= list_item_count[current_screen]) {
		fprintf(stderr, "ERROR: current_item_index=%d current_y=%d\n", current_item_index[current_screen], current_y[current_screen]);
		exit(127);
	}
	return current_item_index[current_screen] + current_y[current_screen];
}

static void ShowCurrent(void) {
	if (current_screen == SCREEN_DOMAIN_LIST) {
		const int current = GetCurrent();
		get();
		memset(buffer, 0, sizeof(buffer));
		snprintf(buffer, sizeof(buffer) - 1, "%s", DomainName(current));
		if (window_width < sizeof(buffer)) buffer[window_width] = '\0';
		move(2, 0);
		clrtoeol();
		printw("%s", buffer);
		put();
	}
	move(header_lines + current_y[current_screen], 0);
	refresh();
}

static void LoadCursorPos(const int current_screen) {
	current_item_index[current_screen] = history.current_item_index[current_screen];
	current_y[current_screen] = history.current_y[current_screen];
}

static void AdjustCursorPos(const int item_count) {
	if (item_count == 0) {
		current_item_index[current_screen] = current_y[current_screen] = 0;
	} else {
		while (current_item_index[current_screen] + current_y[current_screen] >= item_count) {
			if (current_y[current_screen] > 0) current_y[current_screen]--;
			else if (current_item_index[current_screen] > 0) current_item_index[current_screen]--;
		}
	}
}

static void SaveCursorPos(const int current_screen) {
	history.current_item_index[current_screen] = current_item_index[current_screen];
	history.current_y[current_screen] = current_y[current_screen];
}

static int GenericListLoop(void) {
	if (current_screen == SCREEN_SYSTEM_LIST) {
		policy_file = SYSTEM_POLICY_FILE;
		list_caption = "System Policy";
	} else if (current_screen == SCREEN_EXCEPTION_LIST) {
		policy_file = EXCEPTION_POLICY_FILE;
		list_caption = "Exception Policy";
	} else {
		policy_file = DOMAIN_POLICY_FILE;
		list_caption = "Domain Policy";
	}
	LoadCursorPos(current_screen);
 start:
	ReadGenericPolicy();
	AdjustCursorPos(generic_acl_list_count);
	ShowList();
	while (1) {
		const int c = getch2();
		SaveCursorPos(current_screen);
		if (c == 'q' || c == 'Q') return MAXSCREEN;
		if (current_screen == SCREEN_ACL_LIST) {
			if (c == '\r') return SCREEN_DOMAIN_LIST;
		} else {
			if (c == '\t') {
				if (current_screen == SCREEN_SYSTEM_LIST) return SCREEN_EXCEPTION_LIST;
				else if (current_screen == SCREEN_EXCEPTION_LIST) return SCREEN_DOMAIN_LIST;
			}
		}
		if (c == ERR) continue; // Ignore invalid key.
		switch(c) {
		case KEY_RESIZE:
			ResizeWindow();
			ShowList();
			break;
		case KEY_UP:
			UpArrowKey();
			break;
		case KEY_DOWN:
			DownArrowKey();
			break;
		case KEY_PPAGE:
			PageUpKey();
			break;
		case KEY_NPAGE:
			PageDownKey();
			break;
		case 'd':
		case 'D':
			{
				const int index = GetCurrent();
				if (index >= 0 && index < generic_acl_list_count) {
					const int fd = open_write(policy_file);
					if (fd != EOF) {
						if (current_screen == SCREEN_ACL_LIST) {
							write(fd, "select ", 7); write(fd, current_domain, strlen(current_domain)); write(fd, "\n", 1);
						}
						write(fd, "delete ", 7); write(fd, generic_acl_list[index], strlen(generic_acl_list[index])); write(fd, "\n", 1); close(fd);
						close(fd);
					}
					goto start;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				get();
				mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
				memset(buffer, 0, sizeof(buffer));
				scrollok(stdscr, 1); getnstr(buffer, sizeof(buffer) - 1); scrollok(stdscr, 0); noecho();
				NormalizeLine(buffer);
				if (buffer[0]) {
					const int fd = open_write(policy_file);
					if (fd != EOF) {
						if (current_screen == SCREEN_ACL_LIST) {
							write(fd, "select ", 7); write(fd, current_domain, strlen(current_domain)); write(fd, "\n", 1);
						}
						write(fd, buffer, strlen(buffer)); write(fd, "\n", 1); close(fd);
					}
				}
				put();
				goto start;
			}
			break;
		case 'r':
		case 'R':
			goto start;
		}
	}
}

static int DomainListLoop(void) {
	LoadCursorPos(current_screen);
 start: ;
	ReadDomainPolicy();
	AdjustCursorPos(domain_list_count);
	ShowList();
	while (1) {
		const int c = getch2();
		SaveCursorPos(current_screen);
		if (c == 'q' || c == 'Q') return MAXSCREEN;
		if (c == '\t') return SCREEN_SYSTEM_LIST;
		if (c == ERR) continue; // Ignore invalid key.
		switch (c) {
		case KEY_RESIZE:
			ResizeWindow();
			ShowList();
			break;
		case KEY_UP:
			UpArrowKey();
			break;
		case KEY_DOWN:
			DownArrowKey();
			break;
		case KEY_PPAGE:
			PageUpKey();
			break;
		case KEY_NPAGE:
			PageDownKey();
			break;
		case '@':
			{
				int i;
			input_path:
				mvprintw(window_height - 1, 0, "Search> "); clrtoeol(); refresh(); echo();
				memset(history.search_buffer, 0, sizeof(history.search_buffer));
				scrollok(stdscr, 1); getnstr(history.search_buffer, sizeof(history.search_buffer) - 1); scrollok(stdscr, 0); noecho();
				NormalizeLine(history.search_buffer);
				if (history.search_buffer[0]) {
					for (i = 0; i < list_item_count[current_screen]; i++) {
						char *cp = DomainName(i);
						if (strchr(history.search_buffer, '/')) {
							if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
						} else {
							if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
						}
						if (strcmp(cp, history.search_buffer)) continue;
						while (i < current_y[current_screen] + current_item_index[current_screen]) {
							if (current_y[current_screen] > 0) current_y[current_screen]--;
							else current_item_index[current_screen]--;
						}
						while (i > current_y[current_screen] + current_item_index[current_screen]) {
							if (current_y[current_screen] < body_lines - 1) current_y[current_screen]++;
							else current_item_index[current_screen]++;
						}
						break;
					}
				}
				ShowList();
			}
			break;
		case KEY_LEFT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent() - 1; i >= 0; i--) {
					char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i < current_y[current_screen] + current_item_index[current_screen]) {
						if (current_y[current_screen] > 0) current_y[current_screen]--;
						else current_item_index[current_screen]--;
					}
					ShowList();
					break;
				}
			}
			break;
		case KEY_RIGHT:
			{
				int i;
				if (!history.search_buffer[0]) goto input_path;
				for (i = GetCurrent() + 1; i < list_item_count[current_screen]; i++) {
					char *cp = DomainName(i);
					if (strchr(history.search_buffer, '/')) {
						if (strrchr(cp, ' ')) cp = strrchr(cp, ' ') + 1;
					} else {
						if (strrchr(cp, '/')) cp = strrchr(cp, '/') + 1;
					}
					if (strcmp(cp, history.search_buffer)) continue;
					while (i > current_y[current_screen] + current_item_index[current_screen]) {
						if (current_y[current_screen] < body_lines - 1) current_y[current_screen]++;
						else current_item_index[current_screen]++;
					}
					ShowList();
					break;
				}
			}
			break;
		case 'a':
		case 'A':
			{
				const int current = GetCurrent();
				if (!IsVirtualDomain(current)) {
					get();
					mvprintw(window_height - 1, 0, "Enter new entry> "); clrtoeol(); refresh(); echo();
					memset(buffer, 0, sizeof(buffer));
					scrollok(stdscr, 1); getnstr(buffer, sizeof(buffer) - 1); scrollok(stdscr, 0); noecho();
					NormalizeLine(buffer);
					if (IsCorrectPath(buffer, 0) && !IsInitializer(buffer)) {
						const int fd = open_write(DOMAIN_POLICY_FILE);
						if (fd != EOF) {
							char *domainname = domain_list[current].domainname;
							write(fd, domainname, strlen(domainname)); write(fd, " ", 1); write(fd, buffer, strlen(buffer)); write(fd, "\n", 1);
							close(fd);
						}
					}
					put();
					goto start;
				}
			}
			break;
		case 'd':
		case 'D':
			{
				const int current = GetCurrent();
				if (current > 0 && !IsVirtualDomain(current)) { // Never delete ROOT_NAME
					char *key = DomainName(current);
					int c;
					mvprintw(header_lines, 0, "Delete '%s' and its descendant domains? (Y/N):", key);
					clrtoeol();
					refresh();
					do {
						c = getch2();
					} while (!(c == 'Y' || c == 'y' || c == 'N' || c == 'n' || c == EOF));
					ResizeWindow();
					if (c == 'Y' || c == 'y') {
						const int fd = open_write(DOMAIN_POLICY_FILE);
						if (fd != EOF) {
							const int key_len = strlen(key);
							int index;
							for (index = 0; index < domain_list_count; index++) {
								char *cp = DomainName(index);
								if (strncmp(cp, key, key_len) || (cp[key_len] != '\0' && cp[key_len] != ' ')) continue;
								write(fd, "delete ", 7); write(fd, cp, strlen(cp)); write(fd, "\n", 1);
							}
							close(fd);
						}						
						goto start;
					}
					ShowList();
				}
			}
			break;
		case 'r':
		case 'R':
			goto start;
		case '\r':
			{
				const int current = GetCurrent();
				if (IsVirtualDomain(current)) {
					int redirect_index;
					get();
					memset(buffer, 0, sizeof(buffer));
					snprintf(buffer, sizeof(buffer) - 1, ROOT_NAME "%s", strrchr(DomainName(current), ' '));
					redirect_index = FindDomain(buffer);
					put();
					if (redirect_index != EOF) {
						current_item_index[current_screen] = redirect_index - current_y[current_screen];
						while (current_item_index[current_screen] < 0) {
							current_item_index[current_screen]++; current_y[current_screen]--;
						}
						ShowList();
					}
				} else {
					free(current_domain);
					if ((current_domain = strdup(domain_list[current].domainname)) == NULL) OutOfMemory();
					return SCREEN_ACL_LIST;
				}
			}
			break;
		}
	}
}

static void policy_daemon(void);

int main(int argc, char *argv[]) {
	memset(&history, 0, sizeof(history));
	memset(current_y, 0, sizeof(current_y));
	memset(current_item_index, 0, sizeof(current_item_index));
	memset(list_item_count, 0, sizeof(list_item_count));
	if (argc > 1) {
		if (strcmp(argv[1], "s") == 0) current_screen = SCREEN_SYSTEM_LIST;
		else if (strcmp(argv[1], "e") == 0) current_screen = SCREEN_EXCEPTION_LIST;
		else if (strcmp(argv[1], "d") == 0) current_screen = SCREEN_DOMAIN_LIST;
		else {
			printf("Usage: %s [s|e|d]\n", argv[0]);
			return 1;
		}
	}
	{
		char *cp = strrchr(argv[0], '/');
		if (!cp) cp = argv[0];
		else cp++;
		if (strcmp(cp, "editpolicy_offline") == 0) offline_mode = 1;
	}
	if (offline_mode) {
		int fd[2];
		if (chdir("/etc/ccs/")) {
			fprintf(stderr, "/etc/ccs/ doesn't exist.\n");
			return 1;
		}
		if (socketpair(PF_UNIX, SOCK_STREAM, 0, fd)) {
			fprintf(stderr, "socketpair()\n");
			exit(1);
		}
		switch (fork()) {
		case 0:
			close(fd[0]);
			persistent_fd = fd[1];
			policy_daemon();
			_exit(0);
		case -1:
			fprintf(stderr, "fork()\n");
			exit(1);
		}
		close(fd[1]);
		persistent_fd = fd[0];
		{
			int fd0, fd1, len;
			if ((fd0 = open("system_policy.txt", O_RDONLY)) != EOF) {
				fd1 = open_write(SYSTEM_POLICY_FILE);
				while ((len = read(fd0, buffer, sizeof(buffer))) > 0) write(fd1, buffer, len);
				close(fd1); close(fd0);
			}
			if ((fd0 = open("exception_policy.txt", O_RDONLY)) != EOF) {
				fd1 = open_write(EXCEPTION_POLICY_FILE);
				while ((len = read(fd0, buffer, sizeof(buffer))) > 0) write(fd1, buffer, len);
				close(fd1); close(fd0);
			}
			if ((fd0 = open("domain_policy.txt", O_RDONLY)) != EOF) {
				fd1 = open_write(DOMAIN_POLICY_FILE);
				while ((len = read(fd0, buffer, sizeof(buffer))) > 0) write(fd1, buffer, len);
				close(fd1); close(fd0);
			}
		}
	} else {
		if (chdir("/proc/ccs/policy/")) {
			fprintf(stderr, "You can't use this editor for this kernel.\n");
			return 1;
		}
		{
			const int fd1 = open(SYSTEM_POLICY_FILE, O_RDWR), fd2 = open(EXCEPTION_POLICY_FILE, O_RDWR), fd3 = open(DOMAIN_POLICY_FILE, O_RDWR);
			if ((fd1 != EOF && write(fd1, "", 0) != 0) || (fd2 != EOF && write(fd2, "", 0) != 0) || (fd3 != EOF && write(fd3, "", 0) != 0)) {
				fprintf(stderr, "You need to register this program to /proc/ccs/policy/manager to run this program.\n");
				return 1;
			}
			close(fd1); close(fd2); close(fd3);
		}
	}
	initscr();
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);
	keypad(stdscr, TRUE);
	getmaxyx(stdscr, window_height, window_width);
	while (current_screen < MAXSCREEN) {
		if (!offline_mode) {
			if (current_screen == SCREEN_DOMAIN_LIST && access(DOMAIN_POLICY_FILE, F_OK)) current_screen = SCREEN_SYSTEM_LIST;
			else if (current_screen == SCREEN_SYSTEM_LIST && access(SYSTEM_POLICY_FILE, F_OK)) current_screen = SCREEN_EXCEPTION_LIST;
			else if (current_screen == SCREEN_EXCEPTION_LIST && access(EXCEPTION_POLICY_FILE, F_OK)) current_screen = SCREEN_DOMAIN_LIST;
		}
		ResizeWindow();
		if (current_screen == SCREEN_DOMAIN_LIST) current_screen = DomainListLoop();
		else current_screen = GenericListLoop();
	}
	clear();
	move(0, 0);
	refresh();
	endwin();
	if (offline_mode) {
		int fd, len;
		FILE *fp;
		time_t now = time(NULL);
		struct tm *tm = localtime(&now);
		char filename[1024];
		memset(filename, 0, sizeof(filename));
		snprintf(filename, sizeof(filename) - 1, "system_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
			if ((fp = open_read(SYSTEM_POLICY_FILE)) != NULL) {
				get(); while ((len = fread(buffer, 1, sizeof(buffer), fp)) > 0) write(fd, buffer, len); put();
				close(fd); fclose(fp);
				unlink("system_policy.txt");
				symlink(filename, "system_policy.txt");
			}
		}
		snprintf(filename, sizeof(filename) - 1, "exception_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
			if ((fp = open_read(EXCEPTION_POLICY_FILE)) != NULL) {
				get(); while ((len = fread(buffer, 1, sizeof(buffer), fp)) > 0) write(fd, buffer, len); put();
				close(fd); fclose(fp);
				unlink("exception_policy.txt");
				symlink(filename, "exception_policy.txt");
			}
		}
		snprintf(filename, sizeof(filename) - 1, "domain_policy.%02d-%02d-%02d.%02d:%02d:%02d.txt", tm->tm_year % 100, tm->tm_mon + 1, tm->tm_mday, tm->tm_hour, tm->tm_min, tm->tm_sec);
		if ((fd = open(filename, O_WRONLY | O_CREAT, 0600)) != EOF) {
			if ((fp = open_read(DOMAIN_POLICY_FILE)) != NULL) {
				get(); while ((len = fread(buffer, 1, sizeof(buffer), fp)) > 0) write(fd, buffer, len); put();
				close(fd); fclose(fp);
				unlink("domain_policy.txt");
				symlink(filename, "domain_policy.txt");
			}
		}
	}
	return 0;
}



/***** The code below is used only when offline mode. *****/

/* Copied from kernel source. */
static inline unsigned long partial_name_hash(unsigned long c, unsigned long prevhash) {
	return (prevhash + (c << 4) + (c >> 4)) * 11;
}

/* Copied from kernel source. */
static inline unsigned int full_name_hash(const unsigned char *name, unsigned int len) {
	unsigned long hash = 0;
	while (len--) hash = partial_name_hash(*name++, hash);
	return (unsigned int) hash;
}

#define PAGE_SIZE  4096

static char *alloc_element(const unsigned int size) {
	static char *buf = NULL;
	static unsigned int buf_used_len = PAGE_SIZE;
	char *ptr = NULL;
	if (size > PAGE_SIZE) return NULL;
	if (buf_used_len + size > PAGE_SIZE) {
		if ((ptr = malloc(PAGE_SIZE)) == NULL) OutOfMemory();
		buf = ptr;
		memset(buf, 0, PAGE_SIZE);
		buf_used_len = size;
		ptr = buf;
	} else if (size) {
		int i;
		ptr = buf + buf_used_len;
		buf_used_len += size;
		for (i = 0; i < size; i++) if (ptr[i]) OutOfMemory();
	}
	return ptr;
}

#define MAX_HASH 256

typedef struct name_entry {
	struct name_entry *next; /* Pointer to next record. NULL if none.             */
	unsigned int hash;       /* hash and length                                   */
	const char *name;        /* Text form of filename and domainname. Never NULL. */
} NAME_ENTRY;

typedef struct free_memory_block_list {
	struct free_memory_block_list *next; /* Pointer to next record. NULL if none. */
	char *ptr;                           /* Pointer to a free area.               */
	int len;                             /* Length of the area.                   */
} FREE_MEMORY_BLOCK_LIST;

static const char *SaveName(const char *name) {
	static FREE_MEMORY_BLOCK_LIST fmb_list = { NULL, NULL, 0 };
	static NAME_ENTRY name_list[MAX_HASH]; /* The list of names. */
	NAME_ENTRY *ptr, *prev = NULL;
	unsigned int hash;
	FREE_MEMORY_BLOCK_LIST *fmb = &fmb_list;
	int len;
	static int first_call = 1;
	if (!name) return NULL;
	len = strlen(name) + 1;
	if (len > PAGE_SIZE) {
		fprintf(stderr, "ERROR: Name too long for SaveName().\n");
		return NULL;
	}
	hash = full_name_hash((const unsigned char *) name, len - 1);
	if (first_call) {
		int i;
		first_call = 0;
		memset(&name_list, 0, sizeof(name_list));
		for (i = 0; i < MAX_HASH; i++) name_list[i].name = "/";
	}
	ptr = &name_list[hash % MAX_HASH];
	hash ^= len; /* The hash % MAX_HASH are always same for ptr->hash, so embed length into the hash value. */
	while (ptr) {
		if (hash == ptr->hash && strcmp(name, ptr->name) == 0) goto out;
		prev = ptr; ptr = ptr->next;
	}
	while (len > fmb->len) {
		if (fmb->next) {
			fmb = fmb->next;
		} else {
			char *cp;
			if ((cp = (char *) malloc(PAGE_SIZE)) == NULL || (fmb->next = (FREE_MEMORY_BLOCK_LIST *) alloc_element(sizeof(FREE_MEMORY_BLOCK_LIST))) == NULL) OutOfMemory();
			memset(cp, 0, PAGE_SIZE);
			fmb = fmb->next;
			fmb->ptr = cp;
			fmb->len = PAGE_SIZE;
		}
	}
	if ((ptr = (NAME_ENTRY *) alloc_element(sizeof(NAME_ENTRY))) == NULL) OutOfMemory();
	memset(ptr, 0, sizeof(NAME_ENTRY));
	ptr->hash = hash;
	ptr->name = fmb->ptr;
	memmove(fmb->ptr, name, len);
	fmb->ptr += len;
	fmb->len -= len;
	prev->next = ptr; /* prev != NULL because name_list is not empty. */
	if (fmb->len == 0) {
		FREE_MEMORY_BLOCK_LIST *ptr = &fmb_list;
		while (ptr->next != fmb) ptr = ptr->next; ptr->next = fmb->next;
	}
 out:
	return ptr ? (const char *) ptr->name : NULL;
}

static void DeleteDomain(const int index) {
	if (index > 0 && index < domain_list_count) {
		int i;
		free(domain_list[index].string_ptr);
		for (i = index; i < domain_list_count - 1; i++) domain_list[i] = domain_list[i + 1];
		domain_list_count--;
	}
}

static int AddStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		fprintf(stderr, "AddStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	if ((cp = SaveName(entry)) == NULL) OutOfMemory();

	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;

	// Check for the same entry.
	for (i = 0; i < acl_count; i++) {
		// Faster comparison, for they are SaveName'd.
		if (cp == acl_ptr[i]) return 0;
	}

	if ((acl_ptr = (const char **) realloc(acl_ptr, (acl_count + 1) * sizeof(const char *))) == NULL) OutOfMemory();
	acl_ptr[acl_count++] = cp;
	domain_list[index].string_ptr = acl_ptr;
	domain_list[index].string_count = acl_count;
	return 0;
}

static int DelStringEntry(const char *entry, const int index) {
	const char **acl_ptr;
	int acl_count;
	const char *cp;
	int i;
	if (index < 0 || index >= domain_list_count) {
		fprintf(stderr, "DelStringEntry: ERROR: domain is out of range.\n");
		return -EINVAL;
	}
	if (!entry || !*entry) return -EINVAL;
	if ((cp = SaveName(entry)) == NULL) OutOfMemory();

	acl_ptr = domain_list[index].string_ptr;
	acl_count = domain_list[index].string_count;

	for (i = 0; i < acl_count; i++) {
		// Faster comparison, for they are SaveName'd.
		if (cp != acl_ptr[i]) continue;
		domain_list[index].string_count--;
		for (; i < acl_count - 1; i++) acl_ptr[i] = acl_ptr[i + 1];
		return 0;
	}
	return -ENOENT;
}

static void policy_daemon(void) {
	while (1) {
		static const char **exception_list = NULL, **system_list = NULL;
		static int exception_list_count = 0, system_list_count = 0;
		FILE *fp;
		{
			struct msghdr msg;
			struct iovec iov = { buffer, sizeof(buffer) - 1 };
			char cmsg_buf[CMSG_SPACE(sizeof(int))];
			struct cmsghdr *cmsg = (struct cmsghdr *) cmsg_buf;
			memset(&msg, 0, sizeof(msg));
			msg.msg_iov = &iov;
			msg.msg_iovlen = 1;
			msg.msg_control = cmsg_buf;
			msg.msg_controllen = sizeof(cmsg_buf);
			memset(buffer, 0, sizeof(buffer));
			errno = 0;
			if (recvmsg(persistent_fd, &msg, 0) > 0 &&
				(cmsg = CMSG_FIRSTHDR(&msg)) != NULL &&
				cmsg->cmsg_level == SOL_SOCKET &&
				cmsg->cmsg_type == SCM_RIGHTS &&
				cmsg->cmsg_len == CMSG_LEN(sizeof(int))) {
				const int fd = * (int *) CMSG_DATA(cmsg);
				if ((fp = fdopen(fd, "w+")) == NULL) {
					close(fd);
					continue;
				}
			} else {
				break;
			}
		}
		if (strncmp(buffer, "POST ", 5) == 0) {
			if (strcmp(buffer + 5, "domain_policy") == 0) {
				int index = EOF;
				while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
					int is_delete = 0, is_select = 0;
					char *cp = strchr(buffer, '\n');
					if (cp) *cp = '\0';
					else if (!feof(fp)) break;
					NormalizeLine(buffer);
					if (strncmp(buffer, "delete ", 7) == 0) {
						is_delete = 1;
						memmove(buffer, buffer + 7, strlen(buffer + 7) + 1);
					} else if (strncmp(buffer, "select ", 7) == 0) {
						is_select = 1;
						memmove(buffer, buffer + 7, strlen(buffer + 7) + 1);
					}
					if (IsDomainDef(buffer)) {
						if (is_delete) {
							index = FindDomain(buffer);
							if (index > 0) DeleteDomain(index);
							index = EOF;
						} else if (is_select) {
							index = FindDomain(buffer);
						} else {
							index = FindOrAssignNewDomain(buffer);
						}
					} else if (index >= 0 && buffer[0]) {
						unsigned int profile;
						if (sscanf(buffer, "use_profile %u", &profile) == 1) {
							domain_list[index].profile = (unsigned char) profile;
						} else if (is_delete) {
							DelStringEntry(buffer, index);
						} else {
							AddStringEntry(buffer, index);
						}
					}
				}
			} else if (strcmp(buffer + 5, "exception_policy") == 0) {
				while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
					char *cp = strchr(buffer, '\n');
					if (cp) *cp = '\0';
					else if (!feof(fp)) break;
					NormalizeLine(buffer);
					if (!buffer[0]) continue;
					if (strncmp(buffer, "delete ", 7) == 0) {
						int i;
						memmove(buffer, buffer + 7, strlen(buffer + 7) + 1);
						for (i = 0; i < exception_list_count; i++) {
							if (strcmp(exception_list[i], buffer)) continue;
							for (exception_list_count--; i < exception_list_count; i++) exception_list[i] = exception_list[i + 1];
							break;
						}
					} else {
						if ((exception_list = (const char **) realloc(exception_list, (exception_list_count + 1) * sizeof(char *))) == NULL
							|| (exception_list[exception_list_count++] = SaveName(buffer)) == NULL) OutOfMemory();
					}
				}
			} else if (strcmp(buffer + 5, "system_policy") == 0) {
				while (memset(buffer, 0, sizeof(buffer)), fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
					char *cp = strchr(buffer, '\n');
					if (cp) *cp = '\0';
					else if (!feof(fp)) break;
					NormalizeLine(buffer);
					if (!buffer[0]) continue;
					if (strncmp(buffer, "delete ", 7) == 0) {
						int i;
						memmove(buffer, buffer + 7, strlen(buffer + 7) + 1);
						for (i = 0; i < system_list_count; i++) {
							if (strcmp(system_list[i], buffer)) continue;
							for (system_list_count--; i < system_list_count; i++) system_list[i] = system_list[i + 1];
							break;
						}
					} else {
						if ((system_list = (const char **) realloc(system_list, (system_list_count + 1) * sizeof(char *))) == NULL
							|| (system_list[system_list_count++] = SaveName(buffer)) == NULL) OutOfMemory();
					}
				}
			}
		} else if (strncmp(buffer, "GET ", 4) == 0) {
			if (strcmp(buffer + 4, "domain_policy") == 0) {
				int i, j;
				for (i = 0; i < domain_list_count; i++) {
					const char **string_ptr = domain_list[i].string_ptr;
					const int string_count = domain_list[i].string_count;
					fprintf(fp, "%s\nuse_profile %u\n\n", domain_list[i].domainname, domain_list[i].profile);
					for (j = 0; j < string_count; j++) {
						fprintf(fp, "%s\n", string_ptr[j]);
					}
					fprintf(fp, "\n");
				}
			} else if (strcmp(buffer + 4, "exception_policy") == 0) {
				int i;
				for (i = 0; i < exception_list_count; i++) fprintf(fp, "%s\n", exception_list[i]);
			} else if (strcmp(buffer + 4, "system_policy") == 0) {
				int i;
				for (i = 0; i < system_list_count; i++) fprintf(fp, "%s\n", system_list[i]);
			}
		}
		fclose(fp);
	}
	_exit(0);
}
