<?php
/*
 * PHP version 5
 *
 * @copyright Copyright 2010, Cake. (http://trpgtools-onweb.sourceforge.jp/)
 * @category Controller
 * @package  TRPG Data Bank
 * @version  beta
 * @author   Cake <cake_67@users.sourceforge.jp>
 * @license  http://www.opensource.org/licenses/mit-license.php The MIT License
 * @link     http://trpgtools-onweb.sourceforge.jp/
 */

class CharacterPicturesController extends AppController {

	var $name = 'CharacterPictures';
	var $helpers = array(
		'Time',
		'Select',
	);

	var $disableTokenActions = array();

	var $post_data = array();


	/* メソッド */

	function beforeFilter() {

		parent::beforeFilter();

		// 認証なしアクセス可
		$this->AuthPlus->allow('view');
		$this->AuthPlus->allow('index');
	}

	function beforeRender()
	{
		parent::beforeRender();

		$this->set_public_flag4view();
	}


	/* アクションメソッド */

	function index($character_id = null) {
		if (!$character_id) {
			$this->Session->setFlash(__('Invalid URL.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		list($character, $character_pictures) = $this->_index($character_id);
		if (empty($character)) {
			$this->Session->setFlash(__('Invalid Character.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		if ($this->isOwner($character['Character'], $this->user_id)) {
			$this->redirect(array('action'=>'listview', $character_id));
		}

		$this->set('character', $character);
		$this->set('characterPictures', $character_pictures);
		$this->set('isOwner', false);

		$this->set('maxPictures', false);
	}

	function listview($character_id = null) {
		if (!$character_id) {
			$this->Session->setFlash(__('Invalid URL.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		list($character, $character_pictures) = $this->_index($character_id);
		if (empty($character)) {
			$this->Session->setFlash(__('Invalid Character.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}
		if (!$this->isOwner($character['Character'], $this->user_id)) {
			$this->redirect(array('action'=>'index', $character_id));
		}

		$this->set('character', $character);
		$this->set('characterPictures', $character_pictures);
		$this->set('isOwner', true);

		$this->set('maxPictures', $this->check_character_picture_max(count($character_pictures)));
	}

	function view($id = null) {
		if (!$id) {
			$this->Session->setFlash(__('Invalid Id.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		$character_picture = $this->_view($id);
		$this->set('characterPicture', $character_picture);
	}

	function add($character_id = null) {
		if (!$character_id) {
			$this->Session->setFlash(__('Invalid URL.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		$character = $this->_get_character4character_id($character_id, $this->user_id);
		$this->set('character', $character);

		// 写真枚数チェック
		if (!$this->check_character_picture_max(count($character['CharacterPicture']))) {
			$this->Session->setFlash(__('Full CharacterPictures. Please delete for the new CharacterPicture.', true));
			$this->redirect(array('controller' => 'character_pictures', 'action'=>'listview', $character_id));
		}

		if (!empty($this->data)) {
			$this->data['clearCache'] = array(
				'character_id' => $character['Character']['id'],
				'user_id' => $this->user_id,
				'system_id' => null,
			);

			// 重複アップロードチェック
			if (count($this->data['Attachment']) > 1) {
				$this->Session->setFlash(__('Invalid data.', true));
				$this->redirect(array('controller' => 'character_pictures', 'action'=>'add', $character_id));
			}

			$this->data['Attachment'][0]['model'] = 'CharacterPicture';
			$this->data['CharacterPicture']['user_id'] = $this->user_id;
			$this->data['CharacterPicture']['character_id'] = $character_id;

			$this->CharacterPicture->create();
			if ($this->data['Attachment'][0]['file']['size']) {
				if ($this->CharacterPicture->saveAll(
					$this->data, 
					array(
						'validate' => 'first',
						'fieldList' => array_merge($this->CharacterPicture->fields['add'], $this->CharacterPicture->fields['image'])
					)
				)) {
					$this->Session->setFlash(__('The CharacterPicture has been saved', true));
					$this->redirect(array('action'=>'index', $character_id));
				}
			}

			$this->Session->setFlash(__('The data could not be saved. Please, try again.', true));
		}

		$this->set('isOwner', false);

		$this->set('title_for_layout', " - ". __('New Picture', true));
	}

	function edit($id = null) {
		if (!$id) {
			$this->Session->setFlash(__('Invalid CharacterPicture', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}
		if (!$character_picture = $this->_get_character4character_picture_id($id, $this->user_id)) {
			$this->Session->setFlash(__('No CharacterPicture', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		if (!empty($this->data)) {
			// 重複アップロードチェック
			if (count($this->data['Attachment']) > 1) {
				$this->Session->setFlash(__('Invalid data.', true));
				$this->redirect(array('controller' => 'character_pictures', 'action'=>'edit', $id));
			}

			$this->data['clearCache'] = array(
				'character_id' => $character_picture['Character']['id'],
				'user_id' => $this->user_id,
				'system_id' => null,
			);

			$this->data['CharacterPicture']['id'] = $id;
			$this->data['Attachment'][1]['id'] = $character_picture['Attachment'][0]['id'];

			if ($this->CharacterPicture->saveAll(
					$this->data, 
					array(
						'validate' => 'first',
						'fieldList' => array_merge($this->CharacterPicture->fields['edit'], $this->CharacterPicture->fields['image'])
					)
			)) {
				// characters設定の削除
				if ($this->data['CharacterPicture']['public_flag'] == 'private') {
					$this->_delete_setting($character_picture['Attachment'][0]['basename']);
				}

				$this->Session->setFlash(__('The CharacterPicture has been saved', true));
				$this->redirect(array('action'=>'listview', $character_picture['CharacterPicture']['character_id']));
			} else {
				$this->Session->setFlash(__('The data could not be saved. Please, try again.', true));
				$this->data['CharacterPicture']['id'] = $id;
			}
		}
		if (empty($this->data)) {
			$this->data = $character_picture;
		}

		$this->set('isOwner', true);

		$this->set('title_for_layout', " - ". __('Edit Picture', true));
	}

	function delete($id = null) {
		if (!$id) {
			$this->Session->setFlash(__('Invalid id for CharacterPicture', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}
		if (!$character_picture = $this->_get_character4character_picture_id($id, $this->user_id)) {
			$this->Session->setFlash(__('No CharacterPicture', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		if (!empty($this->data)) {
			$this->data['clearCache'] = array(
				'character_id' => $character_picture['Character']['id'],
				'user_id' => $this->user_id,
				'system_id' => null,
			);

			return $this->_delete($character_picture);
		}

		$this->set('isOwner', true);
		$this->set('isAdmin', false);
		$this->set('isDelete', true);
		$this->set('characterPicture', $character_picture);

		$this->set('title_for_layout', " - ". __('Delete Picture', true));
	}

	function admin_index($character_id = null) {
		if (!$character_id) {
			$this->Session->setFlash(__('Invalid URL.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		list($character, $character_pictures) = $this->_index($character_id, null, true);
		if (empty($character)) {
			$this->Session->setFlash(__('Invalid Character.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		$this->set('character', $character);
		$this->set('characterPictures', $character_pictures);
	}

	function admin_view($id = null) {
		if (!$id) {
			$this->Session->setFlash(__('Invalid CharacterPicture', true));
			$this->redirect(array('action'=>'index'));
		}
		if (!$id) {
			$this->Session->setFlash(__('Invalid Id.', true));
			$this->redirect(array('action'=>'index'));
		}

		$character_picture = $this->_view($id, null, true);
		$this->set('characterPicture', $character_picture);

		$this->set('title_for_layout', " - ". sprintf(__("%s's Picture", true), $character_picture['Character']['name']));
	}

	function admin_delete($id = null) {
		if (!$id) {
			$this->Session->setFlash(__('Invalid id for CharacterPicture', true));
			$this->redirect(array('action'=>'index'));
		}
		if (!$character_picture = $this->_get_character4character_picture_id($id)) {
			$this->Session->setFlash(__('No CharacterPicture', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'admin_index'));
		}

		if (!empty($this->data)) {
			return $this->_delete($character_picture);
		}

		$this->set('isOwner', false);
		$this->set('isAdmin', true);
		$this->set('isDelete', true);
		$this->set('characterPicture', $character_picture);

		$this->set('title_for_layout', " - ". __('Delete Picture', true));
	}


	function _index($character_id, $public_flag = '', $isAdmin = false)
	{
		// キャラデータ取得
		$conditions = array(
			'Character.id' => $character_id,
			'Character.deleted' => 0,
		// todo:public_flagチェック
		);
		$character = $this->CharacterPicture->Character->find('first', array(
			'conditions' => $conditions,
			'contain' => array(
			),
			'recursive' => -1,
		));
		if (empty($character)) {
			$this->Session->setFlash(__('Invalid Character.', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}

		$this->set('title_for_layout', " - ". sprintf(__("%s's Pictures", true), $character['Character']['name']));

		if (!$isAdmin) {
			$isOwner = $this->isOwner($character['Character'], $this->user_id);
		} else {
			$isOwner = false;
		}

		// Pictures取得
		$conditions = array(
			'CharacterPicture.character_id' => $character_id,
		);
		if (!$isAdmin && !$isOwner) {
			$conditions['CharacterPicture.public_flag'] = 'public';
		}
		$character_pictures = $this->CharacterPicture->find('all', array(
			'conditions' => $conditions,
			'fields' => array(
				'CharacterPicture.id',
				'CharacterPicture.public_flag',
				'CharacterPicture.created',
			),
			'contain' => array(
				'Attachment',
			),
			'recursive' => 1,
			'order' => array('CharacterPicture.id' => 'asc'),
		));
		if ($isOwner || $isAdmin) {
			$character_pictures = $this->_set_image_config($character_pictures, $character['Character']);
		}

		return array($character, $character_pictures);
	}

	function _view($id, $public_flag = '', $isAdmin = false)
	{
		$character_picture = $this->_get_character4character_picture_id($id, null, $public_flag);

		if (empty($character_picture)) {
			$this->Session->setFlash(__('Invalid CharacterPicture', true));
			$this->redirect(array('controller' => 'characters', 'action'=>'index'));
		}
		// public_flagチェック
		if (!$isAdmin) {
			if (!$this->check_public_flag($character_picture['Character'])) {
				$this->Session->setFlash(__('No Permission', true));
				$this->redirect(array('controller' => 'characters', 'action'=>'index'));
			}
			if (!$this->check_public_flag($character_picture['CharacterPicture'])) {
				$this->Session->setFlash(__('No Permission', true));
				$this->redirect(array('action'=>'view', $character_picture['Character']['id']));
			}

			$isOwner = $this->isOwner($character_picture['CharacterPicture'], $this->user_id);
		} else {
			$isOwner = false;
		}
		$this->set('isAdmin', $isAdmin);
		$this->set('isOwner', $isOwner);

		$this->set('title_for_layout', " - ". sprintf(__("%s's Picture", true), $character_picture['Character']['name']));
		if (isset($character_picture['Attachment'][0]['alternative']) &&!empty($character_picture['Attachment'][0]['alternative'])) {
			$this->set('title_for_layout', " ". $character_picture['Attachment'][0]['alternative']);
		}

		return $character_picture;
	}

	function _delete($character_picture)
	{

		$this->data['CharacterPicture']['id'] = $character_picture['CharacterPicture']['id'];
		if ($this->CharacterPicture->saveAll($this->data, array('validate' => 'first')) && $this->CharacterPicture->delete($character_picture['CharacterPicture']['id'])) {
			// characters設定の削除
			$this->_delete_setting($character_picture['Attachment'][0]['basename']);

			$this->Session->setFlash(__('CharacterPicture deleted', true));
		} else {
			$this->Session->setFlash(__('The data has not been deleted', true));
		}
		$this->redirect(array('action'=>'listview', $character_picture['CharacterPicture']['character_id']));
	}

	function _get_character4character_picture_id($character_picture_id, $user_id = null, $public_flag = null)
	{
		$conditions = array(
			'CharacterPicture.id' => $character_picture_id,
		);
		if ($user_id) {
			$conditions['CharacterPicture.user_id'] = $user_id;
		}
		if ($public_flag) {
			$conditions['CharacterPicture.public_flag'] = $public_flag;
		}
		$characterPicture = $this->CharacterPicture->find('first', array(
			'conditions' => $conditions,
			'fields' => array(
				'CharacterPicture.id',
				'CharacterPicture.user_id',
				'CharacterPicture.character_id',
				'CharacterPicture.public_flag',
				'CharacterPicture.created',
			),
			'contain' => array(
				'Character',
				'Attachment',
			),
			'recursive' => 1,
		));

		return $characterPicture;
	}

	function _set_image_config($character_pictures, $character)
	{
		if (!empty($character) && !empty($character_pictures)) {
			foreach ($character_pictures as $k => $v) {
				if (!empty($character['main_picture']) && $v['Attachment'][0]['basename'] == $character['main_picture'] && $v['CharacterPicture']['public_flag'] == 'public') {
					$character_pictures[$k]['CharacterPicture']['main_picture'] = true;
				} else {
					$character_pictures[$k]['CharacterPicture']['main_picture'] = false;
				}

				if (!empty($character['full_length']) && $v['Attachment'][0]['basename'] == $character['full_length'] && $v['CharacterPicture']['public_flag'] == 'public') {
					$character_pictures[$k]['CharacterPicture']['full_length'] = true;
				} else {
					$character_pictures[$k]['CharacterPicture']['full_length'] = false;
				}
			}
		}

		return $character_pictures;
	}

	function _delete_setting($basename)
	{
		$this->CharacterPicture->Character->updateAll(
			array('Character.main_picture' => null),
			array('Character.main_picture' => $basename)
		);

		$this->CharacterPicture->Character->updateAll(
			array('Character.full_length' => null),
			array('Character.full_length' => $basename)
		);
	}

}
