/**
 * \file framework.h
 * \brief フレームワーク・インクルードファイル
 * \details
 * 原則としてこのファイルの中の宣言をユーザーが変更する必要はない。例外はデバッグマクロのコメントアウト制御である。
 *
 * このファイルの中には複数のデバッグマクロが用意されており、これらのコメントアウトを説いて宣言状態にすると、DSP UARTのシリアルポートから
 * 内部状態を見ることができる。以下にマクロを列挙する。これらはフレームワークの開発中に使用したものをそのまま残している。
 *
 * ユーザーが使うとすれば、DEBUG_DSPLOADか、DEBUG_COMMANDだろう。
 *
 * \li DEBUG_QUEUEDEPTH シリアルポートから、タスク間Queueの深さを表示させる
 * \li DEBUG_DSPLOAD シリアルポートからDSP負荷率を表示させる
 * \li DEBUG_COMMAND DSPに送られてくるコマンドの表示
 * \li DEBUG_COMMAND_API 復調器から使えるAPIのデバッグ用
 * \li DEBUG_BLOCK_FLOAT WFMモードでブロック浮動小数点の整数化がうまく出来ているかダンプする。
 *
 */

/**
 * \mainpage
 * \section section_overview 概要
 * このドキュメントは、HiradoフレームワークのAPIについて解説する
 *
 * HiradoはSDR無線機TRX-305のDSP部の復調器フレームワークである。
 *
 * プログラマは \ref callbacks の関数群を埋める形で復調器を実装する。
 *
 * \section section_api API
 * コールバックの中からは \ref radioAPI にある無線APIを使って
 * 制御用のSHマイコン/FPGAと通信を行う。通信する内容は2種類ある。
 * \li 制御用マイコンからの制御命令
 * \li DSPからのステータス通知
 * \section section_debug デバッグについて
 * フレームワークにはデバッグ支援機能があり、デバッグマクロを宣言することで利用できる。
 * これらの機能はすべてDSPのuartを利用している。通信速度は115200baud, 8bit, parityなし、
 * 1stopbitである。
 *
 * フレームワークのデバッグ機能については \ref framework.h の解説を参照。
 *
 *
 */
#ifndef _FRAMEWORK_H_
#define _FRAMEWORK_H_

/*
 *  RX_IFアプリケーションのヘッダファイル
 */

#include <t_services.h>
#include "demodulator.h"

/**
 * \defgroup radioFramework フレームワーク内部宣言
 * \brief フレームワークの内部で使うオブジェクトの内部宣言。基本的にユーザーは参照しなくていい
 */
/*@{*/

/*
 * デバッグマクロ
 *
 * コメントアウトすると機能がディセーブルになる
*/
// #define DEBUG_QUEUEDEPTH    // シリアルポートから、タスク間Queueの深さを表示させる
// #define DEBUG_DSPLOAD       // シリアルポートからDSP負荷率を表示させる
// #define DEBUG_COMMAND           // DSPに送られてくるコマンドの表示
// #define DEBUG_COMMAND_API       // 復調器から使えるAPIのデバッグ用
// #define DEBUG_BLOCK_FLOAT       // WFMモードでブロック浮動小数点の整数化がうまく出来ているかダンプする。

/**
 * \brief RX_IF受信SPORTバッファのDMAサイズ
 * \details
 * RX_IF受信SPORTバッファのDMAサイズ。単位は[count]。バイト単位ではなく
 * 転送カウントが単位なので注意。
 * ワイドFMモードの場合、1サンプルが16カウントになる。
 * 非ワイドFMモードでは、1サンプルが4カウント。
 *
 * バッファサイズが32の場合、ワイドFMモードでは2サンプル(63uS)ごとに、
 * 非ワイドFMモードでは8サンプル(252uS)ごとに割り込みがかかる。
 *
 * なお、このバッファサイズは必ず4の倍数にすること。
 */
#define RXIF_BUFSIZE 32

/**
 * \brief AF送信SPORTバッファのDMAサイズ
 * \details
 * AF送信SPORTバッファのDMAサイズ。単位は[count]。バイト単位ではなく
 * 転送カウントが単位なので注意。
 * 1サンプルが2カウント。
 *
 * バッファサイズが8の場合、4サンプル(126uS)ごとに割り込みがかかる。
 *
 * なお、このバッファサイズは必ず2の倍数にすること。
 */
#define AF_BUFSIZE 8


/**
 * \brief TASK_RX_IF と TASK_AF の間のデータキューサイズ
 * \details
 * 2つのタスクの間のキューのサイズ[エントリ]を指定する。両タスクの
 * 実行の間にはWideFM/nonWideFM切り替えと、割り込み周期の差異に起因する
 * ジッターが生じするため、キューが必要になる。
 *
 * キューのサイズを決める要素は2つある。ひとつはキュー受信側
 * (TASK_AF)がデータを読み取る周期、もうひとつはキュー送信側
 * (TASK_RX_IF)がデータを読み取る周期である。いずれも単位はサンプルで、
 * このうち大きなほうがデータキューのサイズを決める。
 *
 * キューはオーバーフローとアンダーフローを避けるように作らなければならない。
 * アンダーフローはキューの受信側の割り込みが先行した場合に発生する。
 * これを避けるには、受信側が先行した場合に備えてダミーデータを
 * キューに入れておく必要がある。この数は、先で論じた周期の一番大きい物に
 * しておく。
 *
 * オーバーフローはキューの送信側の割り込みが先行した場合に発生する。
 * これを避けるには、送信側が先行した場合に備えて空き領域を
 * 確保する必要がある。この数は先で論じた周期の一番大きい物に
 * しておく。
 *
 * 結論として、キューには、最も長い割り込み周期[単位はサンプル]の倍の
 * 領域が必要である。
 *
 * なお、このソフトウェアは1サンプルあたり2[エントリ]をキューに格納するため、
 * これを勘案すると、最低でもキューの長さ[エントリ]は、割り込み周期[サンプル]の四倍になる。
 *
 *
 */
#define AF_QUESIZE          80      /* 送受信タスクの間のキューの長さ。8サンプルが最長割り込み周期と仮定してさらにマージンをもたせている */

	/** \brief AF1サンプルあたりのWide FM サンプル数 */
#define WIDE_FM_OVERSAMPE   8

	/** \brief プライマリ・チャンネルのWIDE FM Modeを表すフラグのマスク */
#define WIDE_FM_FLAG        2

	/** \brief SHマイコン/FPGAから送られてくるコマンドの種類の数 */
#define PARAMDATA_NUM 16


/*
 *  各タスクの優先度の定義
 */

	/** breif AF DMAタスクの優先度。送信なので優先度高 */
#define TASK_PRIORITY_AF       5
	/** breif RX-IF DMA処理タスクの優先度。復調も行う */
#define TASK_PRIORITY_RX_IF   10
	/** breif デフォルトの初期優先度を上書きする */
#define LOGTASK_PRIORITY      12
	/** breif モニター・タスクの優先度。優先度低 */
#define TASK_PRIORITY_MONITOR 15

/*
 *  ターゲット依存の定義（CPU例外ハンドラの起動方法など）
 */

	/** breif タスクのスタックサイズ */
#define STACK_SIZE_RX_IF    1024
	/** breif タスクのスタックサイズ */
#define STACK_SIZE_AF       1024
	/** breif タスクのスタックサイズ */
#define STACK_SIZE_MONITOR  1024
	/** breif 文字入力するシリアルポートID */
#define TASK_PORTID         1



/*
 *  関数のプロトタイプ宣言
 */
#ifndef _MACRO_ONLY

/**
 * \brief フレームワークがラジオとのやりとりに使う変数の型。
 * \details
 * フレームワークはFPGAから送られてきたデータをこの変数の中に格納する。復調アルゴリズムは \ref getInfo を通してこれらの値にアクセスする。
 */

    // ラジオアルゴリズムとのやりとりに使う変数
extern struct RADIO {
    unsigned short flags;	/**< DSPからFPGA/SHマイコンへ送るステータスフラグ */
    unsigned short smeter;	/**< DSPからFPGA/SHマイコンへ送るSメータ値 */
    unsigned short comdata[PARAMDATA_NUM];	/**< DSPへSHマイコン/FPGAから送るコマンド群 */
} radio;

/**
 * \brief フレームワークがラジオとのやりとりに使う変数。
 */
extern struct RADIO radio;

	/**
	 * \brief SPORT0 RX DMAバッファの内容を復調するタスク
	 * \param exinf タスクに渡す拡張データ。このタスクには何も渡さなくていい
	 * \details
	 * このタスクは \ref sport0_rx_int_handler() 割り込みハンドラからの
	 * セマフォ・シグナルを受け取って受信処理を行う。
	 */

extern void rx_if_task(VP_INT exinf);

	/**
	 * \brief 復調済みAF信号をSPORT0 TX DMAバッファにコピーするタスク
	 * \param exinf タスクに渡す拡張データ。このタスクには何も渡さなくていい
	 * \details
	 * このタスクは \ref sport0_tx_int_handler() 割り込みハンドラからの
	 * セマフォ・シグナルを受け取り、タスク間キューからデータを取り出しSPORT処理を行う。
	 */
extern void af_task(VP_INT exinf);

	/**
	 * \brief デバッグ用のデータ収集を行うバックグラウン・タスク
	 * \param exinf タスクに渡す拡張データ。このタスクには何も渡さなくていい
	 */
extern void monitor_task(VP_INT exinf);

	/**
	 * \brief SPORT0 RX DMA割り込みのハンドラ
	 */
extern void sport0_rx_int_handler( void );

	/**
	 * \brief SPORT0 TX DMA割り込みのハンドラ
	 */
extern void sport0_tx_int_handler( void );

	/**
	 * \brief ペリフェラルの初期化関数
	 * \param p タスクに渡す拡張データ。このタスクには何も渡さなくていい
	 */
void init_peripherals(VP_INT p);

/*@}*/

#endif /* _MACRO_ONLY */

#endif /* _FRAMEWORK_H_ */
