/*
 * FileOperation class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.util.List;
import java.util.LinkedList;
import ts.util.file.FileAlreadyExistsException;
import ts.util.file.DirectoryNotEmptyException;

/**
 * t@CNXB
 * <br>
 * t@C̈ꗗARs[A폜Aړs邽߂̃\bhpӂB
 *
 * @author  V. 
 * @version $Revision: 1.2 $, $Date: 2007/02/15 15:28:04 $
 */
public final class FileOperation
{
  /**
   * ftHgRXgN^B
   */
  protected FileOperation()
  {}

  /**
   * w肳ꂽfBNg̃t@C̈ꗗ擾B
   * <br>
   * w肳ꂽfBNg̏ꍇ́ÃXgԂB
   * fBNgł͂Ȃꍇ́Ai[XgԂB
   * ɃG[ꍇ͗OX[B
   *
   * @param  baseDir x[XfBNgB
   * @return x[XfBNgɂt@Ci[XgB
   * @throws FileNotFoundException w肳ꂽfBNg̓t@C
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static List<File> list(File baseDir)
    throws FileNotFoundException, IOException
  {
    assert (baseDir != null) : "@param:baseDir is null.";

    if (! baseDir.exists()) {
      throw new FileNotFoundException(baseDir.getPath());
    }

    List<File> lst = new LinkedList<File>();

    File[] ret = baseDir.listFiles();
    if (ret == null) {
      if (! baseDir.isDirectory()) {
        lst.add(baseDir);
        return lst;
      }
      else {
        throw new IOException(baseDir.getPath());
      }
    }

    for (int i=0; i<ret.length; i++) {
      lst.add(ret[i]);
    }

    return lst;
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@C̈ꗗ擾B
   * <br>
   * w肳ꂽfBNgɃfBNg΁ẢɊi[Ă
   * t@CꗗɊ܂߂ĕԂB
   * w肳ꂽfBNg̏ꍇ́A̔zԂB
   * fBNgłȂꍇ́Ai[XgԂB
   * ɃG[ꍇ͗OX[B
   *
   * @param  baseDir x[XfBNgB
   * @return x[XIuWFNgɂt@C{@link java.io.File File}
   *           IuWFNg̃XgB
   * @throws FileNotFoundException w肳ꂽfBNg̓t@C
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static List<File> listRecursive(File baseDir)
    throws FileNotFoundException, IOException
  {
    assert (baseDir != null) : "@param:baseDir is null.";

    if (! baseDir.exists()) {
      throw new FileNotFoundException(baseDir.getPath());
    }

    List<File> lst = new LinkedList<File>();
    if (! baseDir.isDirectory()) {
      lst.add(baseDir);
      return lst;
    }

    return listInner(baseDir, lst);
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@Ci[Xg擾B
   * <br>
   * w肳ꂽfBNg̏ꍇ́ÃXgԂB
   * ɃG[ꍇ͗OX[B
   *
   * @param  dir fBNgB
   * @param  fileLst t@C̈ꗗi[郊XgB
   * @return t@C̈ꗗi[XgB<code>fileLst</code>Ɠ
   *           łB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXȂꍇB
   * @throws AssertionError k̏ꍇA
   *           <code>dir</code>݂ȂꍇA
   *           <code>dir</code>fBNgłȂꍇ
   *           ifobO[ĥ)B
   */
  private static List<File> listInner(File dir, List<File> fileLst)
    throws IOException
  {
    assert (dir != null) : "@param:dir is null.";
    assert (fileLst != null) : "@param:fileLst is null.";
    assert (dir.exists()) : "@param:dir does not exist.";
    assert (dir.isDirectory()) : "@param:dir is not a directory.";
    
    File[] files = dir.listFiles();
    if (files == null) {
      throw new IOException(dir.getPath());
    }

    for (int i=0; i<files.length; i++) {
      fileLst.add(files[i]);
      if (files[i].isDirectory()) {
        listInner(files[i], fileLst);
      }
    }
    return fileLst;
  }

  /**
   * w肳ꂽt@C폜B
   * <br>
   * w肳ꂽt@CfBNg̏ꍇÃfBNgȂ
   * 폜͐B
   *
   * @param  file 폜Ώۂ̃t@CB
   * @throws FileNotFoundException w肳ꂽt@CȂꍇB
   * @throws DirectoryNotEmptyException w肳ꂽt@C̃fBNg
   *           ȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static void delete(File file)
    throws FileNotFoundException, DirectoryNotEmptyException, IOException
  {
    assert (file != null) : "@parm:file is null.";

    if (! file.delete()) {
      if (! file.exists()) {
        throw new FileNotFoundException(file.getPath());
      }

      if (file.isDirectory()) {
        if (file.list().length > 0)
          throw new DirectoryNotEmptyException(file.getPath());
      }

      throw new IOException(file.getPath());
    }
  }

  /**
   * w肳ꂽt@C폜B
   * <br>
   * w肳ꂽt@CfBNg̏ꍇÃfBNgȉ̑SĂ
   * t@C폜B
   *
   * @param  file 폜Ώۂ̃t@CB
   * @throws FileNotFoundException w肳ꂽt@CȂꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static void deleteRecursive(File file)
    throws FileNotFoundException, IOException
  {
    assert (file != null) : "@parm:file is null.";

    if (! file.exists()) {
      throw new FileNotFoundException(file.getPath());
    }

    if (file.isDirectory()) {
      File tmpDir = createTempDirectory("__del", "", file.getParentFile());
      if (file.renameTo(new File(tmpDir, file.getName()))) {
        deleteInner(tmpDir);
      }
      else {
        deleteInner(tmpDir);
        throw new IOException(file.getPath());
      }
    }
    else {
      if (! file.delete()) {
        throw new IOException(file.getPath());
      }
    }
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@C폜B
   * <br>
   * ɃG[ꍇ͗OX[B
   *
   * @param  dir fBNgB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZX͍폜ANZXȂꍇB
   * @throws AssertionError k̏ꍇA
   *           <code>dir</code>fBNgłȂꍇ
   *          ifobO[ĥ)B
   */
  private static void deleteInner(File dir) throws IOException
  {
    assert (dir != null) : "@param:dir is null.";
    assert (dir.exists()) : "@param:dir does not exist.";
    assert (dir.isDirectory()) : "@param:dir is not a directory.";

    File[] files = dir.listFiles();
    if (files == null) {
      throw new IOException(dir.getPath());
    }

    for (int i=0; i<files.length; i++) {
      if (files[i].isDirectory()) {
        deleteInner(files[i]);
      }
      else {
        if (! files[i].delete()) {
          throw new IOException(dir.getPath());
        }
      }
    }

    if (! dir.delete()) {
      throw new IOException(dir.getPath());
    }
  }

  /**
   * P̃t@CQ̃t@CɃRs[B
   * <br>
   * P̃t@C݂Ȃꍇ͗OX[B
   * Q̃t@Cɑ݂ꍇ͗OX[B
   * P̃t@CfBNg̏ꍇÃfBNgłȂꍇ
   * OX[B
   *
   * @param  src Rs[̃t@CB
   * @param  dst Rs[̃t@CB
   * @throws FileNotFoundException Rs[̃t@C݂ȂꍇB
   * @throws DirectoryNotEmptyException Rs[̃fBNgłȂ
   *           ꍇB
   * @throws FileAlreadyExistsException Rs[̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA݊mFANZXAfBNgAt@C
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static void copy(File src, File dst)
    throws FileNotFoundException, DirectoryNotEmptyException,
           FileAlreadyExistsException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    if (src.isDirectory()) {
      if (dst.exists()) {
        throw new FileAlreadyExistsException(dst.getPath());
      }
      if (src.list().length > 0) {
        throw new DirectoryNotEmptyException(src.getPath());
      }
      dst.mkdirs();
      return;
    }

    if (dst.exists()) {
      throw new FileAlreadyExistsException(dst.getPath());
    }
    
    copyFile(src, dst);
  }

  /**
   * t@C̎̂Rs[B
   *
   * @param  src Rs[̃\[XB
   * @param  dst Rs[̃\[XB
   * @throws FileNotFoundException t@C݂ȂAʂ̃t@Cł
   *          ȂfBNgł邩A܂͉炩̗RŊJƂłȂ
   *          ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA̓t@CȂꍇB
   */
  private static void copyFile(File src, File dst)
    throws FileNotFoundException, IOException
  {
    FileInputStream fis = null;
    FileOutputStream fos = null;
    try {
      fis = new FileInputStream(src);
      fos = new FileOutputStream(dst);

      byte[] bytes = new byte[1024];
      while (fis.read(bytes) >= 0) {
        fos.write(bytes);
      }
    }
    finally {
      if (fos != null) {
        try {
          fos.close();
        }
        catch (Exception e) {}
      }
      if (fis != null) {
        try {
          fis.close();
        }
        catch (Exception e) {}
      }
    }
  }

  /**
   * P̃t@CQ̃t@CɃRs[B
   * <br>
   * P̃t@C݂Ȃꍇ͗OX[B
   * Q̃t@Cɑ݂ꍇ͗OX[B
   * P̃t@CfBNg̏ꍇÃfBNgȉ̃t@C
   * SăRs[B
   *
   * @param  src Rs[̃t@CB
   * @param  dst Rs[̃t@CB
   * @throws FileNotFoundException Rs[̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsException Rs[̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA݊mFANZXAfBNgAt@C
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static void copyRecursive(File src, File dst)
    throws FileNotFoundException, FileAlreadyExistsException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    if (dst.exists()) {
      throw new FileAlreadyExistsException(dst.getPath());
    }

    if (src.isDirectory()) {
      copyInner(src, dst);
    }
    else {
      copyFile(src, dst);
    }
  }

  /**
   * w肳ꂽfBNgȉ̑SẴt@CRs[B
   * <br>
   * ɃG[ꍇ͗OX[B
   *
   * @param  src Rs[̃fBNgB
   * @param  dst Rs[̃fBNgB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA݊mFANZXAfBNgAt@C
   *           ȂꍇB
   * @throws AssertionError k̏ꍇA
   *           <code>dir</code>fBNgłȂꍇ
   *           ifobO[ĥ)B
   */
  private static void copyInner(File src, File dst) throws IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";
    assert (src.exists()) : "@param:src does not exists.";
    assert (src.isDirectory()) : "@param:src is not a directory.";
    assert (! dst.exists()) : "@param:dst already exists.";

    dst.mkdirs();

    File[] files = src.listFiles();
    if (files == null) {
      throw new IOException(src.getPath());
    }

    for (int i=0; i<files.length; i++) {
      File dstChild = new File(dst, files[i].getName());
      if (files[i].isDirectory()) {
        copyInner(files[i], dstChild);
      }
      else {
        copyFile(files[i], dstChild);
      }
    }
  }

  /**
   * P̃t@CQ̃t@CɈړB
   * <br>
   * P̃t@C݂Ȃꍇ͗OX[B
   * Q̃t@Cɑ݂ꍇ͗OX[B
   * P̃t@CfBNg̏ꍇÃfBNgłȂꍇ
   * OX[B
   *
   * @param  src ړ̃t@CB
   * @param  dst ړ̃t@CB
   * @throws FileNotFoundException ړ̃t@C݂ȂꍇB
   * @throws DirectoryNotEmptyException ړ̃fBNgłȂꍇB
   * @throws FileAlreadyExistsExcdeption ړ̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA̓t@CȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static void move(File src, File dst)
    throws FileAlreadyExistsException, DirectoryNotEmptyException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    if (src.isDirectory()) {
      if (dst.exists()) {
        throw new FileAlreadyExistsException(dst.getPath());
      }
      if (src.list().length > 0) {
        throw new DirectoryNotEmptyException(src.getPath());
      }
      src.renameTo(dst);
      return;
    }

    if (dst.exists()) {
      throw new FileAlreadyExistsException(dst.getPath());
    }

    src.renameTo(dst);
  }

  /**
   * P̃t@CQ̃t@CɈړB
   * <br>
   * P̃t@C݂Ȃꍇ͗OX[B
   * Q̃t@Cɑ݂ꍇ͗OX[B
   * P̃t@CfBNg̏ꍇÃfBNgȉ̃t@C
   * SĈړB
   *
   * @param  src ړ̃t@CB
   * @param  dst ړ̃t@CB
   * @throws FileNotFoundException ړ̃t@C݂ȂꍇB
   * @throws FileAlreadyExistsExcdeption ړ̃t@Cɑ݂Ă
   *           ꍇB
   * @throws IOException ɓo̓G[ꍇB
   * @throws SecurityException ZLeB}l[Wɂt@Cւ̓Ǎ
   *           ANZXA̓t@CȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   */
  public static void moveRecursive(File src, File dst)
    throws FileAlreadyExistsException, DirectoryNotEmptyException, IOException
  {
    assert (src != null) : "@param:src is null.";
    assert (dst != null) : "@param:dst is null.";

    if (dst.exists()) {
      throw new FileAlreadyExistsException(dst.getPath());
    }

    src.renameTo(dst);
  }

  /**
   * w肳ꂽfBNg̉ɋ̃t@Cj[NȖOŐB
   * <br>
   * t@Cɂ́Aw肳ꂽړyѐڔgpB
   * ړɂ3ȏオKvłB
   * ڔɂ́Ak܂ޔCӂ̕񂪎w\łAkw肳ꂽꍇ
   * <tt>".tmp"</tt>gpB
   * <br>
   * fBNgɃkw肳ꂽꍇ́A
   * VXeˑ̈ꎞt@CfBNggpB
   * <br>
   * ̃\bh͓{@link java.io.File#createTempFile(String,String,File)
   * createTempFile(String,String,File)}\bhĂяoĂB
   *
   * @param  prefix t@CɎgpړB
   * @param  suffix t@CɎgpڔB
   * @param  dir t@CfBNgB
   * @return VKɐꂽt@C{@link java.io.File File}IuWFNgB
   * @throws IllegalArgumentException ړꂲ3ɖȂꍇB
   * @throws IOException t@CłȂꍇB
   * @throws SecurityException ZLeB}l[Wɂt@C̐
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   * @see java.io.File#createTempFile(String, String, File)
   */
  public static File createTempFile(String prefix, String suffix, File dir)
    throws IllegalArgumentException, IOException
  {
    assert (prefix != null) : "@param:prefix is null.";
    assert (suffix != null) : "@param:prefix is null.";
    assert (dir != null) : "@param:dir is null.";

    return File.createTempFile(prefix, suffix, dir);
  }

  /**
   * w肳ꂽfBNg̉ɋ̃fBNgj[NȖOŐB
   * <br>
   * t@Cɂ́Aw肳ꂽړyѐڔgpB
   * ړɂ3ȏオKvłB
   * ڔɂ́Ak܂ޔCӂ̕񂪎w\łAkw肳ꂽꍇ
   * <tt>".tmp"</tt>gpB
   * <br>
   * fBNgɃkw肳ꂽꍇ́A
   * VXeˑ̈ꎞt@CfBNggpB
   * <br>
   * ̃\bh͓{@link java.io.File#createTempFile(String,String,File)
   * createTempFile(String,String,File)}\bhĂяoāA
   * j[NȃfBNg肵ĂB
   *
   * @param  prefix t@CɎgpړB
   * @param  suffix t@CɎgpڔB
   * @param  dir t@CfBNgB
   * @return VKɐꂽfBNg{@link java.io.File
   *           File}IuWFNgB
   * @throws IllegalArgumentException ړꂲ3ɖȂꍇB
   * @throws IOException fBNgłȂꍇB
   * @throws SecurityException ZLeB}l[WɂfBNg̐
   *           ȂꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ)B
   * @see java.io.File#createTempFile(String, String, File)
   */
  public static File createTempDirectory(String prefix, String suffix, File dir)
    throws IllegalArgumentException, IOException
  {
    assert (prefix != null) : "@param:prefix is null.";
    assert (suffix != null) : "@param:prefix is null.";
    assert (dir != null) : "@param:dir is null.";

    File file = File.createTempFile(prefix, suffix, dir);
    file.delete();
    file.mkdirs();
    return file;
  }
}

