/*
 * FileOperationTest class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import ts.tester.function.coverage.FunctionTester;
import ts.tester.function.ObjectInspector;
import ts.tester.function.print.*;
import java.io.*;
import java.util.*;
import java.security.*;

/**
 * <code>FileOperation</code>NX̋@\NXB
 *
 * @author  V. 
 * @version $Revision: 1.2 $, $Date: 2007/02/15 15:28:04 $
 */
public class FileOperationTest extends FunctionTester
{
  public static void main(String[] args)
  {
    try {
      PrinterGroup group = new PrinterGroup();
      group.addPrinter(new ConsolePrinter());
      group.addPrinter(new HtmlPrinter("SATOH Takayuki"));
      setPrinter(group);

      run(FileOperationTest.class, (args.length == 0) ? null : args[0]);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /* -- test case -- */

  public void constructor()
  {
    NOTNULL(new FileOperation());
  }

  public void list()
  {
    MSG("ʏ̑̏ꍇ");
    try {
      List<File> childLst = FileOperation.list(new File("."));
      for (int i=0; i<childLst.size(); i++) {
        OK(childLst.get(i).getName());
      }
    } catch (Exception e) {
      NG(e);
    }
  }

  public void list_01()
  {
    MSG("k̏ꍇ");
    try {
      FileOperation.list(null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void list_02()
  {
    MSG("̃fBNg̏ꍇ");

    String dirname = "testestest";
    File emptydir = new File(dirname);
    emptydir.mkdir();
    try {
      List<File> fileLst = FileOperation.list(new File(dirname));
      EQUAL(fileLst.size(), 0);
    } catch (Exception e) {
      NG(e);
    } finally {
      emptydir.delete();
    }
  }

  public void list_03()
  {
    MSG("łȂfBNg̏ꍇ");
    File dir = null, child1 = null, child2 = null;

    String dirname = "testtestest2";
    dir = new File(dirname);
    dir.mkdir();
    try {
      child1 = new File(dir, "child.txt");
      child1.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> fileLst = FileOperation.list(new File(dirname));
      EQUAL(fileLst.size(), 1);
      EQUAL(fileLst.get(0).getName(), child1.getName());
    } catch (Exception e) {
      NG(e);
    }

    try {
      child2 = new File(dir, "child2.xxx");
      child2.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> fileLst = FileOperation.list(new File(dirname));
      EQUAL(fileLst.size(), 2);
      EQUAL(fileLst.get(0).getName(), child1.getName());
      EQUAL(fileLst.get(1).getName(), child2.getName());
    } catch (Exception e) {
      NG(e);
    }

    child1.delete();
    child2.delete();
    dir.delete();
  }

  public void list_04()
  {
    MSG("ʏ̃t@C̏ꍇ");

    File aFile = new File("aFile.txt");
    try {
      aFile.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> fileLst = FileOperation.list(new File("aFile.txt"));
      EQUAL(fileLst.size(), 1);
      EQUAL(fileLst.get(0).getName(), aFile.getName());
    } catch (Exception e) {
      NG(e);
    }

    aFile.delete();
  }

  public void list_05()
  {
    MSG("݂ȂfBNg^t@C̏ꍇ");

    try {
      List<File> fileLst = FileOperation.list(new File("bbb"));
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> fileLst = FileOperation.list(new File("aaa.txt"));
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void list_06()
  {
    MSG("fBNgɂւ炸ʃt@C̈ꗗ擾łȂꍇ");

    File dir = new File("test/list") {
      public File[] listFiles() {
        return null;
      }
    };
    try {
      dir.mkdir();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.list(dir);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    dir.delete();
  }

  public void listRecursive()
  {
    MSG("ʏ̑̏ꍇ");
    File dir0 = new File("test/listRecursive");
    File dir1 = new File(dir0, "ttt");
    File dir2 = new File(dir0, "ddd");
    File dir3 = new File(dir2, "xxx");
    File file01 = new File(dir0, "test.txt");
    File file02 = new File(dir0, "aaa.bak");
    File file11 = new File(dir1, "bbb.ccc");
    File file12 = new File(dir1, "ddd.e");
    File file23 = new File(dir2, "f.g");
    try {
      dir0.mkdir();
      dir1.mkdir();
      dir2.mkdir();
      dir3.mkdir();
      file01.createNewFile();
      file02.createNewFile();
      file11.createNewFile();
      file12.createNewFile();
      file23.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> lst =
        FileOperation.listRecursive(new File("test/listRecursive"));
      EQUAL(lst.size(), 8);
      EQUAL(lst.get(0).getPath(), "test\\listRecursive\\aaa.bak");
      EQUAL(lst.get(1).getPath(), "test\\listRecursive\\ddd");
      EQUAL(lst.get(2).getPath(), "test\\listRecursive\\ddd\\f.g");
      EQUAL(lst.get(3).getPath(), "test\\listRecursive\\ddd\\xxx");
      EQUAL(lst.get(4).getPath(), "test\\listRecursive\\test.txt");
      EQUAL(lst.get(5).getPath(), "test\\listRecursive\\ttt");
      EQUAL(lst.get(6).getPath(), "test\\listRecursive\\ttt\\bbb.ccc");
      EQUAL(lst.get(7).getPath(), "test\\listRecursive\\ttt\\ddd.e");
    } catch (Exception e) {
      NG(e);
    }

    try {
      file23.delete();
      file12.delete();
      file11.delete();
      file02.delete();
      file01.delete();
      dir3.delete();
      dir2.delete();
      dir1.delete();
      dir0.delete();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void listRecursive_1()
  {
    MSG("k̏ꍇ");
    try {
      FileOperation.listRecursive(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void listRecursive_2()
  {
    MSG("fBNgłȂꍇ");
    File file0 = new File("test/listRecursive.txt");
    try {
      file0.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      List<File> lst = FileOperation.listRecursive(file0);
      EQUAL(lst.size(), 1);
      EQUAL(lst.get(0).getPath(), "test\\listRecursive.txt");
    } catch (Exception e) {
      NG(e);
    }

    try {
      file0.delete();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void listRecursive_3()
  {
    MSG("݂Ȃt@Cw肵ꍇ");
    File dir = new File("test/llll");

    try {
      FileOperation.listRecursive(dir);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void listInner_0()
  {
    MSG("fBNgɂւ炸ʃt@C̈ꗗ擾łȂꍇ");

    File dir = new File("test/list") {
      public File[] listFiles() {
        return null;
      }
    };
    try {
      dir.mkdir();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.listRecursive(dir);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    dir.delete();
  }

  public void delete()
  {
    MSG("ʏ̃t@C폜ꍇ");

    File aFile = new File("aaa.txt");
    try {
      aFile.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.delete(new File("aaa.txt"));
      FALSE(aFile.exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void delete_01()
  {
    MSG("̃fBNg폜ꍇ");

    File emptyDir = new File("ddd");
    try {
      emptyDir.mkdir();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.delete(new File("ddd"));
      FALSE(emptyDir.exists());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void delete_02()
  {
    MSG("łȂfBNg폜悤Ƃꍇ");

    File dir = new File("ddd");
    File child1 = new File(dir, "a1");
    File child2 = new File(dir, "a2");
    try {
      dir.mkdir();
      child1.createNewFile();
      child2.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.delete(new File("ddd"));
      NG();
    } catch (DirectoryNotEmptyException e) {
      EQUAL(e.getPath(), "ddd");
      NULL(e.getMessage());
    } catch (Exception e) {
      NG(e);
    }

    child1.delete();
    child2.delete();
    dir.delete();
  }

  public void delete_03()
  {
    MSG("݂Ȃt@C폜悤Ƃꍇ");

    File aFile = new File("aaa");
    try {
      FileOperation.delete(aFile);
      NG();
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (IOException e) {
      NG(e);
    }
  }

  public void delete_04()
  {
    MSG("k̏ꍇ");

    try {
      FileOperation.delete(null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }
  }

  public void delete_05()
  {
    MSG("t@C̍폜Ɏsꍇ");

    File file = new File("test/FileOperation.txt");
    FileReader fr = null;
    try {
      file.createNewFile();
      fr = new FileReader(file);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.delete(file);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      if (fr != null) fr.close();
      file.delete();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void deleteRecursive()
  {
    MSG("ʏ̑̏ꍇ");

    File dir0 = new File("test", "FileOperation");
    File dir01 = new File(dir0, "dir01");
    File dir02 = new File(dir0, "dir02");
    File dir011 = new File(dir01, "dir011");
    File file01 = new File(dir0, "file01.txt");
    File file02 = new File(dir0, "file02.txt");
    File file03 = new File(dir0, "file03.txt");
    File file011 = new File(dir01, "file011.txt");
    File file012 = new File(dir01, "file012.txt");
    File file013 = new File(dir01, "file013.txt");
    File file021 = new File(dir02, "file021.txt");
    File file022 = new File(dir02, "file022.txt");
    File file0111 = new File(dir011, "file0111.txt");
    File file0112 = new File(dir011, "file0112.txt");

    try {
      dir0.mkdirs();
      dir01.mkdirs();
      dir02.mkdirs();
      dir011.mkdirs();
      file01.createNewFile();
      file02.createNewFile();
      file03.createNewFile();
      file011.createNewFile();
      file012.createNewFile();
      file013.createNewFile();
      file021.createNewFile();
      file022.createNewFile();
      file0111.createNewFile();
      file0112.createNewFile();
    }
    catch (Exception e) {
      NG(e);
    }

    TRUE(dir0.exists());
    TRUE(dir01.exists());
    TRUE(dir02.exists());
    TRUE(dir011.exists());
    TRUE(file01.exists());
    TRUE(file02.exists());
    TRUE(file03.exists());
    TRUE(file011.exists());
    TRUE(file012.exists());
    TRUE(file013.exists());
    TRUE(file021.exists());
    TRUE(file022.exists());
    TRUE(file0111.exists());
    TRUE(file0112.exists());

    try {
      FileOperation.deleteRecursive(dir0);
    } catch (Exception e) {
      e.printStackTrace();
      NG(e);
    }

    FALSE(dir0.exists());
    FALSE(dir01.exists());
    FALSE(dir02.exists());
    FALSE(dir011.exists());
    FALSE(file01.exists());
    FALSE(file02.exists());
    FALSE(file03.exists());
    FALSE(file011.exists());
    FALSE(file012.exists());
    FALSE(file013.exists());
    FALSE(file021.exists());
    FALSE(file022.exists());
    FALSE(file0111.exists());
    FALSE(file0112.exists());
  }

  public void deleteRecursive_1()
  {
    MSG("k̏ꍇ");

    try {
      FileOperation.deleteRecursive(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void deleteRecursive_2()
  {
    MSG("w肳ꂽt@C݂Ȃꍇ");

    try {
      FileOperation.deleteRecursive(new File("test/FileOperation/xxx"));
    } catch (FileNotFoundException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void deleteRecursive_3()
  {
    MSG("w肳ꂽt@CfBNgł͂Ȃꍇ");

    File dir = new File("test/FileOperation");
    File file = new File(dir, "notDirectory.txt");
    try {
      dir.mkdirs();
      file.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    TRUE(dir.exists());
    TRUE(file.exists());

    try {
      FileOperation.deleteRecursive(file);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(dir.exists());
    FALSE(file.exists());

    dir.delete();
  }

  public void deleteRecursive_4()
  {
    MSG("w肳ꂽfBNg̃t@Cgp̏ꍇ");

    File dir0 = new File("test/FileOperation");
    File dir1 = new File(dir0, "d1");
    File dir2 = new File(dir0, "d2");
    File file11 = new File(dir1, "aaa.txt");
    File file12 = new File(dir1, "bbb.txt");
    File file13 = new File(dir1, "ccc.txt");
    File file21 = new File(dir2, "ddd.txt");
    File file22 = new File(dir2, "eee.txt");

    try {
      dir0.mkdirs();
      dir1.mkdirs();
      dir2.mkdirs();
      file11.createNewFile();
      file12.createNewFile();
      file13.createNewFile();
      file21.createNewFile();
      file22.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir0.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(file11.exists());
    TRUE(file12.exists());
    TRUE(file13.exists());
    TRUE(file21.exists());
    TRUE(file22.exists());

    FileReader fr = null;
    try {
      fr = new FileReader(file21);

      FileOperation.deleteRecursive(dir0);
      NG();
    } catch (Exception e) {
      OK(e);
    } finally {
      if (fr != null) try { fr.close(); } catch (Exception e) {}
    }

    TRUE(dir0.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(file11.exists());
    TRUE(file12.exists());
    TRUE(file13.exists());
    TRUE(file21.exists());
    TRUE(file22.exists());

    try {
      FileOperation.deleteRecursive(dir0);
    } catch (Exception e) {
      NG(e);
    }

    FALSE(dir0.exists());
    FALSE(dir1.exists());
    FALSE(dir2.exists());
    FALSE(file11.exists());
    FALSE(file12.exists());
    FALSE(file13.exists());
    FALSE(file21.exists());
    FALSE(file22.exists());
  }

  public void deleteRecursive_5()
  {
    MSG("w肳ꂽt@Cgp̏ꍇ");

    File dir = new File("test/FileOperation");
    File file = new File(dir, "abcdefg.xxx");

    try {
      dir.mkdirs();
      file.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir.exists());
    TRUE(file.exists());

    FileReader fr = null;
    try {
      fr = new FileReader(file);

      FileOperation.deleteRecursive(file);
      NG();
    } catch (Exception e) {
      OK(e);
    } finally {
      if (fr != null) try { fr.close(); } catch (Exception e) {}
    }

    TRUE(dir.exists());
    TRUE(file.exists());

    file.delete();
    dir.delete();
  }

  public void deleteInner_0()
  {
    MSG("폜悤ƂfBNg̃t@C̈ꗗ̎擾Ɏsꍇ");

    File dir = new File("test/FileOperation") {
      public File[] listFiles() {
        return null;
      }
    };

    try {
      dir.mkdirs();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir.exists());

    try {
      ObjectInspector oi = new ObjectInspector(this);
      FileOperation fo = new FileOperation();
      EQUAL(oi.invokeMethod(fo, "deleteInner", File.class, dir), Void.TYPE);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    dir.delete();
  }

  public void deleteInner_1()
  {
    MSG("fBNg̃t@C̍폜Ɏsꍇ");

    File dir = new File("test/FileOperation");
    File file = new File(dir, "xyz.txt");

    try {
      dir.mkdirs();
      file.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir.exists());
    TRUE(file.exists());

    FileReader fr = null;
    try {
      fr = new FileReader(file);

      ObjectInspector oi = new ObjectInspector(this);
      FileOperation fo = new FileOperation();
      EQUAL(oi.invokeMethod(fo, "deleteInner", File.class, dir), Void.TYPE);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    } finally {
      if (fr != null) try { fr.close(); } catch (Exception e) {}
    }

    file.delete();
    dir.delete();
  }

  public void deleteInner_2()
  {
    MSG("fBNg̍폜Ɏsꍇ");

    File dir = new File("test/FileOperation") {
      public boolean delete() {
        super.delete();
        return false;
      }
    };
    File file = new File(dir, "zyx.txt");

    try {
      dir.mkdirs();
      file.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir.exists());
    TRUE(file.exists());

    try {
      ObjectInspector oi = new ObjectInspector(this);
      FileOperation fo = new FileOperation();
      EQUAL(oi.invokeMethod(fo, "deleteInner", File.class, dir), Void.TYPE);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    file.delete();
    dir.delete();
  }

  public void copy()
  {
    MSG("ʏ̃t@C𓯂fBNgɕʖŃRs[ꍇ");

    File dir = new File("aaa");
    File src = new File(dir, "sss");
    File dst = new File(dir, "ddd");

    FALSE(dir.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir.mkdir();
      src.createNewFile();
      TRUE(dir.exists());
      TRUE(src.exists());
      FALSE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(src, dst);
      TRUE(dir.exists());
      TRUE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    src.delete();
    dst.delete();
    dir.delete();
  }

  public void copy_000()
  {
    MSG("ʏ̃t@C𓯂fBNgɕʖŃRs[ꍇ" +
        "iłȂt@Cj");

    File dir = new File("aaa");
    File src = new File(dir, "sss");
    File dst = new File(dir, "ddd");

    FALSE(dir.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir.mkdir();
      PrintWriter pw = new PrintWriter(new FileWriter(src));
      pw.println("");
      pw.flush();
      pw.close();
      TRUE(dir.exists());
      TRUE(src.exists());
      FALSE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(src, dst);
      TRUE(dir.exists());
      TRUE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    src.delete();
    dst.delete();
    dir.delete();
  }

  public void copy_01()
  {
    MSG("ʏ̃t@Cʂ̃fBNgɃRs[ꍇ");

    File dir1 = new File("ddd");
    File src1 = new File(dir1, "aaa");
    File dir2 = new File("eee");
    File dst2 = new File(dir2, "aaa");

    FALSE(dir1.exists());
    FALSE(dir2.exists());
    FALSE(src1.exists());
    FALSE(dst2.exists());

    try {
      dir1.mkdir();
      dir2.mkdir();
      src1.createNewFile();
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      TRUE(src1.exists());
      FALSE(dst2.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(src1, dst2);
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      TRUE(src1.exists());
      TRUE(dst2.exists());
    } catch (Exception e) {
      NG(e);
    }

    src1.delete();
    dir1.delete();
    dst2.delete();
    dir2.delete();
  }

  public void copy_02()
  {
    MSG("̃fBNg𓯂fBNgɕʖŃRs[ꍇ");

    File dir1 = new File("ddd");
    File child1 = new File(dir1, "eee");
    File child2 = new File(dir1, "fff");
    FALSE(dir1.exists());
    FALSE(child1.exists());
    FALSE(child2.exists());

    try {
      dir1.mkdir();
      child1.mkdir();
      TRUE(dir1.exists());
      TRUE(child1.exists());
      FALSE(child2.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(child1, child2);
      TRUE(dir1.exists());
      TRUE(child1.exists());
      TRUE(child2.exists());
    } catch (Exception e) {
      NG(e);
    }

    child1.delete();
    child2.delete();
    dir1.delete();
  }

  public void copy_03()
  {
    MSG("̃fBNgʂ̃fBNgɃRs[ꍇ");

    File dir1 = new File("ddd");
    File dir2 = new File("eee");
    File child1 = new File(dir1, "fff");
    File child2 = new File(dir2, "fff");

    FALSE(dir1.exists());
    FALSE(dir2.exists());
    FALSE(child1.exists());
    FALSE(child2.exists());

    try {
      dir1.mkdir();
      dir2.mkdir();
      child1.mkdir();
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      TRUE(child1.exists());
      FALSE(child2.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(child1, child2);
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      TRUE(child1.exists());
      TRUE(child2.exists());
    } catch (Exception e) {
      NG(e);
    }

    child1.delete();
    child2.delete();
    dir1.delete();
    dir2.delete();
  }

  public void copy_04()
  {
    MSG("łȂfBNgRs[悤Ƃꍇ");
    File dir1 = new File("ddd");
    File child1 = new File(dir1, "fff");
    File child2 = new File(dir1, "ggg");
    File aFile = new File(child1, "hhh.txt");

    try {
      dir1.mkdir();
      child1.mkdir();
      aFile.createNewFile();
      TRUE(dir1.exists());
      TRUE(child1.exists());
      FALSE(child2.exists());
      TRUE(aFile.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(child1, child2);
      NG();
    } catch (DirectoryNotEmptyException e) {
      EQUAL(e.getPath(), child1.getPath());
      NULL(e.getMessage());
    } catch (Exception e) {
      NG(e);
    }

    aFile.delete();
    child1.delete();
    dir1.delete();
  }

  public void copy_05()
  {
    MSG("Rs[̃t@Cɑ݂ꍇ");

    File dir = new File("ddd");
    File src = new File(dir, "sss");
    File dst = new File(dir, "ttt");

    try {
      dir.mkdir();
      src.createNewFile();
      dst.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      EQUAL(e.getPath(), dst.getPath());
      NULL(e.getMessage());
    } catch (Exception e) {
      NG(e);
    }

    src.delete();
    dst.delete();
    dir.delete();
  }

  public void copy_05a()
  {
    MSG("Rs[̃t@CifBNgjɑ݂ꍇ");

    File dir = new File("ddd");
    File src = new File(dir, "sss");
    File dst = new File(dir, "ttt");

    try {
      dir.mkdir();
      src.mkdir();
      dst.mkdir();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      EQUAL(e.getPath(), dst.getPath());
      NULL(e.getMessage());
    } catch (Exception e) {
      NG(e);
    }

    src.delete();
    dst.delete();
    dir.delete();
  }

  public void copy_06()
  {
    MSG("k̏ꍇ");
    try {
      FileOperation.copy(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(null, new File("aaa"));
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(new File("aaa"), null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    File dir = new File("ddd");
    dir.mkdir();

    try {
      FileOperation.copy(dir, null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    File child = new File(dir, "ccc");
    try {
      child.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copy(dir, null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    child.delete();
    dir.delete();
  }

  public void copyRecursive()
  {
    MSG("ʏ̑");

    File dir0 = new File("test/FileOperation/src");
    File dir1 = new File(dir0, "d01");
    File dir2 = new File(dir0, "d02");
    File dir3 = new File(dir0, "d03");
    File file0 = new File(dir0, "f00.txt");
    File file1 = new File(dir1, "f01.txt");
    File file2 = new File(dir2, "f02.txt");
    File file3 = new File(dir2, "f03.txt");

    try {
      dir0.mkdirs();
      dir1.mkdirs();
      dir2.mkdirs();
      dir3.mkdirs();
      file0.createNewFile();
      file1.createNewFile();
      file2.createNewFile();
      file3.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir0.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(dir3.exists());
    TRUE(file0.exists());
    TRUE(file1.exists());
    TRUE(file2.exists());
    TRUE(file3.exists());

    File destDir = new File("test/FileOperation/dst");
    FALSE(destDir.exists());

    try {
      FileOperation.copyRecursive(dir0, destDir);
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir0.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(dir3.exists());
    TRUE(file0.exists());
    TRUE(file1.exists());
    TRUE(file2.exists());
    TRUE(file3.exists());

    File destDir1 = new File(destDir, dir1.getName());
    File destDir2 = new File(destDir, dir2.getName());
    File destDir3 = new File(destDir, dir3.getName());
    File destFile0 = new File(destDir, file0.getName());
    File destFile1 = new File(destDir1, file1.getName()); 
    File destFile2 = new File(destDir2, file2.getName()); 
    File destFile3 = new File(destDir2, file3.getName()); 

    TRUE(destDir.exists());
    TRUE(destDir1.exists());
    TRUE(destDir2.exists());
    TRUE(destDir3.exists());
    TRUE(destFile0.exists());
    TRUE(destFile1.exists());
    TRUE(destFile2.exists());
    TRUE(destFile3.exists());

    try {
      FileOperation.deleteRecursive(new File("test/FileOperation"));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copyRecursive_1()
  {
    MSG("k̏ꍇ");

    try {
      FileOperation.copyRecursive(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copyRecursive(null, new File("aaa"));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copyRecursive(new File("test"), null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copyRecursive_2()
  {
    MSG("Rs[̃t@Cɑ݂ꍇ");

    File src = new File("test/FileOperation/src");
    File file0 = new File(src, "aaa.txt");

    File dst = new File("test/FileOperation/dst");
    File file1 = new File(dst, "bbb.txt");

    try {
      src.mkdirs();
      dst.mkdir();
      file0.createNewFile();
      file1.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(src.exists());
    TRUE(dst.exists());
    TRUE(file0.exists());
    TRUE(file1.exists());

    try {
      FileOperation.copyRecursive(src, dst);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.deleteRecursive(new File("test/FileOperation"));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copyRecursive_3()
  {
    MSG("Rs[fBNgŁẢ̃t@Cꗗ擾łȂꍇ");

    File dir = new File("test/FileOperation");
    File src = new File(dir, "src") {
      public File[] listFiles() { return null; }
    };
    File dst = new File(dir, "dst");

    try {
      src.mkdirs();
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.copyRecursive(src, dst);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.deleteRecursive(new File("test/FileOperation"));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void copyFile_1()
  {
    MSG("Rs[fBNgł͂Ȃꍇ");
    
    File dir = new File("test/FileOperation");
    File src = new File(dir, "src.txt");
    File dst = new File(dir, "dst.txt");

    try {
      dir.mkdirs();
      src.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(src.exists());
    FALSE(dst.exists());

    try {
      FileOperation.copyRecursive(src, dst);
      TRUE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    dst.delete();
    src.delete();
    dir.delete();
  }

  public void move()
  {
    MSG("ʏ̑̏ꍇ");

    File dir = new File("ddd");
    File src = new File(dir, "aaa");
    File dst = new File(dir, "bbb");

    FALSE(dir.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir.mkdir();
      src.createNewFile();
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir.exists());
    TRUE(src.exists());
    FALSE(dst.exists());

    try {
      FileOperation.move(src, dst);
    } catch (Exception e) {
      NG(e);
    }
    TRUE(dir.exists());
    FALSE(src.exists());
    TRUE(dst.exists());
    
    src.delete();
    dst.delete();
    dir.delete();
  }

  public void move_01()
  {
    MSG("k̏ꍇ");

    try {
      FileOperation.move(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(null, new File("aaa"));
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(new File("aaa"), null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    File dir = new File("ddd");
    try {
      dir.mkdir();
      FileOperation.move(dir, null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    File child = new File(dir, "aaa");
    try {
      child.createNewFile();
      FileOperation.move(dir, null);
      NG();
    } catch (AssertionError e) {
      OK(e.toString());
    } catch (Exception e) {
      NG(e);
    }

    child.delete();
    dir.delete();
  }

  public void move_02()
  {
    MSG("ʏ̃t@C𓯂fBNgɕʖňړꍇB");

    File dir = new File("aaa");
    File src = new File(dir, "sss");
    File dst = new File(dir, "ddd");

    FALSE(dir.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir.mkdir();
      src.createNewFile();
      TRUE(dir.exists());
      TRUE(src.exists());
      FALSE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(src, dst);
      TRUE(dir.exists());
      FALSE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    dst.delete();
    dir.delete();
  }

  public void move_03()
  {
    MSG("ʏ̃t@Cʂ̃fBNgɃRs[ꍇ");

    File dir1 = new File("ddd");
    File dir2 = new File("eee");
    File src = new File(dir1, "aaa");
    File dst = new File(dir2, "aaa");

    FALSE(dir1.exists());
    FALSE(dir2.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir1.mkdir();
      dir2.mkdir();
      src.createNewFile();
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      TRUE(src.exists());
      FALSE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(src, dst);
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      FALSE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    dst.delete();
    dir1.delete();
    dir2.delete();
  }

  public void move_04()
  {
    MSG("̃fBNg𓯂fBNgɕʖňړꍇ");

    File dir = new File("ddd");
    File src = new File(dir, "fff");
    File dst = new File(dir, "ggg");
    FALSE(dir.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir.mkdir();
      src.mkdir();
      TRUE(dir.exists());
      TRUE(src.exists());
      FALSE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(src, dst);
      TRUE(dir.exists());
      FALSE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    dst.delete();
    dir.delete();
  }

  public void move_05()
  {
    MSG("̃fBNgʂ̃fBNg艺ɃRs[ꍇ");
    File dir1 = new File("ddd");
    File dir2 = new File("eee");
    File src = new File(dir1, "fff");
    File dst = new File(dir2, "fff");

    FALSE(dir1.exists());
    FALSE(dir2.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir1.mkdir();
      dir2.mkdir();
      src.mkdir();
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      TRUE(src.exists());
      FALSE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(src, dst);
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      FALSE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    dst.delete();
    dir1.delete();
    dir2.delete();
  }

  public void move_06()
  {
    MSG("łȂfBNgړ悤Ƃꍇ");
    File dir1 = new File("ddd");
    File child1 = new File(dir1, "fff");
    File child2 = new File(dir1, "ggg");
    File aFile = new File(child1, "hhh.txt");

    try {
      dir1.mkdir();
      child1.mkdir();
      aFile.createNewFile();
      TRUE(dir1.exists());
      TRUE(child1.exists());
      FALSE(child2.exists());
      TRUE(aFile.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(child1, child2);
      NG();
    } catch (DirectoryNotEmptyException e) {
      NULL(e.getMessage());
    } catch (Exception e) {
      NG(e);
    }

    aFile.delete();
    child1.delete();
    dir1.delete();
  }

  public void move_07()
  {
    MSG("ړ̃t@Cɑ݂ꍇ");

    File dir = new File("ddd");
    File src = new File(dir, "sss");
    File dst = new File(dir, "ttt");

    try {
      dir.mkdir();
      src.createNewFile();
      dst.createNewFile();
      TRUE(dir.exists());
      TRUE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      EQUAL(e.getPath(), dst.getPath());
      NULL(e.getMessage());
    } catch (Exception e) {
      NG(e);
    }
    
    src.delete();
    dst.delete();
    dir.delete();
  }

  public void move_08()
  {
    MSG("fBNg̃Rs[ɂāARs[̃fBNg݂ꍇ");
    File dir1 = new File("d1");
    File dir2 = new File("d2");
    File src  = new File(dir1, "sss");
    File dst  = new File(dir2, "ttt");

    FALSE(dir1.exists());
    FALSE(dir2.exists());
    FALSE(src.exists());
    FALSE(dst.exists());

    try {
      dir1.mkdir();
      dir2.mkdir();
      src.mkdir();
      dst.mkdir();
      TRUE(dir1.exists());
      TRUE(dir2.exists());
      TRUE(src.exists());
      TRUE(dst.exists());
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.move(src, dst);
      NG();
    } catch (FileAlreadyExistsException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    src.delete();
    dst.delete();
    dir1.delete();
    dir2.delete();
  }

  public void moveRecursive()
  {
    MSG("ʏ̑");

    File dir0 = new File("test/FileOperation/dir0");
    File dir1 = new File(dir0, "dir1");
    File dir2 = new File(dir0, "dir2");
    File dir3 = new File(dir1, "dir3");
    File dir4 = new File(dir1, "dir4");
    File file0 = new File(dir0, "file0.txt");
    File file1 = new File(dir1, "file1.txt");
    File file2 = new File(dir2, "file2.txt");
    File file3 = new File(dir3, "file3.txt");
    File file4 = new File(dir4, "file4.txt");

    try {
      dir0.mkdirs();
      dir1.mkdirs();
      dir2.mkdirs();
      dir3.mkdirs();
      dir4.mkdirs();
      file0.createNewFile();
      file1.createNewFile();
      file2.createNewFile();
      file3.createNewFile();
      file4.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    TRUE(dir0.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(dir3.exists());
    TRUE(dir4.exists());
    TRUE(file0.exists());
    TRUE(file1.exists());
    TRUE(file2.exists());
    TRUE(file3.exists());
    TRUE(file4.exists());

    File dest = new File("test/FileOperation/dest");
    try {
      FileOperation.moveRecursive(dir0, dest);
    } catch (Exception e) {
      NG(e);
    }

    FALSE(dir0.exists());
    FALSE(dir1.exists());
    FALSE(dir2.exists());
    FALSE(dir3.exists());
    FALSE(dir4.exists());

    TRUE(dest.exists());

    dir1 = new File(dest, dir1.getName());
    dir2 = new File(dest, dir2.getName());
    dir3 = new File(dir1, "dir3");
    dir4 = new File(dir1, "dir4");
    file0 = new File(dest, "file0.txt");
    file1 = new File(dir1, "file1.txt");
    file2 = new File(dir2, "file2.txt");
    file3 = new File(dir3, "file3.txt");
    file4 = new File(dir4, "file4.txt");

    TRUE(dest.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(dir3.exists());
    TRUE(dir4.exists());
    TRUE(file0.exists());
    TRUE(file1.exists());
    TRUE(file2.exists());
    TRUE(file3.exists());
    TRUE(file4.exists());

    try {
      FileOperation.deleteRecursive(new File("test/FileOperation"));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveRecursive_1()
  {
    MSG("k̏ꍇ");

    try {
      FileOperation.moveRecursive(null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      FileOperation.moveRecursive(new File("test"), null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
    try {
      FileOperation.moveRecursive(null, new File("test/aaa"));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveRecursive_2()
  {
    MSG("ړ̃t@Cɑ݂ꍇ");

    File dir0 = new File("test/FileOperation/dir0");
    File dir1 = new File(dir0, "dir1");
    File dir2 = new File(dir0, "dir2");
    File dir3 = new File(dir1, "dir3");
    File dir4 = new File(dir1, "dir4");
    File file0 = new File(dir0, "file0.txt");
    File file1 = new File(dir1, "file1.txt");
    File file2 = new File(dir2, "file2.txt");
    File file3 = new File(dir3, "file3.txt");
    File file4 = new File(dir4, "file4.txt");

    try {
      dir0.mkdirs();
      dir1.mkdirs();
      dir2.mkdirs();
      dir3.mkdirs();
      dir4.mkdirs();
      file0.createNewFile();
      file1.createNewFile();
      file2.createNewFile();
      file3.createNewFile();
      file4.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    TRUE(dir0.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(dir3.exists());
    TRUE(dir4.exists());
    TRUE(file0.exists());
    TRUE(file1.exists());
    TRUE(file2.exists());
    TRUE(file3.exists());
    TRUE(file4.exists());

    File dest = new File("test/FileOperation/dest");
    dest.mkdirs();
    TRUE(dest.exists());
    try {
      FileOperation.moveRecursive(dir0, dest);
      NG();
    } catch (IOException e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    TRUE(dir0.exists());
    TRUE(dir1.exists());
    TRUE(dir2.exists());
    TRUE(dir3.exists());
    TRUE(dir4.exists());
    TRUE(file0.exists());
    TRUE(file1.exists());
    TRUE(file2.exists());
    TRUE(file3.exists());
    TRUE(file4.exists());

    TRUE(dest.exists());

    try {
      FileOperation.deleteRecursive(new File("test/FileOperation"));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void moveRecursive_3()
  {
    MSG("fBNgȊȌꍇ");

    File dir0 = new File("test/FileOperation/dir0");
    File file0 = new File(dir0, "file0.txt");

    try {
      dir0.mkdirs();
      file0.createNewFile();
    } catch (Exception e) {
      NG(e);
    }

    TRUE(dir0.exists());
    TRUE(file0.exists());

    File dest = new File("test/FileOperation/dest_file");
    try {
      FileOperation.moveRecursive(file0, dest);
    } catch (Exception e) {
      NG(e);
    }

    FALSE(file0.exists());
    TRUE(dest.exists());
    file0 = new File(dest, "file0.txt");

    try {
      FileOperation.deleteRecursive(new File("test/FileOperation"));
    } catch (Exception e) {
      NG(e);
    }
  }

  public void createTempFile()
  {
    MSG("ʏ̑");
    try {
      File f = FileOperation.createTempFile("xxx", ".yyy", new File("test"));
      TRUE(f.getName().startsWith("xxx"));
      TRUE(f.getName().endsWith(".yyy"));
      TRUE(f.exists());
      FALSE(f.isDirectory());
      MSG("temporary file name = " + f.getPath());
      f.delete();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void createTempFile_1()
  {
    MSG("k̏ꍇ");

    try {
      FileOperation.createTempFile(null, null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.createTempFile(null, ".yyy", new File("test"));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.createTempFile("xxx", null, new File("test"));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.createTempFile("xxx", ".yyy", null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }

  public void createTempDirectory()
  {
    MSG("ʏ̑");

    try {
      File f = FileOperation.createTempDirectory("xxx", "y", new File("test"));
      TRUE(f.getName().startsWith("xxx"));
      TRUE(f.getName().endsWith("y"));
      TRUE(f.exists());
      TRUE(f.isDirectory());
      MSG("temporary directory name = " + f.getPath());
      f.delete();
    } catch (Exception e) {
      NG(e);
    }
  }

  public void createTempDirectory_1()
  {
    MSG("k̏ꍇ");

    try {
      FileOperation.createTempDirectory(null, null, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.createTempDirectory(null, "yyy", new File("test"));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.createTempDirectory("xxx", null, new File("test"));
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }

    try {
      FileOperation.createTempDirectory("xxx", "yyy", null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    } catch (Exception e) {
      NG(e);
    }
  }
}

