/*
 * FilePathPatternFilter class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.file;

import ts.util.text.StringCutter;
import ts.util.text.StringOperation;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.LinkedList;

/**
 * t@CEpXEp^[Ƃt@CEtB^ENXB
 *
 * @author  V.
 * @version $Revision: 1.2 $, $Date: 2007/05/27 16:13:11 $
 */
public class FilePathPatternFilter implements FileFilter
{
  /** x[XEfBNgEpXB */
  private String basePath_ ;

  /**
   * p^[dAX^XNApX؂蕶AAX^XNŕ
   * vfi[郊XgB
   */
  private List<List<List<String>>> pathPatternLLL_ ;

  /**
   * pXEp^[ƃx[XEfBNgɂƂRXgN^B
   *
   * @param  baseDir x[XEfBNgB
   * @param  pathPattern pXEp^[B
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   * @throws IllegalArgumentException <tt>baseDir</tt>̎t@CfBN
   *           głȂꍇB̓pXEp^[񂪕sȏꍇB
   */
  public FilePathPatternFilter(File baseDir, String pathPattern)
    throws IllegalArgumentException
  {
    assert (baseDir != null) : "@param:baseDir is null.";

    if (! baseDir.isDirectory()) {
      throw new IllegalArgumentException("@param:baseDir is not directory.");
    }

    try {
      basePath_ = baseDir.getCanonicalPath();
      if (! StringOperation.endsWith(basePath_, File.separator)) {
        basePath_ += File.separator;
      }
    }
    catch (IOException e) {
      IllegalArgumentException exc = new IllegalArgumentException(e);
      exc.setStackTrace(e.getStackTrace());
      throw exc;
    }

    pathPatternLLL_ = parsePathPattern(pathPattern);
  }

  /**
   * pXEp^[͂āAf[^ƂĐݒ肷B
   *
   * @param  pathPattern pXEp^[B
   * @throws IllegalArgumentException@pXEp^[񂪕sȏꍇB
   */
  private List<List<List<String>>> parsePathPattern(String pathPattern)
    throws IllegalArgumentException
  {
    assert (pathPattern != null) : "@param:pathPattern is null.";

    if (StringOperation.startsWith(pathPattern, "/")) {
      throw new IllegalArgumentException("@param:pathPattern must not start " +
        "with '/', because it is relative path.");
    }

    List<String> patternL = StringCutter.split(pathPattern, "/");
    return splitWithAsterisk(patternL);
  }

  /** 
   * fBNgKwƂɕꂽpXEp^[̃XgA
   * dAX^XN̗vf؂ƂăO[sOB
   * ܂ApXEp^[񎩑̂AAX^XN؂蕶Ƃĕ
   * 񃊃XgɕϊB
   *
   * @param  patternL@fBNgKwƂɕꂽpXEp^[
   *           XgB
   * @return pXEp^[AX^XNŕXgB
   * @throws NullPointerException k̏ꍇB
   * @throws IllegalArgumentException dAX^XNt@Cɑ΂
   *           ChEJ[hƂĎgpĂꍇB
   */
  private List<List<List<String>>> splitWithAsterisk(List<String> patternL)
  {
    List<List<List<String>>> patternLLL = new LinkedList<List<List<String>>>();
    List<List<String>> patternLL = new LinkedList<List<String>>();
    patternLLL.add(patternLL);

    int bgn = 0;
    for (String pattern : patternL) {
      if (StringOperation.isEmpty(pattern)) {
        throw new IllegalArgumentException(
          "Specified path pattern includes empty path elements.");
      }
      else if (StringOperation.contains(pattern, "**")) {
        if (pattern.equals("**")) {
          patternLL = new LinkedList<List<String>>();
          patternLLL.add(patternLL);
        }
        else {
          throw new IllegalArgumentException(
            "Wild card '**' cannot be used for a file name.");
        }
      }
      else {
        patternLL.add(StringCutter.split(pattern, "*"));
      }
    }

    return patternLLL;
  }

  /**
   * w肳ꂽt@C̃pXÃIuWFNg̎pXEp^[ɍv
   * ǂ𔻒肷B
   *
   * @param  file Ώۂ̃t@CB
   * @return t@C̃pXpXEp^[ɍvꍇ<tt>true</tt>ԂB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  public boolean accept(File file)
  {
    assert (file != null) : "@param:file is null.";

    try {
      String absPath = file.getCanonicalPath();
      if (! StringOperation.startsWith(absPath, basePath_)) {
        return false;
      }
      int basePathLen = StringOperation.length(basePath_);
      String relPath = StringOperation.substring(absPath, basePathLen);
      relPath = relPath.replace(File.separator, "/");
      return matchPattern(relPath, pathPatternLLL_);
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * t@CEpXAp^[EXgɍv邩ǂ𔻒肷B
   *
   * @param  relativePath x[XEfBNg̑΃pXB
   * @param  patternLLL p^[𕪊XgB
   * @return vꍇ<tt>true</tt>ԂB
   */
  private boolean matchPattern(
    String relativePath, List<List<List<String>>> patternLLL)
  {
    List<String> pathL = StringCutter.split(relativePath, "/");

    Iterator<List<List<String>>> it = patternLLL.iterator();
    if (it.hasNext()) {
      List<List<String>> patternLL = it.next();
      if (! matchPatternElement(pathL, 0, patternLL)) {
        return false;
      }
      int pathIndex = patternLL.size();

      L1:while (it.hasNext()) {
        patternLL = it.next();
        if (patternLL.size() == 0) {
          continue L1;
        }

        while (pathIndex < pathL.size()) {
          if (matchPatternElement(pathL, pathIndex, patternLL)) {
            pathIndex += patternLL.size();
            continue L1;
          }
          else {
            pathIndex ++;
          }
        }

        return false;
      }

      if (pathIndex < pathL.size()) {
        if (patternLL.size() != 0) {
          return false;
        }
      }

      return true;
    }
    
    return false;
  }

  /**
   * fBNgKwƂɕꂽpXEXg̗vfƁAfBNgAAX
   * ^XNƂɕꂽp^[EXgv邩ǂ𔻒肷B
   *
   * @param  pathL fBNgKwƂɕꂽpXEXgB
   * @param  pathIndex pXEXg̃CfbNXB
   * @param  patternLL fBNgAAX^XNƂɕꂽp^[E
   *           XgB
   * @return vꍇ<tt>true</tt>ԂB
   */
  private boolean matchPatternElement(List<String> pathL, int pathIndex,
    List<List<String>> patternLL)
  {
    if (patternLL.size() + pathIndex > pathL.size()) {
      return false;
    }

    for (int i=0; i<patternLL.size(); i++) {
      String path = pathL.get(i + pathIndex);
      List<String> patternL = patternLL.get(i); 

      Iterator<String> patternIt = patternL.iterator();
      if (! patternIt.hasNext()) {
        return StringOperation.isEmpty(path) ? true : false;
      }

      int k = 0;
      String pattern = patternIt.next();
      if (! StringOperation.isEmpty(pattern)) {
        if (! StringOperation.startsWith(path, pattern)) {
          return false;
        }
        k = StringOperation.length(pattern);
      }
      else if (! patternIt.hasNext()) {
        return false;
      }

      while (patternIt.hasNext()) {
        pattern = patternIt.next();
        k = StringOperation.indexOf(path, pattern, k);
        if (k < 0) {
          return false;
        }
        k += StringOperation.length(pattern);
      }

      if (! StringOperation.isEmpty(pattern)) {
        if (k != StringOperation.length(path)) {
          return false;
        }
      }
    }

    return true;
  }
}
