/*
 * ArrayListTable class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.table;

import java.io.Serializable;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.AbstractCollection;
import java.util.AbstractSet;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Arrays;
import java.util.Collections;
import java.util.NoSuchElementException;
import ts.util.table.Record.SimpleEntry;

/**
 * z̃XgƂĎꂽe[uENXB
 * <br>
 * R[h\}bv͒l̏WzɊi[AJEL[̏W
 * {@link ts.util.table.Header Header}IuWFNgɂăR[hԂŋpB
 * <br>
 * {@link ts.util.table.Header Header}IuWFNǵAJEL[Ɣz
 * CfbNXΉt}bvɎA̔z̃CfbNX
 * JEL[ƔzɊi[ꂽlΉtB
 * <br>
 * R[hERNV̓XgŎĂ̂ŁAR[h̏ԂA
 * J̒lR[h𕡐i[邱ƂłB
 *
 * @param <C> JEL[̃^CvB
 * @param <V> J̒l̃^CvB
 *
 * @author  V.
 * @version $Revision: 1.2 $, $Date: 2007/10/09 17:04:51 $
 */
public class ArrayListTable<C,V> extends AbstractTable<C,V>
{
  /** VAEo[WԍB */
  static final long serialVersionUID = 1940969047348687437L;

  /**
   * ftHgRXgN^B
   */
  public ArrayListTable()
  {}

  /**
   * R[heʂƏJeʂɂƂRXgN^B
   *
   * @param  initRecCapacity R[heʁB
   * @param  initColCapacity JeʁB
   * @throws AssertionError ̒l̏ꍇifobOE[ĥ݁jB
   */
  public ArrayListTable(int initRecCapacity, int initColCapacity)
  {
    super(initRecCapacity, initColCapacity);
  }

  /**
   * wb_ɂƂRXgN^B
   *
   * @param  header wb_EIuWFNgB
   * @throws AssertionError k̏ꍇifobOE[ĥ݁jB
   */
  protected ArrayListTable(Header<C> header)
  {
    super(header);
  }

  /**
   * {@inheritDoc}
   *
   * @throws AssertionError ̒l̏ꍇifobOE[ĥ݁jB
   */
  @Override
  protected Header<C> createHeader(int initColCapacity)
  {
    return new ArrayListTableHeader(initColCapacity);
  }

  /**
   * {@inheritDoc}
   *
   * @throws AssertionError ̒l̏ꍇifobOE[ĥ݁jB
   */
  @Override
  protected Collection<Record<C,V>> createRecordCollection(int initRecCapacity)
  {
    assert (initRecCapacity >= 0) : "@param:initRecCapacity is negative.";
    return new LinkedList<Record<C,V>>();
  }

  /**
   * R[hEIuWFNg쐬B
   *
   * @param  initColCapacity JeʁB
   * @return R[hEIuWFNgB
   * @throws AssertionError ̒l̏ꍇifobOE[ĥ݁jB
   */
  @Override
  protected Record<C,V> createRecord(int initColCapacity)
  {
    assert (initColCapacity >= 0) : "@param:initColCapacity is negative.";

    return new ArrayListTableRecord(initColCapacity);
  }

  /* -- inner class -- */

  /**
   * {@link ts.util.table.ArrayListTable ArrayListTable}NX̃wb_
   * \NXB
   * <br>
   * JEL[ƔzCfbNX̑Ήts}bvێA
   * lzɊi[ {@link
   * ts.util.table.ArrayListTable.ArrayListTableRecord ArrayListTableRecord}
   * IuWFNg̋L̃L[WƂĎgpB
   */
  protected class ArrayListTableHeader implements Header<C>, Serializable
  {
    /** VAEo[WԍB */
    static final long serialVersionUID = -114778725908464738L;

    /** JƔzCfbNX̑gi[}bvB */
    private HashMap<C,Integer> columnMap_ ;

    /** ݎgpĂCfbNX̍őlB */
    private int maxArrayIndex_ = -1;

    /**
     * JeʂɂƂRXgN^B
     *
     * @param  initColCapacity JeʁB
     * @throws AssertionError ̒l̏ꍇifobOE[ĥ݁jB
     */
    public ArrayListTableHeader(int initColCapacity)
    {
      assert (initColCapacity >= 0) : "@param:initColCapacity is negative.";
      columnMap_ = new HashMap<C,Integer>(initColCapacity);
    }

    /**
     * ̃wb_Ɋi[ĂJEL[̐擾B
     *
     * @return JEL[̐B
     */
    public int columnCount()
    {
      return columnMap_.size();
    }
  
    /**
     * ̃wb_Ɋi[ĂJEL[Ɏ擾邽߂̃Ce[^
     * 擾B
     *
     * @return JEL[̃Ce[^B
     */
    public Enumeration<C> columns()
    {
      return Collections.enumeration(columnMap_.keySet());
    }

    /**
     * w肳ꂽIuWFNgƓJEL[̃wb_Ɋi[Ă邩
     * ǂmFB
     *
     * @param  column JEL[B
     * @return w肳ꂽJEL[̃IuWFNgɊi[Ăꍇ
     *         <tt>true</tt>ԂB
     */
    public boolean hasColumn(Object column)
    {
      return columnMap_.containsKey(column);
    }
  
    /**
     * wb_ɃJEL[ǉB
     *
     * @param  column JEL[B
     */
    public void addColumn(C column)
    {
      addArrayIndexForColumn(column);
    }

    /**
     * JEL[ɑΉtꂽCfbNX擾B
     * <br>
     * ̃CfbNX́Azŕ\郌R[hł́AeJ̒li[
     * ʒuw肷邽߂ɎgpB
     *
     * @param  column JEL[B
     * @return JEL[ɑΉtꂽCfbNXB
     *         w肳ꂽJEL[̃IuWFNgɊi[ĂȂꍇ
     *         -1ԂB
     */
    protected int getArrayIndexForColumn(Object column)
    {
      Integer v = columnMap_.get(column);
      return (v == null) ? -1 : v.intValue();
    }

    /**
     * wb_ɃJEL[ǉB
     * <br>
     * ̃JEL[ƃCfbNX̑gÃIuWFNgێ}bv
     * ɓo^B
     * CfbNXɂ́AݎgpĂCfbNX̍ől+1ݒ肳B
     * ߂ĺAݒ肵CfbNXԂB
     * <br>
     * ̃JEL[wb_ɊɊi[Ăꍇ́Aǉɐݒ
     * ĂCfbNXԂB
     *
     * @param  column JEL[B
     * @return JEL[̃CfbNXB
     */
    protected int addArrayIndexForColumn(C column)
    {
      Integer i = columnMap_.get(column);
      if (i != null) {
        return i.intValue();
      }

      maxArrayIndex_ ++;
      columnMap_.put(column, maxArrayIndex_);
      return maxArrayIndex_ ;
    }
  }

  /**
   * {@link ts.util.table.ArrayListTable ArrayListTable}NX̃R[h\
   * }bvENXB
   * <br>
   * JEL[ƔzCfbNX̑Ήts}bvێ
   * {@link ts.util.table.ArrayListTable.ArrayListTableHeader 
   * ArrayListTableHeader} IuWFNg̃}bṽL[WƂA
   * }bv̒l͔zɊi[B
   */
  protected class ArrayListTableRecord extends Record<C,V>
  {
    /** VAEo[WԍB */
    static final long serialVersionUID = 4663048964389775476L;

    /** eJ̒li[zB */
    private V[] values_ ;

    /**
     * JeʂɂƂRXgN^B
     *
     * @param  initColCapacity JeʁB
     * @throws AssertionError ̒l̏ꍇifobOE[ĥ݁jB
     */
    public ArrayListTableRecord(int initColCapacity)
    {
      assert (initColCapacity >= 0) : "@param:size is negative.";

      values_ = newValue(initColCapacity);
    }

    /**
     * R[h̊eJ̒li[邽߂̔z쐬B
     *
     * @param  initColCapacity JeʁB
     * @return R[h̊eJ̒li[邽߂̔zB
     */
    @SuppressWarnings("unchecked")
    private V[] newValue(int initColCapacity)
    {
      return (V[]) new Object[(initColCapacity < 0) ? 0 : initColCapacity];
    }

    /**
     * {@inheritDoc}
     */
    @Override
    protected Header<C> header()
    {
      return ArrayListTable.this.header();
    }

    /**
     * {@link ts.util.table.ArrayListTable.ArrayListTableHeader
     * ArrayListTableHeader}IuWFNg擾B
     *
     * @return {@link ts.util.table.ArrayListTable.ArrayListTableHeader
     *         ArrayListTableHeader}IuWFNgB
     */
    @SuppressWarnings("unchecked")
    private ArrayListTableHeader altHeader()
    {
      return (ArrayListTableHeader) header();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean containsValue(Object value)
    {
      if (value == null) {
        if (size() > values_.length) {
          return true;
        }

        for (int i=0; i<size(); i++) {
          if (values_[i] == null) {
            return true;
          }
        }
      }
      else {
        int n = Math.min(size(), values_.length);
        for (int i=0; i<n; i++) {
          if (value.equals(values_[i])) {
            return true;
          }
        }
      }

      return false;
    }
  
    /**
     * {@inheritDoc}
     */
    @Override
    protected V getValue(Object column)
    {
      try {
        return values_[ altHeader().getArrayIndexForColumn(column) ];
      }
      catch (Exception e) {
        return null;
      }
    }
  
    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    @Override
    protected V putValue(C column, V value)
    {
      int ind = altHeader().addArrayIndexForColumn(column);
      if (ind >= values_.length) {
        Object arr = new Object[ind + 10];
        System.arraycopy(values_, 0, arr, 0, values_.length);
        values_ = (V[]) arr;
      }
  
      V old = values_[ind];
      values_[ind] = value;
      return old;
    }
  
    /**
     * {@inheritDoc}
     */
    @Override
    protected V removeValue(Object column)
    {
      try {
        int ind = altHeader().getArrayIndexForColumn(column);
        V old = values_[ind];
        values_[ind] = null;
        return old;
      }
      catch (Exception e) {
        return null;
      }
    }
  
    /**
     * {@inheritDoc}
     */
    public void clear()
    {
      Arrays.fill(values_, null);
    }
    
    /**
     * {@inheritDoc}
     */
    public Collection<V> values()
    {
      return new AbstractCollection<V>() {
        public int size() {
          return header().columnCount();
        }
        public Iterator<V> iterator() {
          return new Iterator<V>() {
            private int index_ = -1;
  
            public boolean hasNext() {
              return (index_ + 1 < size());
            }
            public V next() {
              index_ ++;
              if (index_ >= size()) {
                throw new NoSuchElementException();
              }
              else if (index_ < values_.length) {
                return values_[index_];
              }
              else {
                return null;
              }
            }
            public void remove() {
              if (index_ < 0) {
                throw new IllegalStateException();
              }
              else if (index_ >= size()) {
                throw new NoSuchElementException();
              }
              else if (index_ < values_.length) {
                values_[index_] = null;
              }
            }
          };
        }
      };
    }
  
    /**
     * {@inheritDoc}
     */
    public Set<Map.Entry<C,V>> entrySet()
    {
      return new AbstractSet<Map.Entry<C,V>>() {
        public int size() {
          return header().columnCount();
        }
        public Iterator<Map.Entry<C,V>> iterator() {
          return new Iterator<Map.Entry<C,V>>() {
            private int index_ = -1;
            private Iterator<Map.Entry<C,Integer>> it =
              altHeader().columnMap_.entrySet().iterator();
    
            public boolean hasNext() {
              return it.hasNext();
            }
            public Map.Entry<C,V> next() {
              Map.Entry<C,Integer> ent = it.next();
              index_ = ent.getValue().intValue();
              if (0 <= index_ && index_ < values_.length) {
                return new SimpleEntry<C,V>(ent.getKey(), values_[index_]);
              }
              else {
                throw new NoSuchElementException();
              }
            }
            public void remove() {
              if (index_ < 0) {
                throw new IllegalStateException();
              }
              else if (index_ >= size()) {
                throw new NoSuchElementException();
              }
              else if (index_ < values_.length) {
                values_[index_] = null;
              }
            }
          };
        }
      };
    }
  }
}

