/*
 * GraphImage class.
 *
 * Copyright (C) 2010 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.graph;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.awt.Graphics2D;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.font.TextAttribute;
import java.awt.geom.AffineTransform;
import java.awt.geom.Line2D;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Arc2D;
import java.io.File;
import java.io.OutputStream;
import java.io.IOException;
import java.text.AttributedString;
import java.util.List;
import java.util.LinkedList;
import java.awt.FontMetrics;
import java.awt.Font;

/**
 * Ot摜쐬NXB
 * <br>
 * Ot`悵At@CXg[ɏo͂邽߂̒ۃNXłB
 * 
 * @author  V.
 * @version $Id: GraphImage.java,v 1.1 2010-10-16 13:47:00 tayu Exp $
 */
public abstract class GraphImage
{
  /** PNG`摜^CvlB */
  public final static String TYPE_PNG = "png";

  /** JPEG`摜^CvlB */
  public final static String TYPE_JPEG = "jpeg";

  /** ̃CfbNX萔B */
  protected final static int W = 0;

  /** ̃CfbNX萔B */
  protected final static int H = 1;

  /** ̃CfbNX萔B */
  protected final static int L = 0;

  /** ̃CfbNX萔B */
  protected final static int T = 1;

  /** ẼCfbNX萔B */
  protected final static int R = 2;

  /** ̃CfbNX萔B */
  protected final static int B = 3;

  /** ftHg̋EX^CB */
  private final static LineStyle DEFAULT_BORDER_STYLE = null;

  /** ftHg̐X^CB */
  private final static LineStyle DEFAULT_LINE_STYLE = new LineStyle(null);

  /** ftHg̓hԂX^CB*/
  private final static FillStyle DEFAULT_FILL_STYLE = new FillStyle(null);

  /** ftHg̃eLXgEX^CB */
  private final static TextStyle DEFAULT_TEXT_STYLE = new TextStyle();


  /** 摜̃TCYB*/
  private int[] imageSize_ = { 0, 0 };

  /** Ot`̈ɑ΂}[WB */
  private int[] margin_ = { 0, 0, 0, 0 };

  /** wiFB */
  private Color bgColor_ = Color.white;

  /** ẼX^CB */
  private LineStyle borderStyle_ = DEFAULT_BORDER_STYLE;

  /** ̃X^CB */
  private LineStyle lineStyle_ = DEFAULT_LINE_STYLE;

  /** ̃X^CύXꂽǂtOB */
  private boolean lineStyleChanged_ = true;

  /** hԂ̃X^CB */
  private FillStyle fillStyle_ = DEFAULT_FILL_STYLE;

  /** hԂ̃X^CύXꂽǂtOB */
  private boolean fillStyleChanged_ = true;

  /** eLXg̃X^CB */
  private TextStyle textStyle_ = DEFAULT_TEXT_STYLE;

  /** eLXg̃X^CύXꂽǂtOB */
  private boolean textStyleChanged_ = true;

  /** `ɎgpAtBϊIuWFNgB */
  private AffineTransform transformer_ = new AffineTransform();

  /** `ɎgpOtBbNXEIuWFNgB */
  private Graphics2D graphics_ = null;

  /**
   * ftHgERXgN^B
   */
  public GraphImage()
  {
    transformer_.setToTranslation(0, 0);
    transformer_.setToScale(1.0, 1.0);
  }

  /**
   * 摜̃TCYݒ肷B
   * <br>
   * ɕ̒lw肵ꍇ́A0ݒ肷B
   *
   * @param  w 摜̕B
   * @param  h 摜̍B
   */
  public void setSize(int w, int h)
  {
    imageSize_[W] = Math.max(w, 0);
    imageSize_[H] = Math.max(h, 0);
  }

  /**
   * 摜̕擾B
   *
   * @return 摜̕B
   */
  public int getWidth()
  {
    return imageSize_[W];
  }

  /**
   * 摜̍擾B
   *
   * @return 摜̍B
   */
  public int getHeight()
  {
    return imageSize_[H];
  }

  /**
   * Ot`̈ɑ΂}[Wݒ肷B
   * <br>
   * ɕ̒lw肵ꍇ́A0ݒ肷B
   *
   * @param  left   ̃}[WB
   * @param  top    㑤̃}[WB
   * @param  right  Ẽ}[WB
   * @param  bottom ̃}[WB
   */
  public void setMargin(int left, int top, int right, int bottom)
  {
    margin_[L] = Math.max(left, 0);
    margin_[T] = Math.max(top, 0);
    margin_[R] = Math.max(right, 0);
    margin_[B] = Math.max(bottom, 0);
  }

  /**
   * Ot`̈ɑ΂}[Wi[z擾B
   * <br>
   * @return }[Wi[zBPvf͍[xWAQvf
   *           [yWARvf͉E[xWASvf͉[yW
   *           B
   */
  public int[] getMargin()
  {
    return margin_;
  }

  /**
   * wiFݒ肷B
   * <br>
   * k̏ꍇ͖B
   *
   * @param  color wiFB
   */
  public void setBgColor(Color color)
  {
    if (color != null) {
      bgColor_ = color;
    }
  }

  /**
   * ẼX^Cݒ肷B
   * <br>
   * Ƀkw肵ꍇ́AE`悵ȂB
   *
   * @param style ẼX^CB
   */
  public void setBorderStyle(LineStyle style)
  {
    borderStyle_ = style;
  }

  /**
   * Ot̕`̈߂B
   *
   * @return Ot̕`̈{@link java.awt.Rectangle Rectangle}
   *           IuWFNgB
   */
  public Rectangle getDrawnBounds()
  {
    int x0 = margin_[L];
    int y0 = margin_[T];
    int x1 = imageSize_[W] - margin_[R];
    int y1 = imageSize_[H] - margin_[B];

    int w = Math.max(x1 - x0, 0);
    int h = Math.max(y1 - y0, 0);
    x0 = Math.min(x0, imageSize_[W]);
    y0 = Math.min(y0, imageSize_[H]);

    return new Rectangle(x0, y0, w, h);
  }

  /**
   * Ot`悵摜t@C쐬B
   *
   * @param  type 摜^CvB
   * @param  path t@CEpXB
   * @return 摜t@CB
   * @throws IOException t@C̏o͒ɗOꍇB
   */
  public File createImageFile(String type, String path) throws IOException
  {
    return createImageFile(type, new File(path));
  }

  /**
   * Ot`悵摜t@C쐬B
   *
   * @param  type 摜^CvB
   * @param  file 摜t@C{@link java.io.File File}IuWFNgB
   * @return 摜t@CB
   * @throws IOException t@C̏o͒ɗOꍇB
   */
  public File createImageFile(String type, File file) throws IOException
  {
    BufferedImage bi = new BufferedImage(
      getWidth(), getHeight(), BufferedImage.TYPE_3BYTE_BGR);

    graphics_ = bi.createGraphics();

    setAntiAliasing(true);
    setTextAntiAliasing(false);

    drawBackGround();

    Rectangle rc = getDrawnBounds();
    if (borderStyle_ != null) {
      drawBorder(rc);
    }
    drawGraph(rc);

    ImageIO.write(bi, type, file);

    return file;
  }

  /**
   * Ot`悵ďo̓Xg[ɏo͂B
   *
   * @param type 摜^CvB
   * @param ostream o̓Xg[B
   * @throws IOException Xg[ւ̏o͒ɗOꍇB
   */
  public void outputImage(String type, OutputStream ostream) throws IOException
  {
    BufferedImage bi = new BufferedImage(
      getWidth(), getHeight(), BufferedImage.TYPE_3BYTE_BGR);

    graphics_  = bi.createGraphics();

    drawBackGround();

    Rectangle rc = getDrawnBounds();
    if (borderStyle_ != null) {
      drawBorder(rc);
    }
    drawGraph(rc);

    ImageIO.write(bi, type, ostream);
  }

  /**
   * Ot`悷B
   *
   * @param  drawnBounds `̈B
   */
  protected abstract void drawGraph(Rectangle drawnBounds);

  /**
   * `ɎgpĂOtBbNXEIuWFNg擾B
   *
   * @return `ɎgpĂOtBbNXEIuWFNgB
   */
  protected Graphics2D graphics()
  {
    return graphics_ ;
  }

  /**
   * }`̃A`GCAVO̗LEݒ肷B
   *
   * @param  on A`GCAVOLɂꍇ<tt>true</tt>w肷B
   */
  protected void setAntiAliasing(boolean on)
  {
    Object onoff = on ?
      RenderingHints.VALUE_ANTIALIAS_ON :
      RenderingHints.VALUE_ANTIALIAS_OFF;

    graphics_.setRenderingHint(RenderingHints.KEY_ANTIALIASING, onoff);
  }

  /**
   * eLXg̃A`GCAVO̗LEݒ肷B
   *
   * @param  on A`GCAVOLɂꍇ<tt>true</tt>w肷B
   */
  protected void setTextAntiAliasing(boolean on)
  {
    Object onoff = on ?
      RenderingHints.VALUE_TEXT_ANTIALIAS_ON :
      RenderingHints.VALUE_TEXT_ANTIALIAS_OFF;

    graphics_.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, onoff);
  }

  /**
   * ̃X^Cݒ肷B
   *
   * @param  lineStyle ̃X^CB
   * @return ܂Őݒ肳ẴX^CB
   */
  protected LineStyle setLineStyle(LineStyle lineStyle)
  {
    LineStyle old = lineStyle_ ;
    if (lineStyle_ != lineStyle) {
      lineStyle_ = lineStyle;
      lineStyleChanged_ = true;
    }
    return old;
  }

  /**
   * ̃X^COtBbNXEIuWFNgɃ[hB
   */
  private void loadLineStyle()
  {
    if (lineStyleChanged_) {
      lineStyle_.loadStyle(graphics());
      lineStyleChanged_ = false;
      fillStyleChanged_ = true;
    }
  }

  /**
   * hԂ̃X^Cݒ肷B
   *
   * @param  fillStyle hԂ̃X^CB
   * @return ܂Őݒ肳ĂhԂX^CB
   */
  protected FillStyle setFillStyle(FillStyle fillStyle)
  {
    FillStyle old = fillStyle_ ;
    if (fillStyle_ != fillStyle) {
      fillStyle_ = fillStyle;
      fillStyleChanged_ = true;
    }
    return old;
  }

  /**
   * hԂ̃X^COtBbNXEIuWFNgɃ[hB
   */
  private void loadFillStyle()
  {
    if (fillStyleChanged_) {
      fillStyle_.loadStyle(graphics());
      fillStyleChanged_ = false;
      lineStyleChanged_ = true;
    }
  }

  /**
   * eLXg̃X^Cݒ肷B
   *
   * @param  textStyle eLXg̃X^Cݒ肷B
   * @return ܂Őݒ肳ĂeLXg̃X^CB
   */
  protected TextStyle setTextStyle(TextStyle textStyle)
  {
    TextStyle old = textStyle_ ;
    if (textStyle_ != textStyle) {
      textStyle_ = textStyle;
      textStyleChanged_ = true;
    }
    return old;
  }

  /**
   * eLXg̃X^COtBbNEIuWFNgɃ[hB
   */
  private void loadTextStyle()
  {
    if (textStyleChanged_) {
      textStyle_.loadStyle(graphics());
      textStyleChanged_ = false;
    }
  }

  /**
   * 摜ɑ΂`̈̌_ɐݒ肷B
   * <br>
   * _̍ẂAɉ摜̍ォ̈ʒuw肷B
   * ݂̌_̑Έʒuł͂ȂƂɒӂB
   *
   * @param  x 摜̐̍WB
   * @param  y 摜̐̍WB
   */
  protected void setOrigin(int x, int y)
  {
    double sx = transformer_.getScaleX();
    double sy = transformer_.getScaleY();
    transformer_.setTransform(sx, 0, 0, sy, x, y);
  }

  /**
   * 摜ɑ΂`̈̔{ݒ肷B
   * <br>
   * {́A摜Ŝ̃TCY}[Wɑ΂lłB
   *
   * @param  sx 摜ɑ΂鐅̔{B
   * @param  sy 摜ɑ΂鐂̔{B
   */
  protected void setScale(double sx, double sy)
  {
    double x = transformer_.getTranslateX();
    double y = transformer_.getTranslateY();
    transformer_.setTransform(sx, 0, 0, sy, x, y);
  }

  /**
   * wi`悷B
   */
  protected void drawBackGround()
  {
    graphics().setColor(bgColor_);
    graphics().fillRect(0, 0, getWidth(), getHeight());
  }

  /**
   * `̈̋E`悷B
   *
   * @param  rc `̈B
   */
  protected void drawBorder(Rectangle rc)
  {
    if (borderStyle_ != null) {
      graphics().setStroke(borderStyle_.getStroke());
      graphics().setColor(borderStyle_.getColor());
      graphics().drawRect(rc.x, rc.y, rc.width - 1, rc.height - 1);
    }
  }

  /**
   * `悷B
   *
   * @param  x0 ̎n_xWB
   * @param  y0 ̎n_yWB
   * @param  x1 ̏I_xWB
   * @param  y1 ̏I_yWB
   */
  protected void drawLine(float x0, float y0, float x1, float y1)
  {
    loadLineStyle();

    graphics().draw(transformer_.createTransformedShape(
      new Line2D.Double(x0, y0, x1, y1)));
  }

  /**
   * lp`悷B
   *
   * @param  x0 lp̍[xWB
   * @param  y0 lp̏[yWB
   * @param  x1 lp̉E[xWB
   * @param  y1 lp̉[yWB
   */
  protected void drawRect(float x0, float y0, float x1, float y1)
  {
    loadLineStyle();

    graphics().draw(transformer_.createTransformedShape(
      new Rectangle2D.Float(x0, y0, x1 - x0, y1 - y0)));
  }

  /**
   * hԂꂽlp`悷B
   *
   * @param  x0 lp̍[xWB
   * @param  y0 lp̏[yWB
   * @param  x1 lp̉E[xWB
   * @param  y1 lp̉[yWB
   */
  protected void fillRect(float x0, float y0, float x1, float y1)
  {
    loadFillStyle();

    graphics().fill(transformer_.createTransformedShape(
      new Rectangle2D.Float(x0, y0, x1 - x0, y1 - y0)));

    if (fillStyle_.hasOutline()) {
      drawRect(x0, y0, x1, y1);
    }
  }

  /**
   * ܂`悷B
   *
   * @param  x _xW̔zB
   * @param  y _yW̔zB
   * @throws IllegalArgumentException xW̔zyW̔z̃TCYقȂ
   *           ꍇB
   */
  protected void drawPolyLine(float[] x, float[] y)
  {
    if (x.length != y.length) {
      throw new IllegalArgumentException(
        "Array size of x and y are not equal.");
    }

    loadLineStyle();

    GeneralPath path = new GeneralPath();
    if (x.length > 0) {
      path.moveTo(x[0], y[0]);
      for (int i=1; i<x.length; i++) {
        path.lineTo(x[i], y[i]);
      }
    }
    graphics().draw(transformer_.createTransformedShape(path));
  }

  /**
   * p``悷B
   *
   * @param  x _xW̔zB
   * @param  y _yW̔zB
   * @throws IllegalArgumentException xW̔zyW̔z̃TCYقȂ
   *           ꍇB
   */
  protected void drawPolygon(float[] x, float[] y)
  {
    if (x.length != y.length) {
      throw new IllegalArgumentException(
        "Array size of x and y are not equal.");
    }

    loadLineStyle();

    GeneralPath path = new GeneralPath();
    if (x.length > 0) {
      path.moveTo(x[0], y[0]);
      for (int i=1; i<x.length; i++) {
        path.lineTo(x[i], y[i]);
      }
      path.closePath();
    }
    graphics().draw(transformer_.createTransformedShape(path));
  }

  /**
   * hԂꂽp``悷B
   *
   * @param  x _xW̔zB
   * @param  y _yW̔zB
   * @throws IllegalArgumentException xW̔zyW̔z̃TCYقȂ
   *           ꍇB
   */
  protected void fillPolygon(float[] x, float[] y)
  {
    if (x.length != y.length) {
      throw new IllegalArgumentException(
        "Array size of x and y are not equal.");
    }

    loadFillStyle();

    GeneralPath path = new GeneralPath();
    if (x.length > 0) {
      path.moveTo(x[0], y[0]);
      for (int i=1; i<x.length; i++) {
        path.lineTo(x[i], y[i]);
      }
      path.closePath();
    }
    graphics().fill(transformer_.createTransformedShape(path));

    if (fillStyle_.hasOutline()) {
      drawPolygon(x, y);
    }
  }

  /**
   * ~`悷B
   *
   * @param  x SxWB
   * @param  y SyWB
   * @param  r aB
   */
  protected void drawCircle(float x, float y, float r)
  {
    loadLineStyle();

    graphics().draw(transformer_.createTransformedShape(
      new Ellipse2D.Float(x-r, y-r, r*2-1, r*2-1)));
  }

  /**
   * hԂꂽ~`悷B
   *
   * @param  x SxWB
   * @param  y SyWB
   * @param  r aB
   */
  protected void fillCircle(float x, float y, float r)
  {
    loadFillStyle();

    graphics().fill(transformer_.createTransformedShape(
      new Ellipse2D.Float(x-r, y-r, r*2-1, r*2-1)));

    if (fillStyle_.hasOutline()) {
      drawCircle(x, y, r);
    }
  }

  /**
   * ȉ~`悷B
   *
   * @param  x SxWB
   * @param  y SyWB
   * @param  rx x̔aB
   * @param  ry y̔aB
   */
  protected void drawEllipse(float x, float y, float rx, float ry)
  {
    loadLineStyle();

    graphics().draw(transformer_.createTransformedShape(
      new Ellipse2D.Float(x-rx, y-ry, rx*2-1, ry*2-1)));
  }

  /**
   * hԂꂽȉ~`悷B
   *
   * @param  x SxWB
   * @param  y SyWB
   * @param  rx x̔aB
   * @param  ry y̔aB
   */
  protected void fillEllipse(float x, float y, float rx, float ry)
  {
    loadFillStyle();

    graphics().fill(transformer_.createTransformedShape(
      new Ellipse2D.Float(x-rx, y-ry, rx*2-1, ry*2-1)));

    if (fillStyle_.hasOutline()) {
      drawEllipse(x, y, rx, ry);
    }
  }

  /**
   * ʂ`悷B
   *
   * @param  x SxWB
   * @param  y SyWB
   * @param  rx x̔aB
   * @param  ry y̔aB
   * @param  startAngle Jnpx [deg]B
   * @param  endAngle Jnpx [deg]B
   */
  protected void drawArc(float x, float y, float rx, float ry, 
    float startAngle, float endAngle)
  {
    loadLineStyle();

    graphics().draw(transformer_.createTransformedShape(
      new Arc2D.Float(x-rx, y-ry, rx*2-1, ry*2-1,
      startAngle, endAngle - startAngle, Arc2D.PIE)));
  }

  /**
   * hԂꂽʂ`悷B
   *
   * @param  x SxWB
   * @param  y SyWB
   * @param  rx x̔aB
   * @param  ry y̔aB
   * @param  startAngle Jnpx [deg]B
   * @param  endAngle Jnpx [deg]B
   */
  protected void fillArc(float x, float y, float rx, float ry, 
    float startAngle, float endAngle)
  {
    loadFillStyle();

    graphics().fill(transformer_.createTransformedShape(
      new Arc2D.Float(x-rx, y-ry, rx*2-1, ry*2-1,
      startAngle, endAngle - startAngle, Arc2D.PIE)));
   
    if (fillStyle_.hasOutline()) {
      drawArc(x, y, rx, ry, startAngle, endAngle);
    }
  }

  /**
   * `悷B
   * <br>
   * ȂA{@link GraphImage#setScale(double,double) setScale(float,float)}
   * \bhŔ{ݒ肵ꍇA̕\ʒu͂ɍ킹ĕω
   * AtHgETCY͕ωȂB
   *
   * @param  text `悷镶B
   * @param  x ̍xWB
   * @param  y ̍yWB
   */
  protected void drawText(String text, float x, float y)
  {
    drawText(text, x, y, 0f);
  }

  /**
   * pxtĕ`悷B
   *
   * @param  text `悷镶B
   * @param  x ̍xWB
   * @param  y ̍yWB
   * @param  deg Xpx [deg]B
   */
  protected void drawText(String text, float x, float y, float deg)
  {
    loadTextStyle();

    Font font = graphics().getFont();
    AttributedString attrStr = new AttributedString(text, font.getAttributes());

    double rotate = deg * Math.PI / 180.0;
    double scaleX = transformer_.getScaleX();
    double scaleY = transformer_.getScaleY();
    double transX = scaleX * x + transformer_.getTranslateX();
    double transY = scaleY * y + transformer_.getTranslateY();

    graphics().translate(transX, transY);
    graphics().rotate(rotate);

    graphics().drawString(attrStr.getIterator(), 0, 0);

    graphics().rotate(-rotate);
    graphics().translate(-transX, -transY);
  }

  /**
   * ̃TCY擾B
   * <br>
   * ̃\bhɂ蓾TCÝApx{Ɉ˂ȂlłB
   *
   * @param  text B
   * @return ̃TCYi[zBPvf͕AQvf͍łB
   */
  protected float[] getTextSize(String text)
  {
    loadTextStyle();
    FontMetrics fm = graphics().getFontMetrics();
    float[] ret = new float[2];
    ret[W] = fm.stringWidth(text);
    ret[H] = fm.getHeight();
    return ret;
  }

  /**
   * `悷鎞̗̈擾B
   * <br>
   * ̈ʒu{ll擾B
   *
   * @param  text B
   * @param  x ̕`ʒuxWB
   * @param  y ̕`ʒuyWB
   * @return `̈̍Wi[zBPvf͍[xWAQvf
   *           [yWARvf͉E[xWASvf͉[yW
   *           B
   */
  protected float[] getTextDrawnBounds(String text, float x, float y)
  {
    float[] sz = getTextSize(text);

    double scaleX = transformer_.getScaleX();
    double scaleY = transformer_.getScaleY();

    double w = (double) sz[W] / scaleX;
    double h = (double) sz[H] / scaleY;
    double dh = 3.0 / scaleY;
     
    float[] ret = new float[4];
    ret[L] = x;
    ret[T] = (float) (y - h + dh);
    ret[R] = ret[L] + (float) w;
    ret[B] = ret[T] + (float) h;
    return ret;
  }

  /**
   * `悷鎞̗̈擾B
   * <br>
   * ̈ʒu{Apxll擾B
   *
   * @param  text B
   * @param  x ̕`ʒuxWB
   * @param  y ̕`ʒuyWB
   * @param  deg Xpx [deg]B
   * @return `̈̍Wi[zBPvf͍[xWAQvf
   *           [yWARvf͉E[xWASvf͉[yW
   *           B
   */
  protected float[] getTextDrawnBounds(String text, float x, float y, float deg)
  {
    float[] sz = getTextSize(text);

    double scaleX = transformer_.getScaleX();
    double scaleY = transformer_.getScaleY();

    double reg = deg * Math.PI / 180.0;
    double dh = 3.0;
    float[] pt = new float[4];
    pt[L] = (float)(- dh * Math.sin(reg));
    pt[T] = (float)(- (sz[H] - dh) * Math.cos(reg));
    pt[R] = (float)(sz[W] * Math.cos(reg) + (sz[H] - dh) * Math.sin(reg));
    pt[B] = (float)(sz[W] * Math.sin(reg) + dh * Math.cos(reg));

    pt[L] = x + pt[L] / (float) scaleX;
    pt[T] = y + pt[T] / (float) scaleY;
    pt[R] = x + pt[R] / (float) scaleX;
    pt[B] = y + pt[B] / (float) scaleY;
    return pt;
  }

  /**
   * Obh`悷B
   *
   * @param  gridWidth ObhB
   * @param  gridHeight ObhB
   */
  void drawGrid(int gridWidth, int gridHeight)
  {
    setLineStyle(new LineStyle(Color.lightGray, 1));

    int[] margin = getMargin();
    int x0 = margin[L];
    int y0 = margin[T];
    int x1 = Math.max(getWidth() - margin[R], 0);
    int y1 = Math.max(getHeight() - margin[B], 0);

    for (int i=x0; i<=x1; i+=gridWidth) {
      drawLine(i, y0, i, y1);
    }

    for (int i=y0; i<=y1; i+=gridHeight) {
      drawLine(x0, i, x1, i);
    }
  }
}

