/*
 * XmlResource class.
 *
 * Copyright (C) 2007 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.util.resource;

import ts.util.text.StringSequence;
import ts.util.text.StringOperation;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.NoSuchElementException;
import java.util.Properties;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.Transformer;
import javax.xml.transform.Source;
import javax.xml.transform.Result;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Text;

/**
 * XMLt@CΏۂƂ郊\[XNXB
 * <br>
 * XMLt@C[hāA̓ew肳ꂽL[ɑ΂l擾B
 * ܂Aw肳ꂽL[ɑ΂Ēlݒ肵A̓eXMLt@CɕۑB
 * OԂ̑ΉɂẮAOԂ肵ă[hۑ肷邱
 * ͂łȂAOԂgXMLt@CłAOԂgpȂꍇ
 * lɃ[hĒl擾Alݒ肵ĕۑł悤ɂĂB
 * <p>
 * XMLt@C̋LqƁA\[XIuWFNg̃L[Ƃ̑Ή́AႦΈȉ̂悤
 * ɂȂB
 * <pre>
 *  ` XMLt@C `
 *    &lt;aaa&gt;
 *      &lt;bbb&gt;
 *        &lt;ddd&gt;&lt;/ddd&gt;
 *        &lt;eee&gt;&lt;/eee&gt;
 *        &lt;eee&gt;&lt;/eee&gt;
 *      &lt;/bbb&gt;
 *      &lt;ccc&gt;
 *        &lt;fff id="F1"&gt;&lt;/fff&gt;
 *        &lt;fff id="F2"&gt;&lt;/fff&gt;
 *      &lt;/ccc&gt;
 *    &lt;/aaa&gt;
 *   - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 *    res : XmlResource
 *
 *    res.getFirstValue("aaa.bbb.ddd")  
 *    res.getFirstValue("aaa.bbb.eee")  
 *    res.getFirstValue("aaa.bbb.fff")  
 *
 *    res.getValues("aaa.bbb.ddd")      {  }
 *    res.getValues("aaa.bbb.eee")      { ,  }
 *    res.getValues("aaa.bbb.fff")      { ,  }
 *
 *    res.getFirstValue("aaa.bbb.eee(id=F1)")  
 *    res.getFirstValue("aaa.bbb.eee(id=F2)")  
 *
 *    res.getValues("aaa.bbb.eee(id=F2)")  {  }
 *
 *    res.getFirstAttribute("aaa.bbb.fff", "id")         F1
 *    res.getFirstAttribute("aaa.bbb.fff(id=F2)", "id")  F2
 *
 *    res.getAttributes("aaa.bbb.fff", "id")         F1
 *    res.getAttributes("aaa.bbb.fff(id=F2)", "id")  F2
 * </pre>
 *
 *
 * @author  V. 
 * @version $Revision: 1.3 $, $Date: 2010-11-14 11:51:02 $
 */
public class XmlResource extends AbstractResource
{
  /** XMLf[^ێhLgIuWFNgB */
  private Document document_ ;

  /** XML̊m[hB */
  private Node baseNode_ ;

  /** XML̐`FbNsǂtOB */
  private boolean isValidating_ = true;

  /**
   * ftHgRXgN^B
   */
  public XmlResource()
  {
    renew();
  }

  /**
   * XMLt@C̃pXɂƂRXgN^B
   *
   * @param  path XMLt@C̃pXB
   * @throws FileNotFoundException w肵pX̃t@CȂꍇB
   * @throws IOException t@C̓o͒ɗOꍇB
   */
  public XmlResource(String path) throws FileNotFoundException, IOException
  {
    load(path);
  }

  /**
   * Rs[RXgN^B
   *
   * @param  res Rs[XML\[XIuWFNgB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  protected XmlResource(XmlResource res)
  {
    super(res);

    document_ = res.document_ ;
    baseNode_ = res.baseNode_ ;
  }

  /**
   * XMLt@CǂݍލۂXML̐`FbN邩ǂݒ肷B
   * <br>
   * ftHgł<tt>false</tt>łB
   */
  public void setValidating(boolean isValidating)
  {
    isValidating_ = isValidating;
  }

  /**
   * XMLhLgIuWFNg擾B
   *
   * @return XMLhLgIuWFNgB
   */
  protected Document getDocument()
  {
    return document_ ;
  }

  /**
   * ̃IuWFNghLgIuWFNg̊m[h擾B
   *
   * @return m[hB
   */
  protected Node getBaseNode()
  {
    return baseNode_ ;
  }

  /**
   * vf̖O擾B
   * <br>
   * vfɖOԂ܂܂Ăꍇ́A[JԂB
   *
   * @return vf̖OB
   */
  public String getBaseElementName()
  {
    return getBaseNode().getLocalName();
  }

  /**
   * w肳ꂽL[ɍŏɊYm[h擾B
   * <br>
   * Ym[h݂Ȃꍇ̓kԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @return ̃L[ɍŏɑΉm[hB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  protected Node findFirstNode(String key) throws IllegalKeyException
  {
    assert (key != null) : "@param:key is null.";

    if (StringOperation.isEmpty(key)) {
      return getBaseNode();
    }

    List<String> keyElemLst = splitKey(key);
    return findFirstNode(getBaseNode(), keyElemLst, 0);
  }

  /**
   * L[vfXgɍŏɊY鉺ʃm[hċAIɒTB
   *
   * @param  node  m[hB
   * @param  keyElemLst L[vfXgB 
   * @param  index ݒT̃L[vf̃CfbNXB
   * @return L[vfXgɊYm[hB
   * @throws NullPointerException k̏ꍇB
   * @throws IndexOutOfBoundsException L[vf̃CfbNX̒l̏ꍇB
   */
  private Node findFirstNode(Node node, List<String> keyElemLst, int index)
    throws NullPointerException, IndexOutOfBoundsException
  {
    if (index >= keyElemLst.size()) {
      return node;
    }

    String keyElem = keyElemLst.get(index);
    String elemName = getNameOfKeyElement(keyElem);
    List<String[]> attrLst = listAttributesOfKeyElement(keyElem);

    NEXT_NODE:
    for (Node n = node.getFirstChild(); n != null; n = n.getNextSibling()) {
      if (n.getNodeType() != Node.ELEMENT_NODE) {
        continue;
      }

      // for namespace support.
      //if (! n.getNodeName().equals(elemName)) {
      if (! elemName.equals("*") && ! elemName.equals(n.getLocalName())) {
        continue;
      }

      if (attrLst.size() > 0) {
        NamedNodeMap attrNodeMap = n.getAttributes(); 
        for (String[] attr : attrLst) {
          Node attrNode = attrNodeMap.getNamedItem(attr[0]);
          if (attrNode == null || ! attr[1].equals(attrNode.getNodeValue())) {
            continue NEXT_NODE;
          }
        }
      }

      Node child = findFirstNode(n, keyElemLst, index + 1);
      if (child != null) {
        return child;
      }
    }

    return null;
  }

  /**
   * w肳ꂽL[ɊYSẴm[h擾B
   * <br>
   * Ym[h݂Ȃꍇ͋̃XgԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @return ̃L[ɑΉSẴm[hi[郊XgB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  protected List<Node> findAllNodes(String key) throws IllegalKeyException
  {
    assert (key != null) : "@param:key is null.";

    List<Node> nodeLst = new LinkedList<Node>();

    if (StringOperation.isEmpty(key)) {
      nodeLst.add(getBaseNode());
      return nodeLst;
    }

    List<String> keyElemLst = splitKey(key);
    findAllNodes(nodeLst, getBaseNode(), keyElemLst, 0);
    return nodeLst;
  }

  /**
   * L[vfXgɊYSẲʃm[hċAIɒTāAXg
   * i[B
   *
   * @param  nodeLst  Ym[hi[郊XgB
   * @param  node  m[hB
   * @param  keyElemLst L[vfXgB 
   * @param  index ݒT̃L[vf̃CfbNXB
   * @throws NullPointerException k̏ꍇB
   * @throws IndexOutOfBoundsException CfbNX̒l̏ꍇB
   */
  private void findAllNodes(
    List<Node> nodeLst, Node node, List<String> keyElemLst, int index
  ) throws NullPointerException, IndexOutOfBoundsException
  {
    if (index >= keyElemLst.size()) {
      nodeLst.add(node);
      return;
    }

    String keyElem = keyElemLst.get(index);
    String elemName = getNameOfKeyElement(keyElem);
    List<String[]> attrLst = listAttributesOfKeyElement(keyElem);

    NEXT_NODE:
    for (Node n = node.getFirstChild(); n != null; n = n.getNextSibling()) {
      if (n.getNodeType() != Node.ELEMENT_NODE) {
        continue;
      }

      // for namespace support.
      //if (! n.getNodeName().equals(elemName)) {
      if (! elemName.equals("*") && ! elemName.equals(n.getLocalName())) {
        continue;
      }

      if (attrLst.size() > 0) {
        NamedNodeMap attrNodeMap = n.getAttributes(); 
        for (String[] attr : attrLst) {
          Node attrNode = attrNodeMap.getNamedItem(attr[0]);
          if (attrNode == null || ! attr[1].equals(attrNode.getNodeValue())) {
            continue NEXT_NODE;
          }
        }
      }

      findAllNodes(nodeLst, n, keyElemLst, index + 1);
    }
  }

  /**
   * L[vfXgɍŏɊY鉺ʃm[hTāA݂΂ԂA
   * ݂Ȃ΍쐬ĕԂB
   * <br>
   * ̃L[sȏꍇ́AOX[B
   *
   * @param  key L[B
   * @return ̃L[ɍŏɑΉA͍쐬ꂽm[hB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  protected Node findOrCreateFirstNode(String key) throws IllegalKeyException
  {
    assert (key != null) : "@param:key is null.";

    if (StringOperation.isEmpty(key)) {
      return getBaseNode();
    }

    List<String> keyElemLst = splitKey(key);

    List<String> singleKeyL = new ArrayList<String>(1);
    singleKeyL.add("");

    Node node = getBaseNode();
    int i = 0;
    for (; i<keyElemLst.size(); i++) {
      singleKeyL.set(0, keyElemLst.get(i));
      Node n = findFirstNode(node, singleKeyL, 0);
      if (n == null) {
        break;
      }
      node = n;
    }

    for (; i<keyElemLst.size(); i++) {
      String keyElem = keyElemLst.get(i);
      String elemName = getNameOfKeyElement(keyElem);
      List<String[]> attrLst = listAttributesOfKeyElement(keyElem);

      Element newElem = getDocument().createElementNS(
        node.getNamespaceURI(), elemName);

      for (String[] attr : attrLst) {
        newElem.setAttribute(attr[0], attr[1]);
      }
      node.appendChild(newElem);
      node = newElem;
    }
    return node;
  }

  /**
   * w肳ꂽm[h̃eLXgl擾B
   * <br>
   * k̏ꍇ͋󕶎ԂB
   *
   * @param  node m[hIuWFNgB
   * @return ̃m[h̃eLXglB
   */
  protected String getNodeText(Node node)
  {
    if (node == null) {
      return "";
    }

    StringBuilder buf = new StringBuilder();

    NodeList nodeLst = node.getChildNodes();
    if (nodeLst.getLength() > 0) {
      Node n = nodeLst.item(0);
      switch (n.getNodeType()) {
      case Node.CDATA_SECTION_NODE :
        buf.append(n.getNodeValue());
        break;
      case Node.TEXT_NODE :
        buf.append(shortenWhitespace(n.getNodeValue()));
        break;
      }

      for (int i=1; i<nodeLst.getLength(); i++) {
        n = nodeLst.item(i);

        String s;
        switch (n.getNodeType()) {
        case Node.CDATA_SECTION_NODE :
          s = n.getNodeValue();
          if (! StringOperation.isEmpty(s)) {
            buf.append(" ").append(s);
          }
          break;
        case Node.TEXT_NODE :
          s = shortenWhitespace(n.getNodeValue());
          if (! StringOperation.isEmpty(s)) {
            buf.append(" ").append(s);
          }
          break;
        }
      }
    }

    return buf.toString();
  }

  /**
   * w肳ꂽ񒆂̋󔒂̕т̋󔒂ɒZkB
   * <br>
   * AA̐擪ƏI[̋󔒂͏B
   *
   * @param  s B
   * @return 󔒂̕т̋󔒂ɒZkꂽB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  protected String shortenWhitespace(String s)
  {
    assert (s != null) : "@param:s is null.";

    StringBuilder buf = new StringBuilder(s.length());
    StringSequence seq = new StringSequence(s).skipWhitespaces();
    while (seq.validIndex()) {
      if (seq.index() != seq.skipWhitespaces().index()) {
        if (! seq.validIndex()) {
          break;
        }
        buf.append(" ");
      }
      buf.append(seq.character());
      seq.next();
    }
    return buf.toString();
  }

  /**
   * w肳ꂽL[ɍŏɊYvf̒l擾B
   * <br>
   * Yvf݂Ȃꍇ͋󕶎ԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @return ̃L[ɍŏɊYvf̒lB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public synchronized String getFirstValue(String key)
    throws IllegalKeyException
  {
    return getNodeText(findFirstNode(key));
  }

  /**
   * w肳ꂽL[ɊYSvf̒l̃Xg擾B
   * <br>
   * Yvf݂Ȃꍇ͋̃XgԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @return ̃L[ɊYSvf̒l̃XgB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError ̃k̏ꍇifobO[ĥ݁jB
   */
  public synchronized List<String> getValues(String key)
    throws IllegalKeyException
  {
    List<Node> nodeLst = findAllNodes(key);

    List<String> valueLst = new ArrayList<String>(nodeLst.size());
    for (Node node : nodeLst) {
      valueLst.add(getNodeText(node));
    }

    return valueLst;
  }

  /**
   * w肳ꂽL[ɍŏɊYvf́Aw肳ꂽ̒l擾B
   * <br>
   * Yvf݂Ȃꍇ͋󕶎ԂB
   * YvfɁAw肳ꂽȎ݂Ȃꍇ͋󕶎ԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @param  attr B
   * @return ̃L[ɍŏɊYvf́Aw肳ꂽ̒lB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public synchronized String getFirstAttribute(String key, String attr)
    throws IllegalKeyException
  {
    assert (attr != null) : "@param:attr is null.";

    Node node = findFirstNode(key);
    if (node == null) {
      return getNodeText(null);
    }

    NamedNodeMap attrMap = node.getAttributes();
    if (attrMap == null) {
      return getNodeText(null);
    }

    return getNodeText(attrMap.getNamedItem(attr));
  }

  /**
   * w肳ꂽL[ɊYSvf́Aw肳ꂽ̒l̃Xg擾B
   * <br>
   * Yvf݂Ȃꍇ͋̃XgԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @param  attr B
   * @return ̃L[ɊYSvf́Aw肳ꂽ̒l̃XgB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertError k̏ꍇB
   */
  public synchronized List<String> getAttributes(String key, String attr)
    throws IllegalKeyException
  {
    assert (attr != null) : "@param:attr is null.";

    List<Node> nodeLst = findAllNodes(key);

    List<String> valueLst = new ArrayList<String>(nodeLst.size());
    for (Node node : nodeLst) {
      NamedNodeMap attrMap = node.getAttributes();
      if (attrMap == null) {
        valueLst.add(getNodeText(null));
      }
      else {
        valueLst.add(getNodeText(attrMap.getNamedItem(attr)));
      }
    }

    return valueLst;
  }

  /**
   * w肳ꂽL[ɍŏɊYvfvfƂq\[X擾B
   * <br>
   * w肳ꂽL[ɊYvf݂Ȃꍇ
   * {@link ts.util.resource.Resource#EMPTY}ԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @return ̃L[ɍŏɊYvf̎q\[XB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public synchronized Resource getFirstChild(String key)
    throws IllegalKeyException
  {
    Node node = findFirstNode(key);
    if (node == null) {
      return Resource.EMPTY;
    }

    XmlResource res = new XmlResource(this);
    res.baseNode_ = node;
    return res;
  }

  /**
   * w肳ꂽL[ɍŏɊYq\[X̃Xg擾B
   * <br>
   * w肳ꂽL[ɊYvf݂Ȃꍇ͋̃XgԂB
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @return ̃L[ɊYvf̎q\[X̃XgB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public synchronized List<Resource> getChildren(String key)
    throws IllegalKeyException
  {
    List<Node> nodeLst = findAllNodes(key);

    List<Resource> resLst = new ArrayList<Resource>(nodeLst.size());
    for (Node n : nodeLst) {
      XmlResource res = new XmlResource(this);
      res.baseNode_ = n;
      resLst.add(res);
    }

    return resLst;
  }

  /**
   * w肳ꂽL[ɍŏɊYvfɁAlݒ肷B
   * <br>
   * w肳ꂽL[ɊYvf݂Ȃꍇ́A̗vf쐬
   * lݒ肷B
   *
   * @param  key L[B
   * @param  value ̃L[ɍŏɊYvfɐݒ肷lB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public synchronized void setFirstValue(String key, String value)
    throws IllegalKeyException
  {
    assert (value != null) : "@param:value is null.";

    try {
      Node node = findOrCreateFirstNode(key);
      Text text = getDocument().createCDATASection(value);

      for (Node n = node.getFirstChild(); n != null; n = n.getNextSibling()) {
        if (n.getNodeType() == Node.TEXT_NODE
        ||  n.getNodeType() == Node.CDATA_SECTION_NODE) {
          node.removeChild(n);
        }
      }
      node.appendChild(text);
    }
    catch (IllegalKeyException e) {
      throw e;
    }
    catch (Exception e) {
      throw new IllegalKeyException(key, e);
    }
  }

  /**
   * w肳ꂽL[ɍŏɊYvfvfƂq\[X擾B
   * w肳ꂽL[ɊYvf݂Ȃꍇ͍쐬ĕԂB
   * <br>
   * ̃L[sȏꍇ͗OX[B
   *
   * @param  key L[B
   * @return w肳ꂽL[ɍŏɊYvfvfƂq\[XB
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public synchronized Resource getOrCreateFirstChild(String key)
    throws IllegalKeyException
  {
    try {
      Node node = findOrCreateFirstNode(key);

      XmlResource res = new XmlResource(this);
      res.baseNode_ = node;
      return res;
    }
    catch (IllegalKeyException e) {
      throw e;
    }
  }

  /**
   * w肳ꂽL[ɊYSĂ̗vf폜B
   * <br>
   * w肳ꂽL[ɊYvfȂꍇ́AȂB
   *
   * @param  key L[B
   * @throws IllegalKeyException L[sȏꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  public synchronized void removeChildren(String key)
    throws IllegalKeyException
  {
    List<Node> nodeLst = findAllNodes(key);
    for (Node node : nodeLst) {
      Node parent = node.getParentNode();
      if (parent != null) {
        parent.removeChild(node);
      }
    }

    if (StringOperation.isEmpty(key)) {
      renew();
    }
  }

  /**
   * ݂̊vf̑SẴeLXglƁAqvf̃\[XEIuWFNgi[
   * Xg擾B
   *
   * @return ݂̊vf̃eLXglAyюqvf̃\[XEIuWFNg
   *         i[XgB
   */
  public synchronized List<Object> getValuesAndChildren()
  {
    List<Object> lst = new LinkedList<Object>();
    Node base = getBaseNode();
    for (Node n = base.getFirstChild(); n != null; n = n.getNextSibling()) {
      switch (n.getNodeType()) {
      case Node.ELEMENT_NODE :
        XmlResource res = new XmlResource(this);
        res.baseNode_ = n;
        lst.add(res);
        break;
      case Node.CDATA_SECTION_NODE :
        lst.add(n.getNodeValue());
        break;
      case Node.TEXT_NODE :
        String s = n.getNodeValue().trim();
        if (s.length() == 0) continue;
        lst.add(shortenWhitespace(s));
        break;
      }
    }
    return lst;
  }

  /**
   * ̃IuWFNgďB
   */
  private void renew()
  {
    try {
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      factory.setIgnoringComments(true);
      factory.setNamespaceAware(true);
      factory.setValidating(false);
      DocumentBuilder builder = factory.newDocumentBuilder();

      document_ = builder.newDocument();
      baseNode_ = document_;
    }
    catch (Exception e) {}
  }

  /**
   * ̓̓Xg[XMLt@C̓eǂݍށB
   *
   * @param  stream ̓Xg[B
   * @throws IOException ǂݍݒɗOꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  @Override
  public void load(InputStream stream) throws IOException
  { 
    assert (stream != null) : "@param:stream is null.";

    try {
      DocumentBuilder builder = createDocumentBuilder();

      document_ = builder.parse(stream);
      baseNode_ = document_;
    }
    catch (IOException e) {
      throw e;
    }
    catch (Exception e) {
      IOException exc = new IOException(e.getMessage());
      exc.setStackTrace(e.getStackTrace());
      throw exc;
    }
  }

  /**
   * XMLhLg\zIuWFNg쐬B
   *
   * @return XMLhLg\zIuWFNgB
   * @throws ParserConfigurationException vꂽ\𖞂XMLhLg
   *           \zIuWFNg𐶐łȂꍇ
   */
  protected DocumentBuilder createDocumentBuilder()
    throws ParserConfigurationException
  {
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    factory.setIgnoringComments(true);
    factory.setNamespaceAware(true);
    factory.setValidating(isValidating_);

    DocumentBuilder builder = factory.newDocumentBuilder();
    if (! factory.isValidating()) {
      builder.setEntityResolver(new EntityResolver() {
        public InputSource resolveEntity(String publicId, String systemId)
        throws SAXException, IOException {
          return new InputSource(new StringReader(""));
        }
      });
    }
    return builder;
  }

  /**
   * ̃IuWFNgێĂhLg̓eAw肳ꂽ
   * o̓Xg[ɕۑB
   *
   * @param  stream o̓Xg[B
   * @throws UnsupportedOperationException q\[Xۑ悤ƂꍇB
   * @throws IOException t@C̕ۑɗOꍇB
   * @throws AssertionError k̏ꍇifobO[ĥ݁jB
   */
  @Override
  public void save(OutputStream stream) throws IOException
  {
    assert (stream != null) : "@param:stream is null.";

    if (getBaseNode() != getDocument()) {
      throw new UnsupportedOperationException(
        "XmlResource class cannot save a child resource.");
    }

    try {
      DocumentType doctype = getDocument().getDoctype();

      TransformerFactory factory = TransformerFactory.newInstance();
      Transformer t = factory.newTransformer();
      t.setOutputProperty(OutputKeys.INDENT, "yes");
      t.setOutputProperty(OutputKeys.METHOD, "xml");
      t.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, doctype.getSystemId());
      t.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC,
        (doctype.getPublicId() != null) ? doctype.getPublicId() : "");

      // ۑt@C̕GR[fBOAǍt@CƓɂ
      Properties prop = t.getOutputProperties();
      prop.setProperty(OutputKeys.ENCODING, getDocument().getXmlEncoding());
      t.setOutputProperties(prop);

      Source s = new DOMSource(getBaseNode());
      Result r = new StreamResult(stream);
      t.transform(s, r);
    }
    catch (Exception e) {
      IOException exc = new IOException(e.getMessage());
      exc.setStackTrace(e.getStackTrace());
      throw exc;
    }
  }
}
