package ts.query;

import ts.tester.UnitTest;
import ts.util.*;
import ts.util.file.*;
import ts.util.resource.*;
import ts.util.table.*;
import ts.util.text.*;
import java.io.*;
import java.util.*;

public class AbstractQueryTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(AbstractQueryTest.class, args);
  }


  /*
   * XMLファイルから値を取得するサンプル・クエリ・クラス。
   */
  static class XmlFileQuery extends AbstractQuery {
    static final String VALUE = "Value";
    public enum Attribute { NodePath, }
    public XmlFileQuery(QueryConnection conn, Resource res) {
      super(conn, res);
    }
    @Override
    public List<String> listAllQueryIds() {
      List<Resource> resL = getResource().getChildren("query.*");
      List<String> keyL = new ArrayList<String>(resL.size());
      for (Resource res : resL) {
        String id = res.getFirstAttribute("", "id");
        keyL.add(res.getBaseElementName() + "." + id);
      }
      return keyL;
    }
    @Override
    public List<QueryParam> prepareQuery(QueryContext ctx,
    Map<String,Object> inpMap, QueryResultList rsltLst)
    throws ReasonedException {
      List<QueryParam> paramLst = new LinkedList<QueryParam>();

      List<String> lst = StringCutter.split(ctx.getQueryId(), ".");
      if (lst.size() < 2) return paramLst;

      String key = "query." + lst.get(0) + "(id=" + lst.get(1) + ")";
      ctx.set(Attribute.NodePath, getResource().getFirstValue(key));

      if ("select".equals(lst.get(0))) {
        ctx.addOutput(new QueryOutput(VALUE));
      }
      else if ("update".equals(lst.get(0))) {
        Object obj = inpMap.get(VALUE);
        QueryParam param = new QueryParam(VALUE, QueryParam.IO.IN);
        param.setValueIterator(new QueryParamValueIterator(obj));
        paramLst.add(param);
        ctx.addInputName(VALUE);
      }

      return paramLst;
    }
    @Override
    public int executeQuery(QueryContext ctx, List<QueryParam> paramLst,
    Table<String,Object> tbl) throws ReasonedException {
      XmlFileQueryConnection xmlConn;
      xmlConn = XmlFileQueryConnection.class.cast(getConnection());
      XmlResource xmlRes = xmlConn.getXmlResource();
      try {
        Thread.sleep(1000L);
      } catch (InterruptedException e) {}

      checkTimeout(ctx);

      String nodePath = ctx.getString(Attribute.NodePath);
      List<String> lst = xmlRes.getValues(nodePath);
      for (String s : lst) {
        tbl.appendNew().put(VALUE, s);
      }

      return lst.size();
    }
    @Override
    public int executeQuery(QueryContext ctx, List<QueryParam> paramLst)
    throws ReasonedException {
      XmlFileQueryConnection xmlConn;
      xmlConn = XmlFileQueryConnection.class.cast(getConnection());
      XmlResource xmlRes = xmlConn.getXmlResource();

      try {
        Thread.sleep(1000L);
      } catch (InterruptedException e) {}

      checkTimeout(ctx);

      String nodePath = ctx.getString(Attribute.NodePath);

      int i = 0;
      for (QueryParam param : paramLst) {
        String val = "";
        if (param.getValue() != null) {
          val = param.getValue().toString();
        }
        xmlRes.setFirstValue(nodePath + "(no=" + i + ")", val);
        i ++;
      }
      xmlConn.setModified(true);
      return i;
    }
  }

  /*
   * XMLファイルへのアクセスを表すコネクション・クラス。
   */
  static class XmlFileQueryConnection implements QueryConnection {
    private final String connId;
    private final XmlResource resource;
    private long limitTm;
    private boolean isModified = false;
    public enum Error {
      FileNotFound, FailToLoadFile, FailToSaveFile,
    }
    public XmlFileQueryConnection(String connId) {
      this.connId = connId;
      this.resource = new XmlResource();
      this.resource.setValidating(false);
    }
    public XmlResource getXmlResource() {
      return this.resource;
    }
    public void setModified(boolean isModified) {
      this.isModified = isModified;
    }
    @Override
    public String getId() {
      return this.connId;
    }
    @Override
    public void setLimitTimeMillis(long tm) {
      this.limitTm = tm;
    }
    @Override
    public long getLimitTimeMillis() {
      return this.limitTm;
    }
    @Override
    public Query getQuery(Resource res) {
      return new XmlFileQuery(this, res);
    }
    @Override
    public void open() {
      String path = QueryConfig.getInstance().getString("sample.dat.path");
      try {
        this.resource.load(path);
      } catch (FileNotFoundException e) {
        throw new ReasonedRuntimeException(Error.FileNotFound, path, e);
      } catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToLoadFile, path, e);
      }
    }
    @Override
    public void commit() {
      if (! this.isModified) return;
      String path = QueryConfig.getInstance().getString("sample.dat.path");
      try {
        this.resource.save(path + ".tmp");
        FileOperation.delete(new File(path));
        FileOperation.move(new File(path + ".tmp"), new File(path));
      } catch (Exception e) {
        throw new ReasonedRuntimeException(Error.FailToSaveFile, path, e);
      }
    }
    @Override
    public void rollback() {
      try {
        this.resource.load(
          QueryConfig.getInstance().getString("sample.dat.path"));
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    }
    @Override
    public void close() {
      try {
        this.resource.removeChildren("");
      } catch (Exception e) {
        throw new RuntimeException(e);
      }
    }
  }

  /*
   * XmlFileQueryConnectionオブジェクトを作成するためのファクトリ・クラス。
   */
  static class XmlFileQueryConnectionFactory implements QueryConnectionFactory {
    public QueryConnection create(String connId, QueryConfig config)
    throws ReasonedException {
      return new XmlFileQueryConnection(connId);
    }
  }

  /*
   * 実行時に例外をスローするクエリ・クラス。
   */
  static class XmlFileQuery2 extends XmlFileQuery
  {
    public enum Error {
      Test,
    }
    public XmlFileQuery2(QueryConnection conn, Resource res) {
      super(conn, res);
    }
    @Override
    public int executeQuery(QueryContext ctx, List<QueryParam> paramLst)
      throws ReasonedException
    {
      throw new ReasonedException(Error.Test);
    }
  }

  static class XmlFileQueryConnection2 extends XmlFileQueryConnection {
    public XmlFileQueryConnection2(String connId) {
      super(connId);
    }
    @Override
    public Query getQuery(Resource res) {
      return new XmlFileQuery2(this, res);
    }
  }

  static class XmlFileQueryConnectionFactory2
    extends XmlFileQueryConnectionFactory
  {
    public QueryConnection create(String connId, QueryConfig config)
    throws ReasonedException {
      return new XmlFileQueryConnection2(connId);
    }
  }
  

  /**
   * データ・ファイルを元に戻す。
   */
  void rollbackDataFile()
  {
    try {
      File dir = new File("test/data/ts/query");
      File file0 = new File(dir, "SampleData.dat");
      File file1 = new File(dir, "SampleData.xml");
      if (file1.exists()) {
        FileOperation.delete(file1);
      }
      FileOperation.copy(file0, file1);
    }
    catch (Exception e) {
      NG(e);
    }
  }

  /* -- test -- */

  public void constructor_conn_res()
  {
    MSG("コンストラクタの確認。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    XmlFileQueryConnection conn = new XmlFileQueryConnection("Conn0");
    TRUE(conn instanceof QueryConnection);
    EQUAL(conn.getId(), "Conn0");
    TRUE(conn.getXmlResource() instanceof XmlResource);

    AbstractQuery query = new XmlFileQuery(conn, res);
    EQUAL(query.getConnection(), conn);
    EQUAL(query.getResource(), res);
  }

  public void constructor_conn_res_Null()
  {
    MSG("引数ヌルの場合。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    QueryConnection conn = new XmlFileQueryConnection("Conn0");

    try {
      new XmlFileQuery(null, res);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }

    try {
      new XmlFileQuery(conn, null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void listAllQueryIds()
  {
    MSG("クエリのIDのリストを取得するメソッドの確認。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    QueryConnection conn = new XmlFileQueryConnection("Conn0");

    XmlFileQuery query = new XmlFileQuery(conn, res);
    List<String> lst = query.listAllQueryIds();
    EQUAL(lst.size(), 8);
    EQUAL(lst.get(0), "select.c1");
    EQUAL(lst.get(1), "select.c2");
    EQUAL(lst.get(2), "select.d1");
    EQUAL(lst.get(3), "select.d2");
    EQUAL(lst.get(4), "update.c1");
    EQUAL(lst.get(5), "update.c2");
    EQUAL(lst.get(6), "update.d1");
    EQUAL(lst.get(7), "update.d2");
  }

  public void prepareQuery_ctx_inputMap_resultLst()
  {
    MSG("クエリ実行の準備処理を行うメソッドの確認。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    QueryConnection conn = new XmlFileQueryConnection("Conn0");

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＡＢＣＤＥＦＧ");

    XmlFileQuery query = new XmlFileQuery(conn, res);

    QueryContext ctx;
    QueryResultList rsltL = new QueryResultList();
    List<QueryParam> paramL;

    try {
      ctx = query.createContext("Conn0", "select.c1");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "select.c1");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b1).CCC");
  
      ctx = query.createContext("Conn0", "select.c2");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "select.c2");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b2).CCC");
  
      ctx = query.createContext("Conn0", "select.d1");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "select.d1");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b1).DDD");
  
      ctx = query.createContext("Conn0", "select.d2");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "select.d2");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b2).DDD");
     
      ctx = query.createContext("Conn0", "update.c1");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "update.c1");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b1).CCC");
  
      ctx = query.createContext("Conn0", "update.c2");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "update.c2");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b2).CCC");
  
      ctx = query.createContext("Conn0", "update.d1");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "update.d1");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b1).DDD");
  
      ctx = query.createContext("Conn0", "update.d2");
      EQUAL(ctx.getConnectionId(), "Conn0");
      EQUAL(ctx.getQueryId(), "update.d2");
      query.prepareQuery(ctx, inputMap, rsltL);
      EQUAL(ctx.get(XmlFileQuery.Attribute.NodePath), "AAA.BBB(id=b2).DDD");
    }
    catch (ReasonedException e) {
      NG(e);
    }
  }

  public void createResultTable()
  {
    MSG("結果テーブルを作成するメソッドの確認。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();

    QueryConnection conn = new XmlFileQueryConnection("Conn0");

    QueryResultList rsltL = new QueryResultList();

    XmlFileQuery query = new XmlFileQuery(conn, res);
    QueryContext ctx = query.createContext("Conn0", "select.c1");
    try {
      query.prepareQuery(ctx, inputMap, rsltL);
    } catch (ReasonedException e) {
      NG(e);
    }

    Table<String,Object> tbl = query.createResultTable(ctx);
    TRUE(tbl instanceof ArrayListTable);
    EQUAL(tbl.recordCount(), 0);
    EQUAL(tbl.columnCount(), 1);

    Enumeration<String> enm = tbl.columns();
    TRUE(enm.hasMoreElements());
    EQUAL(enm.nextElement(), "Value");
    FALSE(enm.hasMoreElements());
  }

  public void createResultTable_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    QueryConnection conn = new XmlFileQueryConnection("c0");

    XmlFileQuery query = new XmlFileQuery(conn, res);

    try {
      query.createResultTable(null);
      NG();
    } catch (AssertionError e) {
      OK(e);
    }
  }

  public void execute_queryId_inputMap_resultLst()
  {
    MSG("クエリ単体を実行するメソッドの確認（検索）。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr = null;
    try {
      tr = QueryTransactionManager.createThreadLocalTransaction();
    } catch (ReasonedRuntimeException e) {
      OK(e.getReason().name());
    }
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute("select.c1", inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 1);

    QueryResult rslt1 = rsltLst.getFirstResult("select.c1");
    TRUE(rslt1.isSuccess());
    POSITIVE(rslt1.getSpentTimeMillis());
    EQUAL(rslt1.getResultCount(), 1);
    NULL(rslt1.getException());
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "あいうえお");

    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "select.c1");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    TRUE(rslt1.getContext().hasResultTable());
  }

  public void execute_queryId_inputMap_resultLst_Update()
  {
    MSG("クエリ単体を実行するメソッドの確認（更新）。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＡＢＣＤＥＦＧ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr = null;
    try {
      tr = QueryTransactionManager.createThreadLocalTransaction();
    } catch (ReasonedRuntimeException e) {
      OK(e.getReason().name());
    }
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute("update.c1", inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }

    TRUE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 1);

    QueryResult rslt1 = rsltLst.getFirstResult("update.c1");
    TRUE(rslt1.isSuccess());
    POSITIVE(rslt1.getSpentTimeMillis());
    EQUAL(rslt1.getResultCount(), 1);
    NULL(rslt1.getException());
    NULL(rslt1.getResultTable());

    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "update.c1");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    FALSE(rslt1.getContext().hasResultTable());

    try {
      XmlResource resData = new XmlResource();
      resData.setValidating(false);
      try {
        resData.load(QueryConfig.getInstance().getString("sample.dat.path"));
      } catch (Exception e) {}
      EQUAL(resData.getFirstValue("AAA.BBB(id=b1).CCC(no=0)"),
        "ＡＢＣＤＥＦＧ");
    } catch (Exception e) {
      NG(e);
    }

    rollbackDataFile();
  }

  public void execute_queryId_inputMap_resultLst_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＡＢＣＤＥＦＧ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr = null;
    try {
      tr = QueryTransactionManager.createThreadLocalTransaction();
    } catch (ReasonedRuntimeException e) {
      OK(e.getReason().name());
    }
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);

      try {
        query.execute((String) null, inputMap, rsltLst);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      try {
        query.execute("select.c1", null, rsltLst);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      try {
        query.execute("select.c1", inputMap, null);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }
  }

  public void execute_queryId_inputMap_resultLst_Timeout()
  {
    MSG("途中でタイムアウトが発生した場合。");

    QueryConfig cfg = QueryConfig.getInstance();
    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＡＢＣＤＥＦＧ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr = null;
    try {
      tr = QueryTransactionManager.createThreadLocalTransaction();
    } catch (ReasonedRuntimeException e) {
      OK(e.getReason().name());
    }
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin(500L);
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute("select.c1", inputMap, rsltLst);
      NG();
      tr.commit();
    }
    catch (ReasonedException e) {
      EQUAL(e.getReason(), Query.Error.Timeout);
      EQUAL(e.getMessage(), "select.c1");
      tr.rollback();
    }
    catch (Exception e) {
      NG(e);
      tr.rollback();
    }
    finally {
      tr.end();
    }
  }

  public void execute_queryIdArray_inputMap_resultLst()
  {
    MSG("配列に格納された複数のクエリを実行するメソッドの確認。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(new String[]{"select.c2","update.c2","select.c2"},
        inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    TRUE(rsltLst.isSuccess());
    EQUAL(rsltLst.countResults(), 3);
    POSITIVE(rsltLst.getSpentTimeMillis());

    QueryResult rslt1 = rsltLst.getFirstResult("select.c2");
    QueryResult rslt2 = rsltLst.getFirstResult("update.c2");
    QueryResult rslt3 = rsltLst.getLastResult("select.c2");
    EQUAL(rsltLst.getResults("select.c2").size(), 2);
    EQUAL(rsltLst.getResults("update.c2").size(), 1);
    NOTEQUAL(rslt1, rslt2);
    NOTEQUAL(rslt1, rslt3);
    NOTEQUAL(rslt2, rslt3);

    EQUAL(rslt1.getQueryId(), "select.c2");
    TRUE(rslt1.isSuccess());
    EQUAL(rslt1.getResultCount(), 1);
    POSITIVE(rslt1.getSpentTimeMillis());
    NULL(rslt1.getException());
    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    TRUE(rslt1.getContext().hasResultTable());
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt1.getResultTable().recordCount(), 1);
    EQUAL(rslt1.getResultTable().columnCount(), 1);
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");

    EQUAL(rslt2.getQueryId(), "update.c2");
    TRUE(rslt2.isSuccess());
    EQUAL(rslt2.getResultCount(), 1);
    POSITIVE(rslt2.getSpentTimeMillis());
    NULL(rslt2.getException());
    EQUAL(rslt2.getContext().getQueryId(), "update.c2");
    FALSE(rslt2.getContext().hasResultTable());
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    NULL(rslt2.getResultTable());

    EQUAL(rslt3.getQueryId(), "select.c2");
    TRUE(rslt3.isSuccess());
    EQUAL(rslt3.getResultCount(), 1);
    POSITIVE(rslt3.getSpentTimeMillis());
    NULL(rslt3.getException());
    EQUAL(rslt3.getContext().getQueryId(), "select.c2");
    TRUE(rslt3.getContext().hasResultTable());
    EQUAL(rslt3.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt3.getResultTable().recordCount(), 1);
    EQUAL(rslt3.getResultTable().columnCount(), 1);
    EQUAL(rslt3.getResultTable().recordFirst().get("Value"), "ＸＹＺ");

    rollbackDataFile();
  }

  public void execute_queryIdArray_inputMap_resultLst_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);

      try {
        query.execute((String[]) null, inputMap, rsltLst);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      try {
        query.execute(new String[]{"select.c1","update.c2"}, null, rsltLst);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      try {
        query.execute(new String[]{"select.c1","update.c2"}, inputMap, null);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }
  }

  public void execute_queryIdArray_inputMap_resultLst_Empty()
  {
    MSG("クエリIDの配列が空の場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(new String[0], inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    TRUE(rsltLst.isSuccess());
    EQUAL(rsltLst.countResults(), 0);
    EQUAL(rsltLst.getSpentTimeMillis(), 0);
  }

  public void execute_queryIdArray_inputMap_resultLst_Error()
  {
    MSG("途中でエラーが発生した場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory2());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(new String[]{"select.c2","update.c2","select.c2"},
        inputMap, rsltLst);
      NG();
      tr.commit();
    }
    catch (ReasonedException e) {
      tr.rollback();
      EQUAL(e.getReason(), XmlFileQuery2.Error.Test);
    }
    catch (Exception e) {
      NG(e);
      tr.rollback();
    }
    finally {
      tr.end();
    }

    FALSE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    QueryResult rslt1 = rsltLst.getResultAt(0);
    TRUE(rslt1.isSuccess());
    POSITIVE(rslt1.getSpentTimeMillis());
    EQUAL(rslt1.getResultCount(), 1);
    NULL(rslt1.getException());
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");

    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    TRUE(rslt1.getContext().hasResultTable());

    QueryResult rslt2 = rsltLst.getResultAt(1);
    FALSE(rslt2.isSuccess());
    POSITIVE(rslt2.getSpentTimeMillis());
    EQUAL(rslt2.getResultCount(), 0);
    EQUAL(ReasonedException.class.cast(rslt2.getException()).getReason(),
      XmlFileQuery2.Error.Test);
    NULL(rslt2.getResultTable());

    EQUAL(rslt2.getContext().getConnectionId(), "c0");
    EQUAL(rslt2.getContext().getQueryId(), "update.c2");
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    FALSE(rslt2.getContext().hasResultTable());
  }

  public void execute_queryIdArray_inputMap_resultLst_Timeout()
  {
    MSG("途中でタイムアウトが発生した場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin(1500L);
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(new String[]{"select.c2","update.c2","select.c2"},
        inputMap, rsltLst);
      NG();
      tr.commit();
    }
    catch (ReasonedException e) {
      tr.rollback();
      EQUAL(e.getReason(), Query.Error.Timeout);
    }
    catch (Exception e) {
      NG(e);
      tr.rollback();
    }
    finally {
      tr.end();
    }

    FALSE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    QueryResult rslt1 = rsltLst.getResultAt(0);
    TRUE(rslt1.isSuccess());
    POSITIVE(rslt1.getSpentTimeMillis());
    EQUAL(rslt1.getResultCount(), 1);
    NULL(rslt1.getException());
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");

    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    TRUE(rslt1.getContext().hasResultTable());

    QueryResult rslt2 = rsltLst.getResultAt(1);
    FALSE(rslt2.isSuccess());
    POSITIVE(rslt2.getSpentTimeMillis());
    EQUAL(rslt2.getResultCount(), 0);
    EQUAL(ReasonedException.class.cast(rslt2.getException()).getReason(),
      Query.Error.Timeout);
    NULL(rslt2.getResultTable());

    EQUAL(rslt2.getContext().getConnectionId(), "c0");
    EQUAL(rslt2.getContext().getQueryId(), "update.c2");
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    FALSE(rslt2.getContext().hasResultTable());
  }

  public void execute_queryIdList_inputMap_resultLst()
  {
    MSG("リストに格納された複数のクエリを実行するメソッドの確認。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      List<String> qidL = new ArrayList<String>(3);
      qidL.add("select.c2");
      qidL.add("update.c2");
      qidL.add("select.c2");
      query.execute(qidL, inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    TRUE(rsltLst.isSuccess());
    EQUAL(rsltLst.countResults(), 3);
    POSITIVE(rsltLst.getSpentTimeMillis());

    QueryResult rslt1 = rsltLst.getFirstResult("select.c2");
    QueryResult rslt2 = rsltLst.getFirstResult("update.c2");
    QueryResult rslt3 = rsltLst.getLastResult("select.c2");
    EQUAL(rsltLst.getResults("select.c2").size(), 2);
    EQUAL(rsltLst.getResults("update.c2").size(), 1);
    NOTEQUAL(rslt1, rslt2);
    NOTEQUAL(rslt1, rslt3);
    NOTEQUAL(rslt2, rslt3);

    EQUAL(rslt1.getQueryId(), "select.c2");
    TRUE(rslt1.isSuccess());
    EQUAL(rslt1.getResultCount(), 1);
    POSITIVE(rslt1.getSpentTimeMillis());
    NULL(rslt1.getException());
    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    TRUE(rslt1.getContext().hasResultTable());
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt1.getResultTable().recordCount(), 1);
    EQUAL(rslt1.getResultTable().columnCount(), 1);
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");

    EQUAL(rslt2.getQueryId(), "update.c2");
    TRUE(rslt2.isSuccess());
    EQUAL(rslt2.getResultCount(), 1);
    POSITIVE(rslt2.getSpentTimeMillis());
    NULL(rslt2.getException());
    EQUAL(rslt2.getContext().getQueryId(), "update.c2");
    FALSE(rslt2.getContext().hasResultTable());
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    NULL(rslt2.getResultTable());

    EQUAL(rslt3.getQueryId(), "select.c2");
    TRUE(rslt3.isSuccess());
    EQUAL(rslt3.getResultCount(), 1);
    POSITIVE(rslt3.getSpentTimeMillis());
    NULL(rslt3.getException());
    EQUAL(rslt3.getContext().getQueryId(), "select.c2");
    TRUE(rslt3.getContext().hasResultTable());
    EQUAL(rslt3.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt3.getResultTable().recordCount(), 1);
    EQUAL(rslt3.getResultTable().columnCount(), 1);
    EQUAL(rslt3.getResultTable().recordFirst().get("Value"), "ＸＹＺ");

    rollbackDataFile();
  }

  public void execute_queryIdList_inputMap_resultLst_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    List<String> qidL = new ArrayList<String>(3);
    qidL.add("select.c2");
    qidL.add("update.c2");
    qidL.add("select.c2");

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);

      try {
        query.execute((List<String>) null, inputMap, rsltLst);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      try {
        query.execute(qidL, null, rsltLst);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      try {
        query.execute(qidL, inputMap, null);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }
  }

  public void execute_queryIdList_inputMap_resultLst_Empty()
  {
    MSG("クエリIDのリストが空の場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(new ArrayList<String>(0), inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    TRUE(rsltLst.isSuccess());
    EQUAL(rsltLst.countResults(), 0);
    EQUAL(rsltLst.getSpentTimeMillis(), 0);
  }

  public void execute_queryIdList_inputMap_resultLst_Error()
  {
    MSG("途中でエラーが発生した場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory2());

    List<String> qidL = new ArrayList<String>(3);
    qidL.add("select.c2");
    qidL.add("update.c2");
    qidL.add("select.c2");

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(qidL, inputMap, rsltLst);
      NG();
      tr.commit();
    }
    catch (ReasonedException e) {
      tr.rollback();
      EQUAL(e.getReason(), XmlFileQuery2.Error.Test);
    }
    catch (Exception e) {
      NG(e);
      tr.rollback();
    }
    finally {
      tr.end();
    }

    FALSE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    QueryResult rslt1 = rsltLst.getResultAt(0);
    TRUE(rslt1.isSuccess());
    POSITIVE(rslt1.getSpentTimeMillis());
    EQUAL(rslt1.getResultCount(), 1);
    NULL(rslt1.getException());
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");

    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    TRUE(rslt1.getContext().hasResultTable());

    QueryResult rslt2 = rsltLst.getResultAt(1);
    FALSE(rslt2.isSuccess());
    POSITIVE(rslt2.getSpentTimeMillis());
    EQUAL(rslt2.getResultCount(), 0);
    EQUAL(ReasonedException.class.cast(rslt2.getException()).getReason(),
      XmlFileQuery2.Error.Test);
    NULL(rslt2.getResultTable());

    EQUAL(rslt2.getContext().getConnectionId(), "c0");
    EQUAL(rslt2.getContext().getQueryId(), "update.c2");
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    FALSE(rslt2.getContext().hasResultTable());
  }

  public void execute_queryIdList_inputMap_resultLst_Timeout()
  {
    MSG("途中でタイムアウトが発生した場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    List<String> qidL = new ArrayList<String>(3);
    qidL.add("select.c2");
    qidL.add("update.c2");
    qidL.add("select.c2");

    try {
      tr.begin(1500L);
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(qidL, inputMap, rsltLst);
      NG();
      tr.commit();
    }
    catch (ReasonedException e) {
      tr.rollback();
      EQUAL(e.getReason(), Query.Error.Timeout);
    }
    catch (Exception e) {
      NG(e);
      tr.rollback();
    }
    finally {
      tr.end();
    }

    FALSE(rsltLst.isSuccess());
    POSITIVE(rsltLst.getSpentTimeMillis());
    EQUAL(rsltLst.countResults(), 2);

    QueryResult rslt1 = rsltLst.getResultAt(0);
    TRUE(rslt1.isSuccess());
    POSITIVE(rslt1.getSpentTimeMillis());
    EQUAL(rslt1.getResultCount(), 1);
    NULL(rslt1.getException());
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");

    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    TRUE(rslt1.getContext().hasResultTable());

    QueryResult rslt2 = rsltLst.getResultAt(1);
    FALSE(rslt2.isSuccess());
    POSITIVE(rslt2.getSpentTimeMillis());
    EQUAL(rslt2.getResultCount(), 0);
    EQUAL(ReasonedException.class.cast(rslt2.getException()).getReason(),
      Query.Error.Timeout);
    NULL(rslt2.getResultTable());

    EQUAL(rslt2.getContext().getConnectionId(), "c0");
    EQUAL(rslt2.getContext().getQueryId(), "update.c2");
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    FALSE(rslt2.getContext().hasResultTable());
  }

  public void execute_inputMap_resultLst()
  {
    MSG("リソース内の全てのクエリを実行するメソッドの確認。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    TRUE(rsltLst.isSuccess());
    EQUAL(rsltLst.countResults(), 8);
    POSITIVE(rsltLst.getSpentTimeMillis());

    QueryResult rslt0 = rsltLst.getResultAt(0);
    QueryResult rslt1 = rsltLst.getResultAt(1);
    QueryResult rslt2 = rsltLst.getResultAt(2);
    QueryResult rslt3 = rsltLst.getResultAt(3);
    QueryResult rslt4 = rsltLst.getResultAt(4);
    QueryResult rslt5 = rsltLst.getResultAt(5);
    QueryResult rslt6 = rsltLst.getResultAt(6);
    QueryResult rslt7 = rsltLst.getResultAt(7);

    EQUAL(rslt0.getQueryId(), "select.c1");
    EQUAL(rslt1.getQueryId(), "select.c2");
    EQUAL(rslt2.getQueryId(), "select.d1");
    EQUAL(rslt3.getQueryId(), "select.d2");
    EQUAL(rslt4.getQueryId(), "update.c1");
    EQUAL(rslt5.getQueryId(), "update.c2");
    EQUAL(rslt6.getQueryId(), "update.d1");
    EQUAL(rslt7.getQueryId(), "update.d2");
    TRUE(rslt0.isSuccess());
    TRUE(rslt1.isSuccess());
    TRUE(rslt2.isSuccess());
    TRUE(rslt3.isSuccess());
    TRUE(rslt4.isSuccess());
    TRUE(rslt5.isSuccess());
    TRUE(rslt6.isSuccess());
    TRUE(rslt7.isSuccess());
    EQUAL(rslt0.getResultCount(), 1);
    EQUAL(rslt1.getResultCount(), 1);
    EQUAL(rslt2.getResultCount(), 1);
    EQUAL(rslt3.getResultCount(), 2);
    EQUAL(rslt4.getResultCount(), 1);
    EQUAL(rslt5.getResultCount(), 1);
    EQUAL(rslt6.getResultCount(), 1);
    EQUAL(rslt7.getResultCount(), 1);
    POSITIVE(rslt0.getSpentTimeMillis());
    POSITIVE(rslt1.getSpentTimeMillis());
    POSITIVE(rslt2.getSpentTimeMillis());
    POSITIVE(rslt3.getSpentTimeMillis());
    POSITIVE(rslt4.getSpentTimeMillis());
    POSITIVE(rslt5.getSpentTimeMillis());
    POSITIVE(rslt6.getSpentTimeMillis());
    POSITIVE(rslt7.getSpentTimeMillis());
    NULL(rslt0.getException());
    NULL(rslt1.getException());
    NULL(rslt2.getException());
    NULL(rslt3.getException());
    NULL(rslt4.getException());
    NULL(rslt5.getException());
    NULL(rslt6.getException());
    NULL(rslt7.getException());
    EQUAL(rslt0.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt2.getContext().getConnectionId(), "c0");
    EQUAL(rslt3.getContext().getConnectionId(), "c0");
    EQUAL(rslt4.getContext().getConnectionId(), "c0");
    EQUAL(rslt5.getContext().getConnectionId(), "c0");
    EQUAL(rslt6.getContext().getConnectionId(), "c0");
    EQUAL(rslt7.getContext().getConnectionId(), "c0");
    EQUAL(rslt0.getContext().getQueryId(), "select.c1");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    EQUAL(rslt2.getContext().getQueryId(), "select.d1");
    EQUAL(rslt3.getContext().getQueryId(), "select.d2");
    EQUAL(rslt4.getContext().getQueryId(), "update.c1");
    EQUAL(rslt5.getContext().getQueryId(), "update.c2");
    EQUAL(rslt6.getContext().getQueryId(), "update.d1");
    EQUAL(rslt7.getContext().getQueryId(), "update.d2");
    TRUE(rslt0.getContext().hasResultTable());
    TRUE(rslt1.getContext().hasResultTable());
    TRUE(rslt2.getContext().hasResultTable());
    TRUE(rslt3.getContext().hasResultTable());
    FALSE(rslt4.getContext().hasResultTable());
    FALSE(rslt5.getContext().hasResultTable());
    FALSE(rslt6.getContext().hasResultTable());
    FALSE(rslt7.getContext().hasResultTable());
    EQUAL(rslt0.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).DDD");
    EQUAL(rslt3.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).DDD");
    EQUAL(rslt4.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    EQUAL(rslt5.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt6.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).DDD");
    EQUAL(rslt7.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).DDD");
    EQUAL(rslt0.getResultTable().recordCount(), 1);
    EQUAL(rslt0.getResultTable().columnCount(), 1);
    EQUAL(rslt1.getResultTable().recordCount(), 1);
    EQUAL(rslt1.getResultTable().columnCount(), 1);
    EQUAL(rslt2.getResultTable().recordCount(), 1);
    EQUAL(rslt2.getResultTable().columnCount(), 1);
    EQUAL(rslt3.getResultTable().recordCount(), 2);
    EQUAL(rslt3.getResultTable().columnCount(), 1);
    NULL(rslt4.getResultTable());
    NULL(rslt5.getResultTable());
    NULL(rslt6.getResultTable());
    NULL(rslt7.getResultTable());
    EQUAL(rslt0.getResultTable().recordFirst().get("Value"), "あいうえお");
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");
    EQUAL(rslt2.getResultTable().recordFirst().get("Value"), "１２３４");
    EQUAL(rslt3.getResultTable().recordFirst().get("Value"), "５６７８");

    try {
      XmlResource resData = new XmlResource();
      resData.setValidating(false);
      resData.load(QueryConfig.getInstance().getString("sample.dat.path"));

      EQUAL(resData.getFirstValue("AAA.BBB(id=b1).CCC(no=0)"), "ＸＹＺ");
      EQUAL(resData.getFirstValue("AAA.BBB(id=b2).CCC(no=0)"), "ＸＹＺ");
      EQUAL(resData.getFirstValue("AAA.BBB(id=b1).DDD(no=0)"), "ＸＹＺ");
      EQUAL(resData.getFirstValue("AAA.BBB(id=b2).DDD(no=0)"), "ＸＹＺ");
      EQUAL(resData.getFirstValue("AAA.BBB(id=b1).DDD"), "１２３４");
      EQUAL(resData.getFirstValue("AAA.BBB(id=b2).DDD"), "５６７８");
    } catch (Exception e) {
      NG(e);
    }

    rollbackDataFile();
  }

  public void execute_inputMap_resultLst_Null()
  {
    MSG("引数がヌルの場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);

      try {
        query.execute(null, rsltLst);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      try {
        query.execute(inputMap, null);
        NG();
      } catch (AssertionError ae) {
        OK(ae);
      }

      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }
  }

  public void execute_inputMap_resultLst_Error()
  {
    MSG("途中でエラーが発生した場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory2());

    try {
      tr.begin();
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    FALSE(rsltLst.isSuccess());
    EQUAL(rsltLst.countResults(), 5);
    POSITIVE(rsltLst.getSpentTimeMillis());

    QueryResult rslt0 = rsltLst.getResultAt(0);
    QueryResult rslt1 = rsltLst.getResultAt(1);
    QueryResult rslt2 = rsltLst.getResultAt(2);
    QueryResult rslt3 = rsltLst.getResultAt(3);
    QueryResult rslt4 = rsltLst.getResultAt(4);

    EQUAL(rslt0.getQueryId(), "select.c1");
    EQUAL(rslt1.getQueryId(), "select.c2");
    EQUAL(rslt2.getQueryId(), "select.d1");
    EQUAL(rslt3.getQueryId(), "select.d2");
    EQUAL(rslt4.getQueryId(), "update.c1");
    TRUE(rslt0.isSuccess());
    TRUE(rslt1.isSuccess());
    TRUE(rslt2.isSuccess());
    TRUE(rslt3.isSuccess());
    FALSE(rslt4.isSuccess());
    EQUAL(rslt0.getResultCount(), 1);
    EQUAL(rslt1.getResultCount(), 1);
    EQUAL(rslt2.getResultCount(), 1);
    EQUAL(rslt3.getResultCount(), 2);
    EQUAL(rslt4.getResultCount(), 0);
    POSITIVE(rslt0.getSpentTimeMillis());
    POSITIVE(rslt1.getSpentTimeMillis());
    POSITIVE(rslt2.getSpentTimeMillis());
    POSITIVE(rslt3.getSpentTimeMillis());
    POSITIVE(rslt4.getSpentTimeMillis());
    NULL(rslt0.getException());
    NULL(rslt1.getException());
    NULL(rslt2.getException());
    NULL(rslt3.getException());
    EQUAL(ReasonedException.class.cast(rslt4.getException()).getReason(),
      XmlFileQuery2.Error.Test);
    EQUAL(rslt0.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt2.getContext().getConnectionId(), "c0");
    EQUAL(rslt3.getContext().getConnectionId(), "c0");
    EQUAL(rslt4.getContext().getConnectionId(), "c0");
    EQUAL(rslt0.getContext().getQueryId(), "select.c1");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    EQUAL(rslt2.getContext().getQueryId(), "select.d1");
    EQUAL(rslt3.getContext().getQueryId(), "select.d2");
    EQUAL(rslt4.getContext().getQueryId(), "update.c1");
    TRUE(rslt0.getContext().hasResultTable());
    TRUE(rslt1.getContext().hasResultTable());
    TRUE(rslt2.getContext().hasResultTable());
    TRUE(rslt3.getContext().hasResultTable());
    FALSE(rslt4.getContext().hasResultTable());
    EQUAL(rslt0.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).DDD");
    EQUAL(rslt3.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).DDD");
    EQUAL(rslt4.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    EQUAL(rslt0.getResultTable().recordCount(), 1);
    EQUAL(rslt0.getResultTable().columnCount(), 1);
    EQUAL(rslt1.getResultTable().recordCount(), 1);
    EQUAL(rslt1.getResultTable().columnCount(), 1);
    EQUAL(rslt2.getResultTable().recordCount(), 1);
    EQUAL(rslt2.getResultTable().columnCount(), 1);
    EQUAL(rslt3.getResultTable().recordCount(), 2);
    EQUAL(rslt3.getResultTable().columnCount(), 1);
    NULL(rslt4.getResultTable());
    EQUAL(rslt0.getResultTable().recordFirst().get("Value"), "あいうえお");
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");
    EQUAL(rslt2.getResultTable().recordFirst().get("Value"), "１２３４");
    EQUAL(rslt3.getResultTable().recordFirst().get("Value"), "５６７８");

    rollbackDataFile();
  }

  public void execute_inputMap_resultLst_Timeout()
  {
    MSG("途中でタイムアウトが発生した場合。");

    QueryConfig cfg = QueryConfig.getInstance();

    PropertyResource res = new PropertyResource();
    try {
      res.load(cfg.getString("sample.res.path"));
    } catch (Exception e) {
      NG(e);
    }

    Map<String,Object> inputMap = new HashMap<String,Object>();
    inputMap.put("Value", "ＸＹＺ");

    QueryResultList rsltLst = new QueryResultList();

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    tr.addConnectionFactory("c0", new XmlFileQueryConnectionFactory());

    try {
      tr.begin(4500L);
      QueryConnection conn = tr.getConnection("c0");
      Query query = conn.getQuery(res);
      query.execute(inputMap, rsltLst);
      tr.commit();
    }
    catch (Exception e) {
      tr.rollback();
    }
    finally {
      tr.end();
    }

    FALSE(rsltLst.isSuccess());
    EQUAL(rsltLst.countResults(), 5);
    POSITIVE(rsltLst.getSpentTimeMillis());

    QueryResult rslt0 = rsltLst.getResultAt(0);
    QueryResult rslt1 = rsltLst.getResultAt(1);
    QueryResult rslt2 = rsltLst.getResultAt(2);
    QueryResult rslt3 = rsltLst.getResultAt(3);
    QueryResult rslt4 = rsltLst.getResultAt(4);

    EQUAL(rslt0.getQueryId(), "select.c1");
    EQUAL(rslt1.getQueryId(), "select.c2");
    EQUAL(rslt2.getQueryId(), "select.d1");
    EQUAL(rslt3.getQueryId(), "select.d2");
    EQUAL(rslt4.getQueryId(), "update.c1");
    TRUE(rslt0.isSuccess());
    TRUE(rslt1.isSuccess());
    TRUE(rslt2.isSuccess());
    TRUE(rslt3.isSuccess());
    FALSE(rslt4.isSuccess());
    EQUAL(rslt0.getResultCount(), 1);
    EQUAL(rslt1.getResultCount(), 1);
    EQUAL(rslt2.getResultCount(), 1);
    EQUAL(rslt3.getResultCount(), 2);
    EQUAL(rslt4.getResultCount(), 0);
    POSITIVE(rslt0.getSpentTimeMillis());
    POSITIVE(rslt1.getSpentTimeMillis());
    POSITIVE(rslt2.getSpentTimeMillis());
    POSITIVE(rslt3.getSpentTimeMillis());
    POSITIVE(rslt4.getSpentTimeMillis());
    NULL(rslt0.getException());
    NULL(rslt1.getException());
    NULL(rslt2.getException());
    NULL(rslt3.getException());
    EQUAL(ReasonedException.class.cast(rslt4.getException()).getReason(),
      Query.Error.Timeout);
    EQUAL(rslt0.getContext().getConnectionId(), "c0");
    EQUAL(rslt1.getContext().getConnectionId(), "c0");
    EQUAL(rslt2.getContext().getConnectionId(), "c0");
    EQUAL(rslt3.getContext().getConnectionId(), "c0");
    EQUAL(rslt4.getContext().getConnectionId(), "c0");
    EQUAL(rslt0.getContext().getQueryId(), "select.c1");
    EQUAL(rslt1.getContext().getQueryId(), "select.c2");
    EQUAL(rslt2.getContext().getQueryId(), "select.d1");
    EQUAL(rslt3.getContext().getQueryId(), "select.d2");
    EQUAL(rslt4.getContext().getQueryId(), "update.c1");
    TRUE(rslt0.getContext().hasResultTable());
    TRUE(rslt1.getContext().hasResultTable());
    TRUE(rslt2.getContext().hasResultTable());
    TRUE(rslt3.getContext().hasResultTable());
    FALSE(rslt4.getContext().hasResultTable());
    EQUAL(rslt0.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    EQUAL(rslt1.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).CCC");
    EQUAL(rslt2.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).DDD");
    EQUAL(rslt3.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b2).DDD");
    EQUAL(rslt4.getContext().get(XmlFileQuery.Attribute.NodePath),
      "AAA.BBB(id=b1).CCC");
    EQUAL(rslt0.getResultTable().recordCount(), 1);
    EQUAL(rslt0.getResultTable().columnCount(), 1);
    EQUAL(rslt1.getResultTable().recordCount(), 1);
    EQUAL(rslt1.getResultTable().columnCount(), 1);
    EQUAL(rslt2.getResultTable().recordCount(), 1);
    EQUAL(rslt2.getResultTable().columnCount(), 1);
    EQUAL(rslt3.getResultTable().recordCount(), 2);
    EQUAL(rslt3.getResultTable().columnCount(), 1);
    NULL(rslt4.getResultTable());
    EQUAL(rslt0.getResultTable().recordFirst().get("Value"), "あいうえお");
    EQUAL(rslt1.getResultTable().recordFirst().get("Value"), "かきくけこ");
    EQUAL(rslt2.getResultTable().recordFirst().get("Value"), "１２３４");
    EQUAL(rslt3.getResultTable().recordFirst().get("Value"), "５６７８");

    rollbackDataFile();
  }
}
