/*
 * DefaultQueryTransaction class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.ReasonedRuntimeException;
import java.util.Map;
import java.util.LinkedHashMap;

/**
 * ひとまとまりの処理を表すトランザクション・クラスのインターフェイス。
 * <br>
 * このクラスの{@link #begin()}メソッドの実行から{@link #end()}メソッドの実行
 * までを一つのトランザクションとして、その間に実行されたクエリによる各接続先
 * への更新の確定処理または取消処理をまとめて実行する。
 *
 * @author 佐藤隆之
 * @version $Id: DefaultQueryTransaction.java,v 1.2 2011-08-03 13:42:07 tayu Exp $
 */
public class DefaultQueryTransaction implements QueryTransaction
{
  /** トランザクションの状態。*/
  private State state;

  /** トランザクションの開始時刻[msec]。*/
  private long beginTimeMillis = -1L;

  /** トランザクションのタイムアウトのリミット時刻[msec]。*/
  private long limitTimeMillis = -1L;

  /** トランザクションのデフォルトのタイムアウト時間[msec]。 */
  private long timeoutMillis = -1L;
  
  /** コネクション・オブジェクトを格納するマップ。 */
  private Map<String,QueryConnection> connectionMap;

  /** コネクション・ファクトリ・オブジェクトを格納するマップ。 */
  private final Map<String,QueryConnectionFactory> connectionFactoryMap
    = new LinkedHashMap<String,QueryConnectionFactory>();

  /**
   * デフォルト・コンストラクタ。
   */
  protected DefaultQueryTransaction()
  {
    this.state = State.CREATED;
    this.connectionMap = new LinkedHashMap<String,QueryConnection>();
  }

  /**
   * トランザクションの開始処理を実行する。
   * <br>
   * トランザクションのタイムアウト時間は、このオブジェクトに設定されている
   * デフォルトのタイムアウト時間を使用する。デフォルトのタイムアウト時間が
   * 負の値をとる場合は、タイムアウト制御は行わない。
   */
  @Override
  public void begin()
  {
    begin(this.timeoutMillis);
  }

  /**
   * タイムアウト時間を指定して、トランザクションの開始処理を実行する。
   * <br>
   * このオブジェクトが保持するデフォルトのタイムアウト時間は変更しない。
   * 引数に負値を指定した場合は、タイムアウト時間の指定がないものとする。
   *
   * @param timeoutMillis タイムアウト時間[msec]。
   */
  @Override
  public void begin(long timeoutMillis)
  {
    changeState(new State[]{ State.CREATED }, State.BEGINING);
    
    this.beginTimeMillis = System.currentTimeMillis();

    if (timeoutMillis >= 0) {
      this.limitTimeMillis = this.beginTimeMillis + timeoutMillis;
    }
    else {
      this.limitTimeMillis = -1L;
    }

    this.state = State.BEGINED;
  }

  /**
   * クエリの実行結果の確定処理を実行する。
   */
  @Override
  public void commit()
  {
    changeState(new State[]{ State.BEGINED }, State.COMMITTING);

    for (QueryConnection conn : this.connectionMap.values()) {
      conn.commit();
    }

    this.state = State.COMMITTED;
  }

  /**
   * クエリの実行でエラーが発生した場合に、トランザクション内のクエリ実行結果
   * を取り消す処理を実行する。
   */
  @Override
  public void rollback()
  {
    changeState(new State[]{State.BEGINING,State.BEGINED,State.COMMITTING},
      State.ROLLBACKING);

    for (QueryConnection conn : this.connectionMap.values()) {
      conn.rollback();
    }

    this.state = State.ROLLBACKED;
  }

  /**
   * トランザクションを終了する。
   */
  @Override
  public void end()
  {
    State prevState = this.state;
    this.state = State.ENDING;

    Map<String,QueryConnection> prevMap = this.connectionMap;
    this.connectionMap = new LinkedHashMap<String,QueryConnection>();

    if (prevState == State.BEGINED) {
      for (QueryConnection conn : prevMap.values()) {
        conn.rollback();
      }
    }

    for (QueryConnection conn : prevMap.values()) {
      conn.close();
    }

    prevMap.clear();

    this.state = State.ENDED;
  }

  /**
   * デフォルトのトランザクション・タイムアウト値を取得する。
   *
   * @return デフォルトのトランザクション・タイムアウト値[msec]。
   */
  protected long getTimeoutMillis()
  {
    return this.timeoutMillis;
  }

  /**
   * デフォルトのトランザクション・タイムアウト値を設定する。
   *
   * @param timeout デフォルトのトランザクション・タイムアウト値[msec]。
   */
  public void setTimeoutMillis(long timeout)
  {
    checkState(new State[]{ State.CREATED });
    this.timeoutMillis = timeout;
  }

  /**
   * トランザクションの開始時刻を取得する。
   *
   * @return トランザクションの開始時刻[msec]。
   */
  @Override
  public long getBeginTimeMillis()
  {
    return this.beginTimeMillis;
  }

  /**
   * トランザクションのタイムアウトのリミット時刻を取得する。
   *
   * @return トランザクションのタイムアウトのリミット時刻[msec]。
   */
  @Override
  public long getLimitTimeMillis()
  {
    return this.limitTimeMillis;
  }

  /**
   * トランザクションの状態を取得する。
   *
   * @return トランザクションの状態。
   */
  @Override
  public State getState()
  {
    return this.state;
  }

  /**
   * トランザクションの状態を変更する。
   * <br>
   * 現在の状態が、第一引数に指定された状態に含まれていない場合は、例外をスロー
   * する。
   *
   * @param froms 許される現在の状態。
   * @param to 変更後の状態。
   */
  protected final void changeState(State[] froms, State to)
  {
    checkState(froms);
    this.state = to;
  }

  /**
   * 現在の状態が、指定された状態に含まれているかどうかを判定する。
   *
   * @param allows 許される現在の状態。
   */
  protected final void checkState(State[] allows)
  {
    for (State st : allows) {
      if (st == this.state)
        return;
    }

    throw new ReasonedRuntimeException(Error.IllegalState, this.state.name());
  }

  /**
   * 指定されたIDに対応づけられたコネクション・オブジェクトを取得する。
   *
   * @param connId 接続先を示すID。
   * @return コネクション・オブジェクト。
   * @throws ReasonedException コネクション・オブジェクトの作成に失敗した場合。
   */
  @Override
  public QueryConnection getConnection(String connId) throws ReasonedException
  {
    checkState(new State[]{ State.BEGINED });

    QueryConnection conn = this.connectionMap.get(connId);
    if (conn == null) {
      QueryConnectionFactory factory = this.connectionFactoryMap.get(connId);
      if (factory == null)
        throw new ReasonedException(Error.ConnectionNotFound, connId);

      conn = factory.create(connId, QueryConfig.getInstance());
      conn.setLimitTimeMillis(this.limitTimeMillis);
      conn.open();
      this.connectionMap.put(connId, conn);
    }

    return conn;
  }
  
  /**
   * 指定されたIDに対応付けられるコネクションのファクトリ・オブジェクトを
   * 設定する。
   *
   * @param connId コネクションを識別するためのID。
   * @param factory コネクションのファクトリ・オブジェクト。
   */
  @Override
  public void addConnectionFactory(
    String connId, QueryConnectionFactory factory)
  {
    assert (connId != null && factory != null) : (connId == null) ?
      "@param:connId is null." : "@param:factory is null.";

    checkState(new State[]{ State.CREATED });

    this.connectionFactoryMap.put(connId, factory);
  }
}
