/*
 * Query class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import java.util.List;
import java.util.Map;

/**
 * クエリを実行するクラスのインターフェイス。
 * <br>
 * 指定された{@link QueryConnection}オブジェクトが示す接続先に対して、{@link
 * ts.util.resource.Resource}オブジェクトと入力パラメータ・マップから作成した
 * 実行内容を実行する。 
 * クエリの実行結果は{@link QueryResult}オブジェクトを作成して格納し、引数として
 * 渡された{@link QueryResultList}オブジェクトに追加する。
 * <br>
 * クエリの実行中にエラーが発生した場合は例外をスローするが、その前に実行結果
 * オブジェクトを作成して{@link QueryResultList}オブジェクトに追加する。
 * <br>
 * <br>
 * クエリを実行する手順のイメージは以下のようになる：
 * <tt><pre>
 *     transaction : QueryTransaction
 *     connAFactory, connBFactory : QueryConnectionFactory
 *     resourceA0, resourceA1, resourceB0 : Resource
 *
 *     transaction.addConnectionFactory("ConnA", connAFactory);
 *     transaction.addConnectionFactory("ConnB", connBFactory);
 *     ...
 *
 *     QueryResultList resultLst = new QueryResultList();
 *     try {
 *       transaction.begin();
 *
 *       QueryConnection connA = transaction.getConnection("ConnA");
 *       Query queryA0 = connA.getQuery(resourceA0);
 *       queryA0.execute("qidA00", inputMap, resultLst);
 *       queryA0.execute("qidA01", inputMap, resultLst);
 *
 *       Query queryA1 = connA.getQuery(resourceA1);
 *       queryA1.execute(new String[]{"qidA10","qidA11"}, inputMap, resultLst);
 *
 *       QueryConnection connB = transaction.getConnection("ConnB");
 *       Query queryB0 = connB.getQuery(resourceB0);
 *       queryB0.execute(inputMap, resultLst);
 *
 *       transaction.commit();
 *     }
 *     catch (Exception e) {
 *       transaction.rollback();
 *     }
 *     finally {
 *       transaction.end();
 *     }
 *
 *     QueryResult result0 = resultLst.getResultAt(0);
 *     Table<String,Object> table0 = result0.getResulTable();
 *     ...
 *
 *     QueryResult result1 = resultLst.getFirstResult("qidA01");
 *     int resultCount = result1.getResultCount();
 *     ...
 * </pre></tt>
 *
 * @author 佐藤隆之
 * @version $Id: Query.java,v 1.4 2011-08-09 14:24:18 tayu Exp $
 */
public interface Query
{
  /** このクラスから発生しうるエラーを示す列挙型。 */
  public enum Error {
    /** トランザクション・タイムアウトの場合。 */
    Timeout,

    /** 指定されたクエリIDが見つからなかった場合。 */
    QueryIdNotFound,

    /** クエリ実行時エラーの場合。 */
    FailToExecute,
  }

  /**
   * このオブジェクトが保持するクエリの中から、指定されたIDに対する1つのクエリ
   * を実行する。
   *
   * @param queryId クエリID。
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  void execute(String queryId,
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException;

  /**
   * このオブジェクトが保持するクエリの中から、指定された配列内のIDに対する
   * 複数のクエリを順番に実行する。
   *
   * @param queryIds クエリIDの配列。
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  void execute(String[] queryIds,
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException;

  /**
   * このオブジェクトが保持するクエリの中から、指定されたリスト内のIDに対する
   * 複数のクエリを順番に実行する。
   *
   * @param queryIdLst クエリIDのリスト。
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  void execute(List<String> queryIdLst,
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException;

  /**
   * このオブジェクトが保持する全てのクエリを、クエリ・リソースの中で指定されて
   * いる順番に実行する。
   *
   * @param inputMap 入力パラメータを格納したマップ。
   * @param resultLst 実行結果を格納する{@link QueryResultList}オブジェクト。
   * @throws ReasonedException クエリの実行中に例外が発生した場合。
   */
  void execute(
    Map<String,Object> inputMap, QueryResultList resultLst
  ) throws ReasonedException;
}
