/*
 * QueryConnection class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedException;
import ts.util.resource.Resource;

/**
 * クエリを実行する対象への接続を表すコネクション・クラス。
 * <br>
 * 接続を操作するためのメソッド {@link #open()}、{@link #commit()}、{@link
 * #rollback()}、{@link #close()}を用意している。これらのメソッドは{@link
 * QueryTransaction}オブジェクトから呼び出される。
 * <br>
 * また、このクラスは{@link Query}オブジェクトを取得するためのメソッド{@link
 * #getQuery(Resource)}を用意している。
 * このクラスのインスタンスが示す接続先へのクエリの実行は、この{@link Query}
 * オブジェクトによって行われる。
 * 
 * @author 佐藤隆之
 * @version $Id: QueryConnection.java,v 1.5 2011-07-30 17:08:51 tayu Exp $
 */
public interface QueryConnection
{
  /**
   * 接続先を識別するIDを取得する。
   *
   * @return 接続先を識別するID。
   */
  String getId();

  /**
   * クエリ実行のタイムアウトのリミット時刻をミリ秒単位で設定する。
   * <br>
   * 引数に負値が指定された場合は、リミット時刻が指定されていないものとする。
   *
   * @param limitTimeMillis タイムアウトのリミット時刻[msec]。
   */
  void setLimitTimeMillis(long limitTimeMillis);

  /**
   * クエリ実行のタイムアウトのリミット時刻をミリ秒単位で取得する。
   * <br>
   * 戻り値が負値の場合は、リミット時刻が指定されていないものとする。
   *
   * @return タイムアウトのリミット時刻[msec]。
   */
  long getLimitTimeMillis();

  /**
   * クエリ・オブジェクトを作成する。
   * <br>
   * 引数に指定されたクエリの実行内容についてのリソースとこのコネクション・
   * オブジェクトを使ってクエリを実行する{@link Query}オブジェクトを作成して
   * 返す。
   *
   * @param queryResource クエリ実行内容のリソース。
   * @return {@link Query}オブジェクト。
   */
  Query getQuery(Resource queryResource);

  /**
   * 接続を開始する。
   */
  void open();

  /**
   * 接続先への更新を確定する。
   */
  void commit(); 

  /**
   * 接続先への更新を取り消す。
   */
  void rollback();

  /**
   * 接続を終了する。
   */
  void close();
}
