/*
 * QueryResultList class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import java.io.Serializable;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Enumeration;
import java.util.Collections;

/**
 * 複数のクエリ実行結果オブジェクトを格納するためのクラス。
 * <br>
 * 1つのトランザクションの中で複数のクエリが実行される場合などに、その実行結果
 * オブジェクトを実行した順番に格納するために使用される。
 * <br>
 * 各メソッドの戻り値は、格納されている実行結果オブジェクトの内容により決定
 * される。
 * 例えば、{@link #isSuccess()}は格納されている結果オブジェクトの中に一つでも
 * {@link QueryResult#isSuccess()}が<tt>false</tt>を返す場合には<tt>false</tt>
 * を返し、全てが<tt>true</tt>を返す場合にだけ<tt>true</tt>を返す。
 *
 * @author 佐藤隆之。
 * @version $Id: QueryResultList.java,v 1.3 2011-08-09 14:24:18 tayu Exp $
 */
public class QueryResultList implements Serializable
{
  /** シリアル・バージョン番号。 */
  static final long serialVersionUID = 7570497809387060417L;

  /** 実行結果オブジェクトを順番に格納するリスト。 */
  private final List<QueryResult> resultLst = new LinkedList<QueryResult>();

  /** 所要時間[msec]。 */
  private long spentTimeMillis = -1L;

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryResultList()
  {}

  /**
   * クエリの実行結果オブジェクトを追加する。
   *
   * @param result 実行結果オブジェクト。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addResult(QueryResult result)
  {
    assert (result != null) : "@param:result is null.";

    this.resultLst.add(result);
  }

  /**
   * 指定されたクエリIDに対する実行結果オブジェクトのリストを取得する。
   * <br>
   * 該当する実行結果オブジェクトが存在しない場合は空のリストを返す。
   *
   * @param queryId クエリID。
   * @return 指定されたクエリIDに対する実行結果オブジェクトのリスト。
   */
  public List<QueryResult> getResults(String queryId)
  {
    List<QueryResult> lst = new LinkedList<QueryResult>();
    for (QueryResult rslt : this.resultLst) {
      if (rslt.getQueryId().equals(queryId))
        lst.add(rslt);
    }
    return lst;
  }

  /**
   * 指定されたクエリIDに対する最初の実行結果オブジェクトを取得する。
   * <br>
   * 該当する実行結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @param queryId クエリID。
   * @return 指定されたクエリIDに対する最初の実行結果オブジェクト。
   */
  public QueryResult getFirstResult(String queryId)
  {
    for (QueryResult rslt : this.resultLst) {
      if (rslt.getQueryId().equals(queryId))
        return rslt;
    }
    return null;
  }

  /**
   * 指定されたクエリIDに対する最後の実行結果オブジェクトを取得する。
   * <br>
   * 該当する実行結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @param queryId クエリID。
   * @return 指定されたクエリIDに対する最後の実行結果オブジェクト。
   */
  public QueryResult getLastResult(String queryId)
  {
    for (int i=this.resultLst.size()-1; i>=0; i--) {
      QueryResult rslt = this.resultLst.get(i);
      if (rslt.getQueryId().equals(queryId))
        return rslt;
    }
    return null;
  }

  /**
   * 指定されたインデックスに格納されているの実行結果オブジェクトを取得する。
   *
   * @param index インデックス。
   * @return 指定されたクエリIDの実行結果オブジェクト。
   * @throws IndexOutOfBoundsException インデックスが範囲外の場合。
   */
  public QueryResult getResultAt(int index)
  {
    return this.resultLst.get(index);
  }

  /**
   * このオブジェクトに格納されている実行結果オブジェクトの数を取得する。
   *
   * @return このオブジェクトに格納されている実行結果オブジェクトの数。
   */
  public int countResults()
  {
    return this.resultLst.size();
  }

  /**
   * このオブジェクトに格納されている全ての実行結果オブジェクトのリストを
   * 取得する。
   *
   * @return 実行結果オブジェクトを格納した列挙オブジェクト。
   */
  public List<QueryResult> getAllResults()
  {
    return Collections.unmodifiableList(this.resultLst);
  }

  /**
   * このオブジェクトに格納されている最初の実行結果オブジェクトを取得する。
   * <br>
   * 格納されている実行結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @return このオブジェクトに格納されている最初の実行結果オブジェクト。
   */
  public QueryResult getFirstResult()
  {
    if (this.resultLst.size() > 0) {
      return this.resultLst.get(0);
    }
    return null;
  }

  /**
   * このオブジェクトに格納されている最後の実行結果オブジェクトを取得する。
   * <br>
   * 格納されている実行結果オブジェクトが存在しない場合はヌルを返す。
   *
   * @return このオブジェクトに格納されている最後の実行結果オブジェクト。
   */
  public QueryResult getLastResult()
  {
    if (this.resultLst.size() > 0) {
      return this.resultLst.get(this.resultLst.size()-1);
    }
    return null;
  }

  /**
   * 複数のクエリが全て成功したかどうかを取得する。
   * <br>
   * このオブジェクトに格納されている複数のクエリ結果が全て成功かどうかを調べ
   * る。
   * このオブジェクトに一つでも失敗のクエリ結果が含まれている場合は
   * <tt>false</tt>を返す。
   * <br>
   * クエリ結果が一つも格納されていない場合は<tt>true</tt>を返す。
   *
   */
  public boolean isSuccess()
  {
    for (QueryResult rslt : this.resultLst) {
      if (! rslt.isSuccess())
        return false;
    }
    return true;
  }
  
  /**
   * 複数のクエリを実行するのに要した時間を取得する。
   * <br>
   * このオブジェクトに格納されている複数のクエリ結果の実行時間の合計を求めて
   * 返す。
   * <br>
   * 但し{@link #setSpentTimeMillis(long)}メソッドを使って処理時間を設定した
   * 場合は、その値を返す。
   * <br>
   * 時間の単位はミリ秒である。
   *
   * @return 複数のクエリを実行するのに要した時間。
   */
  public long getSpentTimeMillis()
  {
    if (this.spentTimeMillis >= 0L) {
      return this.spentTimeMillis;
    }
    else {
      long tm = 0L;
      for (QueryResult rslt : this.resultLst) {
        tm += rslt.getSpentTimeMillis();
      }
      return tm;
    }
  }

  /**
   * 複数のクエリを全て実行するのに要した時間を設定する。
   * <br>
   * もし引数に負の値を設定した場合は、例外をスローする。
   *
   * @param millis 複数のクエリを実行するのに要した時間[msec]。
   * @throws IllegalArgumentException 引数が負値の場合。
   */
  public void setSpentTimeMillis(long millis)
  {
    if (millis < 0L)
      throw new IllegalArgumentException("@param:millis is negative.");

    this.spentTimeMillis = millis;
  }
 
  /**
   * 複数のクエリを全て実行するのに要した時間をリセットする。
   * <br>
   * このメソッドを実行することにより、過去に{@link #setSpentTimeMillis(long)}
   * メソッドを使って設定した所要時間をリセットし、このオブジェクトに格納されて
   * いる結果オブジェクトの実行時間の合計を、{@link #getSpentTimeMillis()}
   * メソッドの戻り値として返すようにする。
   */
  public void resetSpentTimeMillis()
  {
    this.spentTimeMillis = -1L;
  }
}
