/*
 * QueryTransactionManager class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.ReasonedRuntimeException;
import ts.util.ReasonedException;
import java.util.Map;
import java.util.HashMap;

/**
 * トランザクションの作成や取得を管理するためのクラス。
 * <br>
 * 指定された又はデフォルトのトランザクション・クラスのインスタンスをスレッド・
 * ローカル化又はスレッド・セーフ化して作成したり、それらのインスタンスを取得
 * したりするメソッドを用意している。
 *
 * @author 佐藤隆之
 * @version $Id: QueryTransactionManager.java,v 1.7 2011-08-09 14:24:19 tayu Exp $
 */
public class QueryTransactionManager
{
  /** このクラスで発生しうるエラーの列挙型。 */
  public enum Error {
    /**
     * スレッド・ローカルなトランザクション・オブジェクトの作成に失敗したこと
     * を示す列挙値。
     */
    ThreadLocalFailToCreate,

    /** スレッド・ローカルなトランザクションが存在しないことを示す列挙値。 */
    ThreadLocalNotExist,

    /**
     * スレッド・ローカルなトランザクション作成時に、終了していない別の
     * トランザクションが既に存在することを示す列挙値。
     */
    ThreadLocalAlreadyExists,

    /**
     * スレッド・セーフなトランザクション・オブジェクトの作成に失敗したことを
     * 示す列挙値。
     */
    ThreadSafeFailToCreate,

    /** スレッド・セーフなトランザクションが存在しないことを示す列挙値。 */
    ThreadSafeNotExist,

    /**
     * スレッド・セーフなトランザクション作成時に、終了していない別の
     * トランザクションが既に存在することを示す列挙値。
     */
    ThreadSafeAlreadyExists,
  }

  /**
   * スレッド・ローカルなトランザクション・オブジェクトを格納するための
   * {@link ThreadLocal}オブジェクト。
   */
  private static ThreadLocal<QueryTransaction> threadLocal =
    new ThreadLocal<QueryTransaction>();

  /**
   * スレッド・セーフなトランザクション・オブジェクトを格納するためのマップ・
   * オブジェクト。
   */
  private static Map<String,QueryTransaction> threadMap =
    new HashMap<String,QueryTransaction>();

  /**
   * デフォルト・コンストラクタ。
   * <br>
   * このクラスはインスタンス化して使用することはないため、アクセス指定に
   * <tt>private</tt>を指定している。
   */
  private QueryTransactionManager()
  {}

  /**
   * スレッド・ローカルなトランザクション・オブジェクトを作成する。
   *
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction getThreadLocalTransaction()
    throws ReasonedRuntimeException
  {
    return new ThreadLocalTransaction();
  }

  /**
   * 指定されたキーに結びつけられたスレッド・セーフなトランザクション・
   * オブジェクトを作成する。
   *
   * @param key キー。
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction getThreadSafeTransaction(String key)
    throws ReasonedRuntimeException
  {
    return new ThreadSafeTransaction(key);
  }

  /**
   * デフォルト・トランザクション・クラスのインスタンスをスレッド・ローカル化
   * したトランザクション・オブジェクトを作成する。
   *
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction createThreadLocalTransaction()
    throws ReasonedRuntimeException
  {
    return createThreadLocalTransaction(DefaultQueryTransaction.class);
  }

  /**
   * デフォルト・トランザクション・クラスのインスタンスをスレッド・セーフ化
   * したトランザクション・オブジェクトを、指定したキーに結びつけて作成する。
   *
   * @param key キー。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction createThreadSafeTransaction(String key)
    throws ReasonedRuntimeException
  {
    return createThreadSafeTransaction(key, DefaultQueryTransaction.class);
  }

  /**
   * 指定されたトランザクション・クラスのインスタンスをスレッド・ローカル化した
   * トランザクション・オブジェクトを作成する。
   *
   * @param transactionClass インスタンス化するトランザクション・クラス。
   * @return スレッド・ローカル化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction createThreadLocalTransaction(
    Class<? extends QueryTransaction> transactionClass
  ) throws ReasonedRuntimeException
  {
    QueryTransaction tr = threadLocal.get();
    if (tr != null && tr.getState() != QueryTransaction.State.ENDED)
      throw new ReasonedRuntimeException(Error.ThreadLocalAlreadyExists);

    try {
      threadLocal.set(transactionClass.newInstance());
    }
    catch (Exception e) {
      throw new ReasonedRuntimeException(Error.ThreadLocalFailToCreate, e);
    }

    return new ThreadLocalTransaction();
  }

  /**
   * 指定されたトランザクション・クラスのインスタンスをスレッド・セーフ化した
   * トランザクション・オブジェクトを、指定したキーに結びつけて作成する。
   *
   * @param key キー。
   * @param transactionClass インスタンス化するトランザクション・クラス。
   * @return スレッド・セーフ化されたトランザクション・オブジェクト。
   */
  public static QueryTransaction createThreadSafeTransaction(
    String key, Class<? extends QueryTransaction> transactionClass
  ) throws ReasonedRuntimeException
  {
    synchronized (threadMap) {
      QueryTransaction tr = threadMap.get(key);
      if (tr != null && tr.getState() != QueryTransaction.State.ENDED)
        throw new ReasonedRuntimeException(Error.ThreadSafeAlreadyExists, key);

      try {
        tr = transactionClass.newInstance();
      }
      catch (Exception e) {
        throw new ReasonedRuntimeException(
          Error.ThreadSafeFailToCreate, key, e);
      }

      threadMap.put(key, tr);

      return new ThreadSafeTransaction(key);
    }
  }

  /**
   * トランザクション・オブジェクトをスレッド・ローカル化するための
   * ラップ・クラス。
   */
  private static final class ThreadLocalTransaction implements QueryTransaction
  {
    /**
     * スレッド・ローカル化するトランザクション・オブジェクトを引数にとる
     * コンストラクタ。
     *
     * @param transaction スレッド・ローカル化するトランザクション・
     *          オブジェクト。
     */
    private ThreadLocalTransaction()
    {
      getInnerTransaction();
    }

    /**
     * 基になったトランザクション・オブジェクトを取得する。
     *
     * @return 基になったトランザクション・オブジェクト。
     */
    private QueryTransaction getInnerTransaction()
    {
      QueryTransaction tr = threadLocal.get();
      if (tr == null)
        throw new ReasonedRuntimeException(
          QueryTransactionManager.Error.ThreadLocalNotExist);

      return tr;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin()
    {
      getInnerTransaction().begin();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void begin(long timeout)
    {
      getInnerTransaction().begin(timeout);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void commit()
    {
      getInnerTransaction().commit();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void rollback()
    {
      getInnerTransaction().rollback();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void end()
    {
      QueryTransaction tr = getInnerTransaction();
      threadLocal.remove();
      tr.end();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public long getBeginTimeMillis()
    {
      return getInnerTransaction().getBeginTimeMillis();
    }

    /**
     * {@inheritDoc}
     */
    @Override 
    public long getLimitTimeMillis()
    {
      return getInnerTransaction().getLimitTimeMillis();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public QueryConnection getConnection(String connId)
      throws ReasonedException
    {
      return getInnerTransaction().getConnection(connId);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void addConnectionFactory(
      String connId, QueryConnectionFactory factory)
    {
      getInnerTransaction().addConnectionFactory(connId, factory);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public State getState()
    {
      QueryTransaction tr = threadLocal.get();
      if (tr == null) {
        return State.ENDED;
      }
      else {
        return tr.getState();
      }
    }
  }

  /**
   * トランザクション・オブジェクトをスレッド・セーフ化するための
   * ラップ・クラス。
   */
  private static final class ThreadSafeTransaction implements QueryTransaction
  {
    /** このトランザクションに結びつけられたキー。 */
    private final String key;

    /**
     * スレッド・セーフ化するトランザクション・オブジェクトとそのキーを引数に
     * とるコンストラクタ。
     *
     * @param key キー。
     * @throws 引数のトランザクションオブジェクトがヌルの場合（デバッグ／モード
     *           のみ）。
     */
    private ThreadSafeTransaction(String key)
    {
      this.key = key;
      getInnerTransaction();
    }

    /**
     * 基になったトランザクション・オブジェクトを取得する。
     *
     * @return 基になったトランザクション・オブジェクト。
     */
    private QueryTransaction getInnerTransaction()
    {
      synchronized (threadMap) {
        QueryTransaction tr = threadMap.get(this.key);
        if (tr == null) {
          throw new ReasonedRuntimeException(
            QueryTransactionManager.Error.ThreadSafeNotExist, this.key);
        }
        return tr;
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void begin()
    {
      getInnerTransaction().begin();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void begin(long timeout)
    {
      getInnerTransaction().begin(timeout);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void commit()
    {
      getInnerTransaction().commit();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void rollback()
    {
      getInnerTransaction().rollback();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void end()
    {
      QueryTransaction tr = getInnerTransaction();
      threadMap.remove(key);
      tr.end();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized long getBeginTimeMillis()
    {
      return getInnerTransaction().getBeginTimeMillis();
    }

    /**
     * {@inheritDoc}
     */
    @Override 
    public synchronized long getLimitTimeMillis()
    {
      return getInnerTransaction().getLimitTimeMillis();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized QueryConnection getConnection(String connId)
      throws ReasonedException
    {
      return getInnerTransaction().getConnection(connId);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public synchronized void addConnectionFactory(
      String connId, QueryConnectionFactory factory)
    {
      getInnerTransaction().addConnectionFactory(connId, factory);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public QueryTransaction.State getState()
    {
      try {
        return getInnerTransaction().getState();
      }
      catch (ReasonedRuntimeException e) {
        if (e.getReason().equals(
            QueryTransactionManager.Error.ThreadSafeNotExist)) {
          return State.ENDED;
        }
        else {
          throw e;
        }
      }
    }
  }
}
