package ts.query;

import ts.tester.UnitTest;
import ts.util.*;
import ts.util.resource.*;
import java.util.*;

public class QueryTransactionManagerTest extends UnitTest
{
  public static void main(String[] args)
  {
    run(QueryTransactionManagerTest.class, args);
  }

  static class MyTransaction extends DefaultQueryTransaction {
    public MyTransaction() {
      throw new RuntimeException("!!!");
    }
  }

  static StringBuilder buf = new StringBuilder();

  static void CLEAR_LOG() {
    buf = new StringBuilder();
  }
  static void LOG(String str) {
    buf.append(str);
  }
  static String OUTPUT_LOG() {
    return buf.toString();
  }

  @Override
  protected void preInvocation(String method)
  {
    CLEAR_LOG();
  }

  static long SLEEP_TIME = 0L;

  static void SLEEP() {
    try {
      Thread.sleep(SLEEP_TIME);
    } catch (InterruptedException e) {}
  }

  static class MyConnectionFactory implements QueryConnectionFactory {
    public MyConnectionFactory() {}
    @Override 
    public QueryConnection create(final String connId, QueryConfig cfg) {
      return new QueryConnection() {
        private long ltime = -1L;
        @Override
        public String getId() { return connId; }
        @Override
        public void setLimitTimeMillis(long tm) { this.ltime = tm; }
        @Override
        public long getLimitTimeMillis() { return this.ltime; }
        @Override
        public Query getQuery(Resource res) { return null; }
        @Override
        public void open() {
          LOG("Open(begin)"); SLEEP(); LOG("Open(end)");
        }
        @Override
        public void commit() {
          LOG("Commit(begin)"); SLEEP(); LOG("Commit(end)");
        }
        @Override
        public void rollback() {
          LOG("Rollback(begin)"); SLEEP(); LOG("Rollback(end)");
        }
        @Override
        public void close() {
          LOG("Close(begin)"); SLEEP(); LOG("Close(end)");
        }
      };
    }
  }

  public void createAndGetThreadLocalTransaction_InLocalThread()
  {
    MSG("スレッド・ローカルなトランザクションをローカルなスレッド内で操作。");

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    tr.begin();
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(OUTPUT_LOG(), "");

    tr = QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    tr.begin();
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    tr.commit();
    EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    tr.end();
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    EQUAL(OUTPUT_LOG(), "");

    QueryTransactionManager.createThreadLocalTransaction();
    tr = QueryTransactionManager.getThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    tr.begin();
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
    try {
      tr.getConnection("c0");
      EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
      tr.commit();
      EQUAL(tr.getState(), QueryTransaction.State.COMMITTED);
    }
    catch (ReasonedException e) {
      tr.rollback();
      EQUAL(tr.getState(), QueryTransaction.State.ROLLBACKED);
    }
    finally {
      tr.end();
      EQUAL(tr.getState(), QueryTransaction.State.ENDED);
    }
    EQUAL(OUTPUT_LOG(),
      "Open(begin)Open(end)Commit(begin)Commit(end)Close(begin)Close(end)");
  }

  public void createAndGetThreadLocalTransaction_InOtherThread()
  {
    MSG("スレッド・ローカルなトランザクションを別のスレッド内で操作。");

    final QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    Thread th0 = new Thread() {
      public void run() {
        try {
          tr.begin();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th0.start();
      th0.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    tr.begin(1000L);
    OK("... transaction begin");
    EQUAL(tr.getState(), QueryTransaction.State.BEGINED);

    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);

    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th1 = new Thread() {
      public void run() {
        try {
          tr.getConnection("c0");
          NG();
        } catch (ReasonedException e) {
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th1.start();
      th1.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th2 = new Thread() {
      public void run() {
        try {
          tr.rollback();
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th2.start();
      th2.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    tr.rollback();
    OK("... transaction rollback");
    EQUAL(tr.getState(), QueryTransaction.State.ROLLBACKED);

    Thread th3 = new Thread() {
      public void run() {
        try {
          tr.end();
          NG();
        } catch (ReasonedRuntimeException e) {
          EQUAL(e.getReason(),
            QueryTransactionManager.Error.ThreadLocalNotExist);
        }
      }
    };
    try {
      th3.start();
      th3.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    tr.end();
    OK("... transaction end");
    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
  }

  public void createAndGetThreadLocalTransaction_trClass_Null()
  {
    MSG("引数がヌルの場合。");

    QueryTransaction tr;
    try {
      QueryTransactionManager.createThreadLocalTransaction(null);
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryTransactionManager.Error.ThreadLocalFailToCreate);
    }
  }

  public void createAndGetThreadLocalTransaction_AlreadExists()
  {
    MSG("既にスレッド・ローカルなトランザクションが作成済みの場合。");

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadLocalTransaction();
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);

    try {
      QueryTransactionManager.createThreadLocalTransaction();
      NG();
    }
    catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryTransactionManager.Error.ThreadLocalAlreadyExists);
    }

    tr.end();
  }

  public void createAndGetThreadSafeTransaction_key_InOtherThreads()
  {
    MSG("スレッド・セーフなトランザクションを別のスレッド内で操作。");

    final QueryTransaction tr;
    tr = QueryTransactionManager.createThreadSafeTransaction("tr00");
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);
    tr.addConnectionFactory("c0", new MyConnectionFactory());

    Thread th0 = new Thread() {
      public void run() {
        tr.begin(1000L);
        EQUAL(tr.getState(), QueryTransaction.State.BEGINED);
      }
    };
    try {
      th0.start();
      th0.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);

    try {
      EQUAL(tr.getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th1 = new Thread() {
      public void run() {
        try {
          EQUAL(tr.getConnection("c0").getId(), "c0");
        } catch (ReasonedException e) {
          NG(e);
        }
      }
    };
    try {
      th1.start();
      th1.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th2 = new Thread() {
      public void run() {
        tr.rollback();
        EQUAL(tr.getState(), QueryTransaction.State.ROLLBACKED);
      }
    };
    try {
      th2.start();
      th2.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th3 = new Thread() {
      public void run() {
        tr.end();
        EQUAL(tr.getState(), QueryTransaction.State.ENDED);
      }
    };
    try {
      th3.start();
      th3.join();
    } catch (InterruptedException e) {
      NG(e);
    }


    QueryTransactionManager.createThreadSafeTransaction("tr00");
    EQUAL(QueryTransactionManager.getThreadSafeTransaction("tr00").getState(), QueryTransaction.State.CREATED);
    QueryTransactionManager.getThreadSafeTransaction("tr00").addConnectionFactory("c0", new MyConnectionFactory());
    
    Thread th10 = new Thread() {
      public void run() {
        QueryTransactionManager.getThreadSafeTransaction("tr00").begin(1000L);
        EQUAL(QueryTransactionManager.getThreadSafeTransaction("tr00").getState(), QueryTransaction.State.BEGINED);
      }
    };
    try {
      th10.start();
      th10.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getLimitTimeMillis(), tr.getBeginTimeMillis() + 1000L);

    try {
      EQUAL(QueryTransactionManager.getThreadSafeTransaction("tr00").getConnection("c0").getId(), "c0");
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th11 = new Thread() {
      public void run() {
        try {
          EQUAL(QueryTransactionManager.getThreadSafeTransaction("tr00").getConnection("c0").getId(), "c0");
        } catch (ReasonedException e) {
          NG(e);
        }
      }
    };
    try {
      th11.start();
      th11.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th12 = new Thread() {
      public void run() {
        QueryTransactionManager.getThreadSafeTransaction("tr00").commit();
        EQUAL(QueryTransactionManager.getThreadSafeTransaction("tr00").getState(), QueryTransaction.State.COMMITTED);
      }
    };
    try {
      th12.start();
      th12.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    Thread th13 = new Thread() {
      public void run() {
        QueryTransactionManager.getThreadSafeTransaction("tr00").end();
        EQUAL(QueryTransactionManager.getThreadSafeTransaction("tr00").getState(), QueryTransaction.State.ENDED);
      }
    };
    try {
      th13.start();
      th13.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    EQUAL(tr.getState(), QueryTransaction.State.ENDED);
  }

  public void createAndGetThreadSafeTransaction_trClass_Null()
  {
    MSG("引数がヌルの場合。");

    final QueryTransaction tr;
    tr = QueryTransactionManager.createThreadSafeTransaction(null);
    tr.addConnectionFactory("c0", new MyConnectionFactory());
    tr.begin();

    QueryTransaction tr1;
    tr1 = QueryTransactionManager.getThreadSafeTransaction(null);
    try {
      EQUAL(tr.getConnection("c0"), tr1.getConnection("c0")); 
    } catch (ReasonedException e) {
      NG(e);
    }

    Thread th = new Thread() {
      public void run() {
        QueryTransaction tr2;
        tr2 = QueryTransactionManager.getThreadSafeTransaction(null);
        try {
          EQUAL(tr.getConnection("c0"), tr2.getConnection("c0")); 
        } catch (ReasonedException e) {
          NG(e);
        }
      }
    };
    try {
      th.start();
      th.join();
    } catch (InterruptedException e) {
      NG(e);
    }

    tr.end();
  }

  public void createAndGetThreadSafeTransaction_key_AlreadExists()
  {
    MSG("既に指定されたキーに対するトランザクションが作成済みの場合。");

    QueryTransaction tr;
    tr = QueryTransactionManager.createThreadSafeTransaction("t0");
    EQUAL(tr.getState(), QueryTransaction.State.CREATED);

    try {
      QueryTransactionManager.createThreadSafeTransaction("t0");
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),
        QueryTransactionManager.Error.ThreadSafeAlreadyExists);
      EQUAL(e.getMessage(), "t0");
    }

    tr.end();
  }

  public void createThreadSafeTransaction_key_FailToCreate()
  {
    MSG("スレッド・セーフなトランザクションの作成に失敗した場合。");

    QueryTransaction tr;
    try {
      tr = QueryTransactionManager.createThreadSafeTransaction(
        "t0", MyTransaction.class);
      NG();
    } catch (ReasonedRuntimeException e) {
      EQUAL(e.getReason(),QueryTransactionManager.Error.ThreadSafeFailToCreate);
      EQUAL(e.getMessage(), "t0");
      EQUAL(e.getCause().getClass(), RuntimeException.class);
      EQUAL(e.getCause().getMessage(), "!!!");
    }
  }
}
