/*
 * QueryContent class.
 *
 * Copyright (C) 2011 SATOH Takayuki All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */
package ts.query;

import ts.util.AbstractTypedGetter;
import java.io.Serializable;
import java.util.Collection;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.LinkedList;
import java.util.Set;
import java.util.LinkedHashSet;

/**
 * クエリの実行内容を格納するためのクラス。
 * <br>
 * 接続先を識別するコネクションID、実行内容を識別するクエリID、クエリの実行内容
 * を作成する元となる情報を格納するリソース・オブジェクトをコンストラクタで
 * 受け取り、作成された実行内容を格納する。
 * <br>
 * 実行内容は、各クエリ・クラスに応じて任意の属性を格納することができる。
 * そして、このクラスは{@link ts.util.TypedGetter}インターフェイスを
 * インプリメントしているので、属性値を基本データ型や文字列、{@link
 * ts.util.DateTime}オブジェクトに変換して取得することが可能である。
 * <br>
 * このクラスのオブジェクトは実行結果オブジェクトの属性にも使用される。
 *
 * @author 佐藤隆之
 * @version $Id: QueryContent.java,v 1.2 2011-09-13 14:28:23 tayu Exp $
 */
public class QueryContent extends AbstractTypedGetter<Enum,Serializable>
{
  /** シリアル・バージョン番号。 */
  private static final long serialVersionUID = 6608747065918026460L;

  /** 各クエリ・クラスに応じた属性を格納するマップ。 */
  private Map<Enum,Serializable> attrMap = null;

  /** 入力パラメータ・リスト。 */
  private final List<QueryParam> paramLst = new LinkedList<QueryParam>();

  /** 出力パラメータ・セット。 */
  private final Set<QueryOutput> outputSet = new LinkedHashSet<QueryOutput>();

  /**
   * デフォルト・コンストラクタ。
   */
  public QueryContent()
  {}

  /**
   * 指定されたキーに結びつけられた属性値を取得する。
   *
   * @param key キー。
   * @return 引数のキーに結びつけられた属性値。
   */
  public Serializable get(Enum key)
  {
    assert (key != null) : "@param:key is null.";

    return (this.attrMap == null) ? null : this.attrMap.get(key);
  }

  /**
   * 指定されたキーに対する属性値を設定する。
   *
   * @param key キー。
   * @param value 引数のキーに結びつけられた属性値。
   * @throws AssertionError 引数のキーがヌルの場合（デバッグ・モードのみ）。
   */
  public void set(Enum key, Serializable value)
  {
    assert (key != null) : "@param:key is null.";

    if (this.attrMap == null) {
      this.attrMap = new HashMap<Enum,Serializable>();
    }

    if (value == null) {
      this.attrMap.remove(key);
    }
    else {
      this.attrMap.put(key, value);
    }
  }

  /**
   * クエリの実行時に使用される入力パラメータのリストを取得する。
   * <br>
   * 同じ入力パラメータが複数回使用される場合は、その重複分も含めて取得される。
   *
   * @return クエリの実行時に使用される入力パラメータのリスト。
   */
  public List<QueryParam> listParams()
  {
    return this.paramLst;
  }

  /**
   * クエリの実行時に使用される入力パラメータの数を取得する。
   * <br>
   * 同じ入力パラメータが複数回使用される場合は、その重複分も数える。
   *
   * @return クエリの実行時に使用される入力パラメータの数。
   */
  public int countParams()
  {
    return this.paramLst.size();
  }

  /**
   * クエリの実行結果の出力項目を取得する。
   * <br>
   * 出力項目の重複は除かれて取得される。
   *
   * @return クエリの実行結果の出力項目。
   */
  public Collection<QueryOutput> getOutputs()
  {
    return this.outputSet;
  }

  /**
   * クエリの実行時に使用される出力項目の数を取得する。
   * <br>
   * 出力項目の重複は除かれて数えられる。
   */
  public int countOutputs()
  {
    return this.outputSet.size();
  }

  /**
   * 入力パラメータを追加する。
   *
   * @param param 入力パラメータ。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addParam(QueryParam param)
  {
    assert (param != null) : "@param:param is null.";

    this.paramLst.add(param);
  }

  /**
   * 複数の入力パラメータを追加する。
   *
   * @param params 入力パラメータのコレクション。
   * @throws AssertionError 引数がヌルの場合、又は引数のコレクションの要素に
   *           ヌルが含まれている場合（デバッグ・モードのみ）。
   */
  public void addParams(Collection<QueryParam> params)
  {
    assert (params != null) : "@param:param is null.";

    for (QueryParam p : params) {
      assert (p != null) : "@param:params has null elements.";
      this.paramLst.add(p);
    }
  }

  /**
   * 複数の入力パラメータを追加する。
   *
   * @param params 入力パラメータの配列。
   * @throws AssertionError 引数がヌルの場合、又は引数の配列の要素にヌルが
   *           含まれている場合（デバッグ・モードのみ）。
   */
  public void addParams(QueryParam[] params)
  {
    assert (params != null) : "@param:param is null.";

    for (QueryParam p : params) {
      assert (p != null) : "@param:params has null elements.";
      this.paramLst.add(p);
    }
  }

  /**
   * 出力項目を追加する。
   *
   * @param output 出力項目。
   * @throws AssertionError 引数がヌルの場合（デバッグ・モードのみ）。
   */
  public void addOutput(QueryOutput output)
  {
    assert (output != null) : "@param:output is null.";

    this.outputSet.add(output);
  }

  /**
   * 複数の出力項目を追加する。
   *
   * @param outputs 出力項目のコレクション。
   * @throws AssertionError 引数がヌルの場合、又は引数のコレクションの要素に
   *           ヌルが含まれている場合（デバッグ・モードのみ）。
   */
  public void addOutputs(Collection<QueryOutput> outputs)
  {
    assert (outputs != null) : "@param:outputs is null.";

    for (QueryOutput o : outputs) {
      assert (o != null) : "@param:outputs has null elements.";
      this.outputSet.add(o);
    }
  }

  /**
   * 複数の出力項目を追加する。
   *
   * @param outputs 出力項目の配列。
   * @throws AssertionError 引数がヌルの場合、又は引数の配列の要素にヌルが
   *           含まれている場合（デバッグ・モードのみ）。
   */
  public void addOutputs(QueryOutput[] outputs)
  {
    assert (outputs != null) : "@param:outputs is null.";

    for (QueryOutput o : outputs) {
      assert (o != null) : "@param:outputs has null elements.";
      this.outputSet.add(o);
    }
  }

  /**
   * クエリが結果テーブルを返すかどうかを取得する。
   * <br>
   * クエリの内容が結果テーブルを返すものであるかどうかを判別し、結果テーブル
   * を返す場合は<tt>true</tt>を返す。
   */
  public boolean hasResultTable()
  {
    return (countOutputs() > 0) ? true : false;
  }
}
