/*
 * Copyright 2006-2007 Sxip Identity Corporation
 */

package jp.sourceforge.tsukuyomi.openid.rp.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import jp.sourceforge.tsukuyomi.openid.association.Association;
import jp.sourceforge.tsukuyomi.openid.rp.ConsumerAssociationStore;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public class InMemoryConsumerAssociationStore implements
		ConsumerAssociationStore {
	private static final Log LOG =
		LogFactory.getLog(InMemoryConsumerAssociationStore.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	private Map<String, Map<String, Association>> idpMap =
		new HashMap<String, Map<String, Association>>();

	public synchronized void save(String idpUrl, Association association) {
		removeExpired();

		Map<String, Association> handleMap = idpMap.get(idpUrl);

		if (handleMap == null) {
			handleMap = new HashMap<String, Association>();

			idpMap.put(idpUrl, handleMap);
		}

		String handle = association.getHandle();

		if (DEBUG) {
			LOG.debug("Adding association to the in-memory store: "
				+ handle
				+ " with OP: "
				+ idpUrl);
		}

		handleMap.put(association.getHandle(), association);
	}

	public synchronized Association load(String idpUrl, String handle) {
		removeExpired();

		if (idpMap.containsKey(idpUrl)) {
			Map<String, Association> handleMap = idpMap.get(idpUrl);

			if (handleMap.containsKey(handle)) {
				return handleMap.get(handle);
			}
		}

		return null;
	}

	public synchronized Association load(String idpUrl) {
		removeExpired();

		Association latest = null;

		if (idpMap.containsKey(idpUrl)) {
			Map<String, Association> handleMap = idpMap.get(idpUrl);

			for (Entry<String, Association> ent : handleMap.entrySet()) {
				Association association = ent.getValue();

				if (latest == null
					|| latest.getExpiry().before(association.getExpiry())) {
					latest = association;
				}
			}
		}

		return latest;
	}

	public synchronized void remove(String idpUrl, String handle) {
		removeExpired();

		if (idpMap.containsKey(idpUrl)) {
			Map<String, Association> handleMap = idpMap.get(idpUrl);

			LOG.info("Removing association: " + handle + " widh OP: " + idpUrl);

			handleMap.remove(handle);

			if (handleMap.size() == 0) {
				idpMap.remove(idpUrl);
			}
		}
	}

	private synchronized void removeExpired() {
		Set<String> idpToRemove = new HashSet<String>();
		for (Entry<String, Map<String, Association>> ent : idpMap.entrySet()) {
			String idpUrl = ent.getKey();
			Map<String, Association> handleMap = ent.getValue();

			Set<String> handleToRemove = new HashSet<String>();
			for (Entry<String, Association> ent2 : handleMap.entrySet()) {
				String handle = ent2.getKey();
				Association association = ent2.getValue();

				if (association.hasExpired()) {
					handleToRemove.add(handle);
				}
			}

			for (String handle : handleToRemove) {
				LOG.info("Removing expired association: "
					+ handle
					+ " with OP: "
					+ idpUrl);

				handleMap.remove(handle);
			}

			if (handleMap.size() == 0) {
				idpToRemove.add(idpUrl);
			}
		}

		for (String idpUrl : idpToRemove) {
			idpMap.remove(idpUrl);
		}
	}

	public synchronized int size() {
		int total = 0;

		for (Map<String, Association> handleMap : idpMap.values()) {
			total += handleMap.size();
		}

		return total;
	}
}
