/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.association;

import java.io.Serializable;
import java.security.GeneralSecurityException;
import java.security.NoSuchAlgorithmException;
import java.util.Date;

import javax.crypto.KeyGenerator;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public class Association implements Serializable {
	private static final long serialVersionUID = 1683487719331303148L;
	private static final Log LOG = LogFactory.getLog(Association.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	public static final String FAILED_ASSOC_HANDLE = " ";
	public static final String TYPE_HMAC_SHA1 = "HMAC-SHA1";
	public static final String TYPE_HMAC_SHA256 = "HMAC-SHA256";

	public static final String HMAC_SHA1_ALGORITHM = "HmacSHA1";
	public static final String HMAC_SHA256_ALGORITHM = "HmacSHA256";
	public static final int HMAC_SHA1_KEYSIZE = 160;
	public static final int HMAC_SHA256_KEYSIZE = 256;

	private String type;
	private String handle;
	private SecretKey macKey;
	private Date expiry;

	private Association(String type, String handle, SecretKey macKey,
			Date expiry) {
		if (DEBUG) {
			LOG.debug("Creating association, type: "
				+ type
				+ " handle: "
				+ handle
				+ " expires: "
				+ expiry);
		}
		this.type = type;
		this.handle = handle;
		this.macKey = macKey;
		this.expiry = expiry;
	}

	private Association(String type, String handle, SecretKey macKey,
			int expiryIn) {
		this(type, handle, macKey, new Date(System.currentTimeMillis()
			+ expiryIn
			* 1000L));
	}

	public static Association getFailedAssociation(int expiryIn) {
		return new Association(null, FAILED_ASSOC_HANDLE, null, new Date(System
			.currentTimeMillis()
			+ expiryIn
			* 1000L));
	}

	public static Association generate(String type, String handle, int expiryIn)
			throws AssociationException {
		if (TYPE_HMAC_SHA1.equals(type)) {
			return generateHmacSha1(handle, expiryIn);
		} else if (TYPE_HMAC_SHA256.equals(type)) {
			return generateHmacSha256(handle, expiryIn);
		} else {
			throw new AssociationException("Unknown association type: " + type);
		}
	}

	public static Association generateHmacSha1(String handle, int expiryIn) {
		SecretKey macKey = generateMacSha1Key();

		if (DEBUG) {
			LOG.debug("Generated SHA1 MAC key: " + macKey);
		}

		return new Association(TYPE_HMAC_SHA1, handle, macKey, expiryIn);
	}

	public static Association createHmacSha1(String handle, byte[] macKeyBytes,
			int expiryIn) {
		SecretKey macKey = createMacKey(HMAC_SHA1_ALGORITHM, macKeyBytes);

		return new Association(TYPE_HMAC_SHA1, handle, macKey, expiryIn);
	}

	public static Association createHmacSha1(String handle, byte[] macKeyBytes,
			Date expDate) {
		SecretKey macKey = createMacKey(HMAC_SHA1_ALGORITHM, macKeyBytes);

		return new Association(TYPE_HMAC_SHA1, handle, macKey, expDate);
	}

	public static Association generateHmacSha256(String handle, int expiryIn) {
		SecretKey macKey = generateMacSha256Key();

		if (DEBUG) {
			LOG.debug("Generated SHA256 MAC key: " + macKey);
		}

		return new Association(TYPE_HMAC_SHA256, handle, macKey, expiryIn);
	}

	public static Association createHmacSha256(String handle,
			byte[] macKeyBytes, int expiryIn) {
		SecretKey macKey = createMacKey(HMAC_SHA256_ALGORITHM, macKeyBytes);

		return new Association(TYPE_HMAC_SHA256, handle, macKey, expiryIn);
	}

	public static Association createHmacSha256(String handle,
			byte[] macKeyBytes, Date expDate) {
		SecretKey macKey = createMacKey(HMAC_SHA256_ALGORITHM, macKeyBytes);

		return new Association(TYPE_HMAC_SHA256, handle, macKey, expDate);
	}

	protected static SecretKey generateMacKey(String algorithm, int keySize) {
		try {
			KeyGenerator keyGen = KeyGenerator.getInstance(algorithm);

			keyGen.init(keySize);

			return keyGen.generateKey();
		} catch (NoSuchAlgorithmException e) {
			LOG.error("Unsupported algorithm: "
				+ algorithm
				+ ", size: "
				+ keySize, e);
			return null;
		}
	}

	protected static SecretKey generateMacSha1Key() {
		return generateMacKey(HMAC_SHA1_ALGORITHM, HMAC_SHA1_KEYSIZE);
	}

	protected static SecretKey generateMacSha256Key() {
		return generateMacKey(HMAC_SHA256_ALGORITHM, HMAC_SHA256_KEYSIZE);
	}

	public static boolean isHmacSupported(String hMacType) {
		String hMacAlgorithm;

		if (TYPE_HMAC_SHA1.equals(hMacType)) {
			hMacAlgorithm = HMAC_SHA1_ALGORITHM;
		} else if (TYPE_HMAC_SHA256.equals(hMacType)) {
			hMacAlgorithm = HMAC_SHA256_ALGORITHM;
		} else {
			return false;
		}

		try {
			KeyGenerator.getInstance(hMacAlgorithm);
		} catch (NoSuchAlgorithmException e) {
			return false;
		}

		return true;
	}

	public static boolean isHmacSha256Supported() {
		try {
			KeyGenerator.getInstance(HMAC_SHA256_ALGORITHM);

			return true;
		} catch (NoSuchAlgorithmException e) {
			return false;
		}
	}

	public static boolean isHmacSha1Supported() {
		try {
			KeyGenerator.getInstance(HMAC_SHA1_ALGORITHM);

			return true;
		} catch (NoSuchAlgorithmException e) {
			return false;
		}
	}

	protected static SecretKey createMacKey(String algorithm, byte[] macKey) {
		return new SecretKeySpec(macKey, algorithm);
	}

	public String getType() {
		return type;
	}

	public String getHandle() {
		return handle;
	}

	public SecretKey getMacKey() {
		return macKey;
	}

	public Date getExpiry() {
		return (Date) expiry.clone();
	}

	public boolean hasExpired() {
		Date now = new Date();

		return expiry.before(now);
	}

	protected byte[] sign(byte[] data) throws AssociationException {
		try {
			String algorithm = macKey.getAlgorithm();
			Mac mac = Mac.getInstance(algorithm);

			mac.init(macKey);

			return mac.doFinal(data);
		} catch (GeneralSecurityException e) {
			throw new AssociationException("Cannot sign!", e);
		}
	}

	public String sign(String text) throws AssociationException {
		if (DEBUG) {
			LOG.debug("Computing signature for input data:\n" + text);
		}

		return new String(Base64.encodeBase64(sign(text.getBytes())));
	}

	public boolean verifySignature(String text, String signature)
			throws AssociationException {
		if (DEBUG) {
			LOG.debug("Verifying signature: " + signature);
		}

		return signature.equals(sign(text));
	}
}
