/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.discovery;

import java.io.Serializable;
import java.net.URL;

public class DiscoveryInformation implements Serializable {
	private static final long serialVersionUID = -2586780820659310850L;

	private URL idpEndpoint;
	private Identifier claimedIdentifier;
	private String delegate;
	private String version;

	public final static String OPENID10 = "http://openid.net/signon/1.0";
	public final static String OPENID11 = "http://openid.net/signon/1.1";
	public final static String OPENID2 =
		"http://specs.openid.net/auth/2.0/signon";
	public final static String OPENID2_OP =
		"http://specs.openid.net/auth/2.0/server";

	public DiscoveryInformation(URL idpEndpoint) throws DiscoveryException {
		this(idpEndpoint, null, OPENID2_OP);
	}

	public DiscoveryInformation(URL idpEndpoint, Identifier claimedIdentifier)
			throws DiscoveryException {
		this(idpEndpoint, claimedIdentifier, OPENID2);
	}

	public DiscoveryInformation(URL idpEndpoint, Identifier claimedIdentifier,
			String version) throws DiscoveryException {
		this(idpEndpoint, claimedIdentifier, null, version);
	}

	public DiscoveryInformation(URL idpEndpoint, Identifier claimedIdentifier,
			String delegate, String version) throws DiscoveryException {
		this.idpEndpoint = idpEndpoint;
		this.claimedIdentifier = claimedIdentifier;
		this.version = version;
		this.delegate = delegate;

		if (this.idpEndpoint == null) {
			throw new DiscoveryException("Null OpenID Provider endpoint.");
		}

		if (this.delegate != null && this.claimedIdentifier == null) {
			throw new DiscoveryException("Claimed ID must be present "
				+ "if delegated ID is used.");
		}
	}

	public boolean hasClaimedIdentifier() {
		return claimedIdentifier != null;
	}

	public boolean hasDelegateIdentifier() {
		return delegate != null;
	}

	public URL getIdpEndpoint() {
		return idpEndpoint;
	}

	public Identifier getClaimedIdentifier() {
		return claimedIdentifier;
	}

	public String getDelegateIdentifier() {
		return delegate;
	}

	public String getVersion() {
		return version;
	}

	public void setVersion(String version) {
		this.version = version;
	}

	public boolean isVersion2() {
		return OPENID2.equals(version) || OPENID2_OP.equals(version);
	}

	@Override
	public String toString() {
		return (isVersion2() ? "OpenID2" : "OpenID1")
			+ "\nOP-endpoint:"
			+ idpEndpoint
			+ "\nClaimedID:"
			+ claimedIdentifier
			+ "\nDelegate:"
			+ delegate;
	}
}
