/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.discovery;

import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class IdentifierParser {
	private static final Log LOG = LogFactory.getLog(IdentifierParser.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	private static final String XRI_PREFIX = "xri://";
	private static final Pattern URL_PATTERN =
		Pattern.compile("^https?://", Pattern.CASE_INSENSITIVE);
	private static final Pattern XRI_PATTERN =
		Pattern.compile("^[!=@\\$\\+\\(]", Pattern.CASE_INSENSITIVE);

	public static Identifier parse(String identifier)
			throws IdentifierException {
		try {
			// strip the xri:// prefix if it exists
			if (identifier.toLowerCase().startsWith(XRI_PREFIX)) {
				if (DEBUG) {
					LOG.debug("Dropping xri:// prefix from identifier: "
						+ identifier);
				}
				identifier = identifier.substring(6);
			}

			if (URL_PATTERN.matcher(identifier).find()) {
				if (DEBUG) {
					LOG.debug("Creating URL identifier for: " + identifier);
				}
				return new UrlIdentifier(identifier);
			} else if (XRI_PATTERN.matcher(identifier).find()) {
				if (DEBUG) {
					LOG.debug("Creating XRI identifier for: " + identifier);
				}
				return new XriIdentifier(identifier);
			} else {
				if (DEBUG) {
					LOG
						.debug("Creating URL identifier (http:// prepended) for: "
							+ identifier);
				}
				return new UrlIdentifier("http://" + identifier);
			}
		} catch (Exception e) {
			throw new IdentifierException("Cannot parse identifier: "
				+ identifier, e);
		}
	}
}
