/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.discovery.impl;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;
import java.util.regex.Pattern;

import jp.sourceforge.tsukuyomi.openid.discovery.Discovery;
import jp.sourceforge.tsukuyomi.openid.discovery.DiscoveryException;
import jp.sourceforge.tsukuyomi.openid.discovery.DiscoveryInformation;
import jp.sourceforge.tsukuyomi.openid.discovery.HtmlResolver;
import jp.sourceforge.tsukuyomi.openid.discovery.HtmlResult;
import jp.sourceforge.tsukuyomi.openid.discovery.Identifier;
import jp.sourceforge.tsukuyomi.openid.discovery.IdentifierException;
import jp.sourceforge.tsukuyomi.openid.discovery.IdentifierParser;
import jp.sourceforge.tsukuyomi.openid.discovery.UrlIdentifier;
import jp.sourceforge.tsukuyomi.openid.discovery.XriIdentifier;
import jp.sourceforge.tsukuyomi.openid.yadis.YadisResolver;
import jp.sourceforge.tsukuyomi.openid.yadis.YadisResult;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.openxri.resolve.Resolver;
import org.openxri.resolve.ResolverState;
import org.openxri.resolve.TrustType;
import org.openxri.resolve.exception.PartialResolutionException;
import org.openxri.xml.CanonicalID;
import org.openxri.xml.SEPType;
import org.openxri.xml.SEPUri;
import org.openxri.xml.Service;
import org.openxri.xml.XRD;
import org.openxri.xml.XRDS;
import org.w3c.dom.Element;

public class DiscoveryImpl implements Discovery {
	private static final Log LOG = LogFactory.getLog(DiscoveryImpl.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();
	private Resolver xriResolver = new Resolver();
	private YadisResolver yadisResolver;
	private HtmlResolver htmlResolver;

	public List<DiscoveryInformation> discover(Identifier identifier)
			throws DiscoveryException, IdentifierException {
		List<DiscoveryInformation> result =
			new ArrayList<DiscoveryInformation>();

		if (identifier instanceof XriIdentifier) {
			LOG.info("Starting discovery on XRI identifier: " + identifier);

			XRDS xrds;
			XriIdentifier xriIdentifier = (XriIdentifier) identifier;

			try {
				TrustType trustAll = new TrustType(TrustType.TRUST_NONE);
				xrds =
					xriResolver.resolveAuthToXRDS(
						xriIdentifier.getXriIdentifier(),
						trustAll,
						true,
						new ResolverState());

				if (DEBUG) {
					LOG.debug("Retrieved XRDS:\n" + xrds.dump());
				}

				XRD xrd = xrds.getFinalXRD();
				CanonicalID canonical = xrd.getCanonicalidAt(0);

				// todo: this is not the right place to put
				// isProviderAuthoritative
				if (isProviderAuthoritative(xrd.getProviderID(), canonical)) {
					LOG.info("XRI resolution succeeded on " + identifier);
					result =
						extractDiscoveryInformation(
							xrds,
							(XriIdentifier) identifier,
							xriResolver);
				} else {
					LOG
						.warn("ProviderID is not authoritative for the CanonicalID. "
							+ "Returning empty discovery result set.");
				}
			} catch (Exception e) {
				throw new DiscoveryException("Cannot resolve XRI: "
					+ identifier.toString(), e);
			}

		} else if (identifier instanceof UrlIdentifier) {
			LOG.info("Starting discovery on URL identifier: " + identifier);

			UrlIdentifier urlId = (UrlIdentifier) identifier;

			YadisResult yadis =
				yadisResolver.discover(urlId.getUrl().toString());

			if (YadisResult.OK == yadis.getStatus()) {
				LOG.info("Using Yadis normalized URL as claimedID: "
					+ yadis.getNormalizedUrl());

				result =
					extractDiscoveryInformation(
						yadis.getXrds(),
						new UrlIdentifier(yadis.getNormalizedUrl()));
			}

			if (result.size() == 0) {
				LOG
					.info("No OpenID service endpoints discovered through Yadis;"
						+ " attempting HTML discovery...");

				result =
					extractDiscoveryInformation(htmlResolver.discover(urlId));
			}
		} else {
			throw new DiscoveryException("Unknown identifier type: "
				+ identifier.toString());
		}

		LOG.info("Discovered " + result.size() + " OpenID endpoints.");

		return result;
	}

	/**
	 * Extracts OpenID discovery endpoints from a HTML discovery result.
	 * 
	 * @param htmlResult
	 *            HTML discovery result.
	 * @return List of DiscoveryInformation endpoints.
	 * @throws DiscoveryException
	 *             when invalid information is discovered.
	 */
	private List<DiscoveryInformation> extractDiscoveryInformation(
			HtmlResult htmlResult) throws DiscoveryException {
		ArrayList<DiscoveryInformation> htmlList =
			new ArrayList<DiscoveryInformation>();

		if (htmlResult.getIdp2Endpoint() != null) {
			DiscoveryInformation extracted =
				new DiscoveryInformation(
					htmlResult.getIdp2Endpoint(),
					htmlResult.getClaimedId(),
					htmlResult.getDelegate2(),
					DiscoveryInformation.OPENID2);

			if (DEBUG) {
				LOG.debug("OpenID2-signon HTML discovery endpoint: "
					+ extracted);
			}

			htmlList.add(extracted);
		}

		if (htmlResult.getIdp1Endpoint() != null) {
			DiscoveryInformation extracted =
				new DiscoveryInformation(
					htmlResult.getIdp1Endpoint(),
					htmlResult.getClaimedId(),
					htmlResult.getDelegate1(),
					DiscoveryInformation.OPENID11);

			if (DEBUG) {
				LOG.debug("OpenID1-signon HTML discovery endpoint: "
					+ extracted);
			}

			htmlList.add(extracted);
		}

		return htmlList;
	}

	private boolean isProviderAuthoritative(String providerId,
			CanonicalID canonicalId) {
		// todo: also handle xri delegation / community names
		// todo: isProviderAuthoritative does not work on multi-level i-names
		if (canonicalId == null || canonicalId.getValue() == null) {
			return false;
		}

		String auth = canonicalId.getValue().substring(0, 1);
		XRD rootAuth = xriResolver.getAuthority(auth);

		if (!rootAuth.getProviderID().equals(providerId)) {
			return false;
		}

		int lastbang = canonicalId.getValue().lastIndexOf("!");
		String parent =
			lastbang > -1
				? canonicalId.getValue().substring(0, lastbang)
				: canonicalId.getValue();

		String parentNoPrefix =
			parent.startsWith("xri://") ? parent.substring(6) : parent;

		String providerIDNoPrefix =
			providerId.startsWith("xri://")
				? providerId.substring(6)
				: providerId;

		return parentNoPrefix.equals(providerIDNoPrefix);
	}

	public static boolean extractDiscoveryInformationOpenID(
			Resolver xriResolver, ArrayList<DiscoveryInformation> out,
			XRD baseXRD, XriIdentifier identifier, String srvType,
			boolean wantCID) {
		try {
			XRDS tmpXRDS =
				xriResolver.selectServiceFromXRD(baseXRD, identifier
					.getXriIdentifier(), new TrustType(), srvType, null, // sepMediaType
					true, // followRefs
					new ResolverState());

			Identifier claimedIdentifier = null;
			URL opEndpointUrl;
			CanonicalID canonID;

			XRD tmpXRD = tmpXRDS.getFinalXRD();

			if (wantCID) {
				canonID = tmpXRD.getCanonicalidAt(0);

				if (canonID == null) {
					LOG.error("No CanonicalID found for "
						+ srvType
						+ " after XRI resolution of: "
						+ identifier.getIdentifier());
					return false;
				}

				// todo: canonicalID verification?
				claimedIdentifier = IdentifierParser.parse(canonID.getValue());
				LOG.info("Using canonicalID as claimedID: "
					+ claimedIdentifier.getIdentifier()
					+ " for "
					+ srvType);
			}

			for (Object s : tmpXRD.getSelectedServices().getList()) {
				Service srv = (Service) s;
				for (Object u : srv.getPrioritizedURIs()) {
					try {
						SEPUri sepURI = (SEPUri) u;
						String urlString =
							xriResolver.constructURI(sepURI.getURI(), sepURI
								.getAppend(), identifier.getXriIdentifier());

						opEndpointUrl = new URL(urlString);

						DiscoveryInformation extracted =
							new DiscoveryInformation(opEndpointUrl, wantCID
								? claimedIdentifier
								: null, null, srvType);

						LOG.info("Added "
							+ srvType
							+ " endpoint: "
							+ opEndpointUrl);

						out.add(extracted);
					} catch (MalformedURLException mue) {
						LOG.error("Error parsing URI in XRDS result for "
							+ srvType, mue);
					}
				}
			}

			return true;
		} catch (PartialResolutionException e) {
			LOG.error("XRI resolution failed for " + srvType, e);
		} catch (DiscoveryException e) {
			LOG.error("XRDS discovery failed for " + srvType, e);
		} catch (IdentifierException e) {
			LOG.error("XRDS discovery failed for " + srvType, e);
		}

		return false;
	}

	/**
	 * Extracts OpenID discovery endpoints from a XRDS discovery result. Can be
	 * used for both URLs and XRIs, however the
	 * {@link #extractDiscoveryInformation(XRDS, XriIdentifier, Resolver)}
	 * offers additional functionality for XRIs.
	 * 
	 * @param xrds
	 *            The discovered XRDS document.
	 * @param identifier
	 *            The identifier on which discovery was performed.
	 * @return A list of DiscoveryInformation endpoints.
	 * @throws DiscoveryException
	 *             when invalid information is discovered.
	 * @throws IdentifierException
	 */
	protected static List<DiscoveryInformation> extractDiscoveryInformation(
			XRDS xrds, Identifier identifier) throws DiscoveryException,
			IdentifierException {
		ArrayList<DiscoveryInformation> opSelectList =
			new ArrayList<DiscoveryInformation>();
		ArrayList<DiscoveryInformation> signonList =
			new ArrayList<DiscoveryInformation>();
		ArrayList<DiscoveryInformation> openid1 =
			new ArrayList<DiscoveryInformation>();

		XRD xrd = xrds.getFinalXRD();

		// iterate through all services
		for (Object s : xrd.getPrioritizedServices()) {
			Service service = (Service) s;

			for (Object u : service.getPrioritizedURIs()) {
				URL opEndpointUrl;
				try {
					opEndpointUrl = ((SEPUri) u).getURI().toURL();
				} catch (MalformedURLException e) {
					continue;
				}

				if (matchType(service, DiscoveryInformation.OPENID2_OP)) {
					DiscoveryInformation extracted =
						new DiscoveryInformation(opEndpointUrl);

					if (DEBUG) {
						LOG.debug("OpenID2-server XRDS discovery result:\n"
							+ extracted);
					}

					opSelectList.add(extracted);
				}

				if (matchType(service, DiscoveryInformation.OPENID2)) {
					Identifier claimedIdentifier = identifier;
					CanonicalID canonicalId = xrd.getCanonicalidAt(0);
					String providerId = xrd.getProviderID();

					if (identifier instanceof XriIdentifier) {
						if (canonicalId == null) {
							throw new DiscoveryException(
								"No CanonicalID found after XRI resolution of: "
									+ identifier.getIdentifier());
						}

						if (providerId == null || providerId.length() == 0) {
							throw new DiscoveryException(
								"No Provider ID found after XRI resolution of: "
									+ identifier.getIdentifier());
						}

						claimedIdentifier =
							IdentifierParser.parse(canonicalId.getValue());
					}

					DiscoveryInformation extracted =
						new DiscoveryInformation(
							opEndpointUrl,
							claimedIdentifier,
							getDelegate(service, false),
							DiscoveryInformation.OPENID2);

					if (DEBUG) {
						LOG.debug("OpenID2-signon XRDS discovery result:\n"
							+ extracted);
					}

					signonList.add(extracted);
				}

				if (matchType(service, DiscoveryInformation.OPENID10)
					|| matchType(service, DiscoveryInformation.OPENID11)) {
					DiscoveryInformation extracted =
						new DiscoveryInformation(
							opEndpointUrl,
							identifier,
							getDelegate(service, true),
							DiscoveryInformation.OPENID11);

					if (DEBUG) {
						LOG.debug("OpenID1-signon XRDS discovery result:\n"
							+ extracted);
					}

					openid1.add(extracted);
				}
			}
		}

		opSelectList.addAll(signonList);
		opSelectList.addAll(openid1);

		if (opSelectList.size() == 0) {
			LOG.info("No OpenID service types found in the XRDS.");
		}

		return opSelectList;
	}

	// deprecated in open-xri, copied here to avoid warnings
	public static boolean matchType(Service service, String sVal) {
		for (int i = 0; i < service.getNumTypes(); i++) {
			SEPType type = service.getTypeAt(i);
			if (type.match(sVal)) {
				return true;
			}
		}
		return false;

	}

	public static String getDelegate(Service service, boolean compatibility) {
		String delegate = null;
		String delegateTag;
		String nsPattern;

		if (compatibility) {
			delegateTag = "Delegate";
			nsPattern = "http://openid\\.net/xmlns/1\\.0";
		} else {
			delegateTag = "LocalID";
			nsPattern = "xri://\\$xrd\\*\\(\\$v\\*2\\.0\\)";
		}

		Vector<?> delegateTags = service.getOtherTagValues(delegateTag);
		for (int i = 0; delegateTags != null && i < delegateTags.size(); i++) {
			Element element = (Element) delegateTags.elementAt(i);

			if (Pattern.matches(nsPattern, element.getNamespaceURI())) {
				delegate = element.getFirstChild().getNodeValue();

				// todo: multiple delegate tags?
				if (DEBUG) {
					LOG.debug("Found delegate: " + delegate);
				}
			}
		}

		return delegate;
	}

	/**
	 * Extracts OpenID discovery endpoints from a XRDS discovery result for XRI
	 * identifiers.
	 * 
	 * @param xrds
	 *            The discovered XRDS document.
	 * @param identifier
	 *            The identifier on which discovery was performed.
	 * @param xriResolver
	 *            The XRI resolver to use for extraction of OpenID service
	 *            endpoints.
	 * @return A list of DiscoveryInformation endpoints.
	 * @throws DiscoveryException
	 *             when invalid information is discovered.
	 */
	protected static List<DiscoveryInformation> extractDiscoveryInformation(
			XRDS xrds, XriIdentifier identifier, Resolver xriResolver)
			throws DiscoveryException {
		ArrayList<DiscoveryInformation> endpoints =
			new ArrayList<DiscoveryInformation>();

		XRD xrd = xrds.getFinalXRD();

		// try OP Identifier
		extractDiscoveryInformationOpenID(
			xriResolver,
			endpoints,
			xrd,
			identifier,
			DiscoveryInformation.OPENID2_OP,
			false // no CID
		);

		// OpenID 2 signon
		extractDiscoveryInformationOpenID(
			xriResolver,
			endpoints,
			xrd,
			identifier,
			DiscoveryInformation.OPENID2, // sepType
			true // want CID
		);

		// OpenID 1.x
		extractDiscoveryInformationOpenID(
			xriResolver,
			endpoints,
			xrd,
			identifier,
			DiscoveryInformation.OPENID11,
			true // wantCID
		);

		extractDiscoveryInformationOpenID(
			xriResolver,
			endpoints,
			xrd,
			identifier,
			DiscoveryInformation.OPENID10,
			true // wantCID
		);

		if (endpoints.size() == 0) {
			LOG.info("No OpenID service types found in the XRDS.");
		}

		return endpoints;
	}

	public void setYadisResolver(YadisResolver yadisResolver) {
		this.yadisResolver = yadisResolver;
	}

	public void setHtmlResolver(HtmlResolver htmlResolver) {
		this.htmlResolver = htmlResolver;
	}

}
