/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message;

import java.util.Arrays;
import java.util.List;

import jp.sourceforge.tsukuyomi.openid.association.AssociationSessionType;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public class AssociationError extends DirectError {
	private static final Log LOG = LogFactory.getLog(AssociationError.class);

	public static final String ASSOC_ERR = "unsupported-type";

	protected final static List<String> requiredFields =
		Arrays.asList(new String[] {
			"ns",
			"error_code",
			"error",
			"session_type" });

	protected final static List<String> optionalFields =
		Arrays.asList(new String[] { "assoc_type" });

	protected AssociationError(String msg, AssociationSessionType type) {
		super(msg);

		set("ns", OPENID2_NS);
		set("error_code", ASSOC_ERR);
		set("session_type", type.getSessionType());
		set("assoc_type", type.getAssociationType());
	}

	protected AssociationError(ParameterList params) {
		super(params);
	}

	public static AssociationError createAssociationError(String msg,
			AssociationSessionType type) {
		AssociationError err = new AssociationError(msg, type);

		if (!err.isValid()) {
			LOG.error("Invalid association error message created, "
				+ "type: "
				+ type
				+ " message: "
				+ msg);
		}

		return err;
	}

	public static AssociationError createAssociationError(ParameterList params) {
		AssociationError err = new AssociationError(params);

		if (!err.isValid()) {
			LOG.error("Invalid association error message created: "
				+ err.keyValueFormEncoding());
		}

		return err;
	}

	@Override
	public List<String> getRequiredFields() {
		return requiredFields;
	}

	public void setAssociationSessionType(AssociationSessionType type) {
		set("session_type", type.getSessionType());
		set("assoc_type", type.getAssociationType());
	}

	public String getSessionType() {
		return getParameterValue("session_type");
	}

	public String getAssocType() {
		return getParameterValue("assoc_type");
	}

	private String getErrorCode() {
		return getParameterValue("error_code");
	}

	@Override
	public boolean isValid() {
		if (!super.isValid()) {
			return false;
		}

		return ASSOC_ERR.equals(getErrorCode())
			&& OPENID2_NS.equals(getParameterValue("ns"));

	}

}
