/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * A list of parameters that are part of an OpenID message. Please note that you
 * can have multiple parameters with the same name.
 * 
 * @author Marius Scurtescu, Johnny Bufu
 */
public class ParameterList implements Serializable {
	private static final long serialVersionUID = 5250211330154176713L;

	private static final Log LOG = LogFactory.getLog(ParameterList.class);

	private static final boolean DEBUG = LOG.isDebugEnabled();

	private Map<String, Parameter> parameterMap;

	public ParameterList() {
		parameterMap = new LinkedHashMap<String, Parameter>();

		if (DEBUG) {
			LOG.debug("Created empty parameter list.");
		}
	}

	public ParameterList(ParameterList that) {
		if (DEBUG) {
			LOG.debug("Cloning parameter list:\n" + that);
		}

		this.parameterMap =
			new LinkedHashMap<String, Parameter>(that.parameterMap);
	}

	public ParameterList(Map<?, ?> parameterMap) {
		this.parameterMap = new LinkedHashMap<String, Parameter>();

		for (Entry<?, ?> entry : parameterMap.entrySet()) {
			String name = (String) entry.getKey();
			Object v = entry.getValue();

			String value;
			if (v instanceof Object[]) {
				Object[] values = (Object[]) v;
				if (values.length > 1) {
					throw new IllegalArgumentException(
						"Multiple parameters with the same name: "
							+ Arrays.toString(values));
				}

				value = values.length > 0 ? (String) values[0] : null;
			} else {
				value = (String) v;
			}

			set(new Parameter(name, value));
		}

		if (DEBUG) {
			LOG.debug("Creating parameter list:\n" + this);
		}
	}

	public void copyOf(ParameterList that) {
		if (DEBUG) {
			LOG.debug("Copying parameter list:\n" + that);
		}

		this.parameterMap =
			new LinkedHashMap<String, Parameter>(that.parameterMap);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}

		if (obj == null || getClass() != obj.getClass()) {
			return false;
		}

		final ParameterList that = (ParameterList) obj;

		return parameterMap.equals(that.parameterMap);
	}

	@Override
	public int hashCode() {
		return parameterMap.hashCode();
	}

	public void set(Parameter parameter) {
		parameterMap.put(parameter.getKey(), parameter);
	}

	public void addParams(ParameterList params) {
		for (Parameter param : params.getParameters()) {
			set(param);
		}
	}

	public Parameter getParameter(String name) {
		return parameterMap.get(name);
	}

	public String getParameterValue(String name) {
		Parameter param = getParameter(name);

		return param != null ? param.getValue() : null;
	}

	public List<Parameter> getParameters() {
		return new ArrayList<Parameter>(parameterMap.values());
	}

	public void removeParameters(String name) {
		parameterMap.remove(name);
	}

	public boolean hasParameter(String name) {
		return parameterMap.containsKey(name);
	}

	/**
	 * Create a parameter list based on a URL encoded HTTP query string.
	 */
	public static ParameterList createFromQueryString(String queryString)
			throws MessageException {
		if (DEBUG) {
			LOG.debug("Creating parameter list from query string: "
				+ queryString);
		}

		ParameterList parameterList = new ParameterList();

		StringTokenizer tokenizer = new StringTokenizer(queryString, "&");
		while (tokenizer.hasMoreTokens()) {
			String keyValue = tokenizer.nextToken();
			int posEqual = keyValue.indexOf('=');

			if (posEqual == -1) {
				throw new MessageException(
					"Invalid query parameter, = missing: " + keyValue);
			}

			try {
				String key =
					URLDecoder.decode(keyValue.substring(0, posEqual), "UTF-8");
				String value =
					URLDecoder
						.decode(keyValue.substring(posEqual + 1), "UTF-8");

				parameterList.set(new Parameter(key, value));
			} catch (UnsupportedEncodingException e) {
				throw new MessageException(
					"Cannot URL decode query parameter: " + keyValue,
					e);
			}
		}

		return parameterList;
	}

	public static ParameterList createFromKeyValueForm(String keyValueForm)
			throws MessageException {
		if (DEBUG) {
			LOG.debug("Creating parameter list from key-value form:\n"
				+ keyValueForm);
		}

		ParameterList parameterList = new ParameterList();

		StringTokenizer tokenizer = new StringTokenizer(keyValueForm, "\n");
		while (tokenizer.hasMoreTokens()) {
			String keyValue = tokenizer.nextToken();
			int posColon = keyValue.indexOf(':');

			if (posColon == -1) {
				throw new MessageException(
					"Invalid Key-Value form, colon missing: " + keyValue);
			}

			String key = keyValue.substring(0, posColon);
			String value = keyValue.substring(posColon + 1);

			parameterList.set(new Parameter(key, value));
		}

		return parameterList;
	}

	// todo: same as Message.keyValueFormEncoding()
	@Override
	public String toString() {
		StringBuffer allParams = new StringBuffer("");

		for (Parameter parameter : getParameters()) {
			allParams.append(parameter.getKey());
			allParams.append(':');
			allParams.append(parameter.getValue());
			allParams.append('\n');
		}

		return allParams.toString();
	}
}
