/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * VerifyRequest is a AuthSuccess with the openid.mode switched to
 * check_authentication.
 * 
 * @author Marius Scurtescu, Johnny Bufu
 */
public class VerifyRequest extends AuthSuccess {
	private static final Log LOG = LogFactory.getLog(VerifyRequest.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	public static final String MODE_CHKAUTH = "check_authentication";

	protected VerifyRequest(AuthSuccess authResp) {
		super(convertAuthSuccessParams(authResp));
	}

	private static ParameterList convertAuthSuccessParams(AuthSuccess authResp) {
		ParameterList params = new ParameterList(authResp.getParameterMap());

		params.set(new Parameter("openid.mode", MODE_CHKAUTH));

		return params;
	}

	protected VerifyRequest(ParameterList params) {
		super(params);
	}

	public static VerifyRequest createVerifyRequest(AuthSuccess authResp)
			throws MessageException {
		VerifyRequest req = new VerifyRequest(authResp);

		if (!req.isValid()) {
			throw new MessageException(
				"Invalid set of parameters for a verification request");
		}

		if (DEBUG) {
			LOG.debug("Created verification request "
				+ "from a positive auth response:\n"
				+ req.keyValueFormEncoding());
		}

		return req;
	}

	public static VerifyRequest createVerifyRequest(ParameterList params)
			throws MessageException {
		VerifyRequest req = new VerifyRequest(params);

		if (!req.isValid()) {
			throw new MessageException(
				"Invalid set of parameters for a verification request");
		}

		if (DEBUG) {
			LOG.debug("Created verification request:\n"
				+ req.keyValueFormEncoding());
		}

		return req;
	}

	@Override
	public String getHandle() {
		return getParameterValue("openid.assoc_handle");
	}

	@Override
	public String getInvalidateHandle() {
		return getParameterValue("openid.invalidate_handle");
	}

	@Override
	public boolean isValid() {
		if (!MODE_CHKAUTH.equals(getParameterValue("openid.mode"))) {
			LOG.warn("Invalid openid.mode in verification request: "
				+ getParameterValue("openid.mode"));
			return false;
		}

		set("openid.mode", MODE_IDRES);

		if (DEBUG) {
			LOG.debug("Delegating verification request validity check "
				+ "to auth response...");
		}

		if (!super.isValid()) {
			LOG.warn("Invalid verification request: "
				+ "related auth response verification failed.");
			return false;
		}

		set("openid.mode", MODE_CHKAUTH);

		return true;
	}
}
