/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message;

import java.util.Arrays;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public class VerifyResponse extends Message {
	private static final Log LOG = LogFactory.getLog(VerifyResponse.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	protected final static List<String> REQUIRED_FIELDS =
		Arrays.asList(new String[] { "is_valid" });

	protected final static List<String> OPTIONAL_FIELDS =
		Arrays.asList(new String[] { "ns", "invalidate_handle" });

	protected VerifyResponse(boolean compatibility) {
		setSignatureVerified(false);

		if (!compatibility) {
			set("ns", OPENID2_NS);
		}
	}

	protected VerifyResponse(ParameterList params) {
		super(params);
	}

	public static VerifyResponse createVerifyResponse(boolean compatibility)
			throws MessageException {
		VerifyResponse resp = new VerifyResponse(compatibility);

		if (!resp.isValid()) {
			throw new MessageException(
				"Invalid set of parameters for the requested message type");
		}

		if (DEBUG) {
			LOG.debug("Created verification response:\n"
				+ resp.keyValueFormEncoding());
		}

		return resp;
	}

	public static VerifyResponse createVerifyResponse(ParameterList params)
			throws MessageException {
		VerifyResponse resp = new VerifyResponse(params);

		if (!resp.isValid()) {
			throw new MessageException(
				"Invalid set of parameters for the requested message type");
		}

		if (DEBUG) {
			LOG.debug("Created verification response:\n"
				+ resp.keyValueFormEncoding());
		}

		return resp;
	}

	@Override
	public List<String> getRequiredFields() {
		return REQUIRED_FIELDS;
	}

	public boolean isVersion2() {
		return hasParameter("ns") && OPENID2_NS.equals(getParameterValue("ns"));
	}

	public void setSignatureVerified(boolean verified) {
		set("is_valid", verified ? "true" : "false");
	}

	public boolean isSignatureVerified() {
		return "true".equals(getParameterValue("is_valid"));
	}

	public void setInvalidateHandle(String handle) {
		set("invalidate_handle", handle);
	}

	public String getInvalidateHandle() {
		return getParameterValue("invalidate_handle");
	}

	@Override
	public boolean isValid() {
		if (!super.isValid()) {
			return false;
		}

		if (!"true".equals(getParameterValue("is_valid"))
			&& !"false".equals(getParameterValue("is_valid"))) {
			LOG.warn("Invalid is_valid value in verification response: "
				+ getParameterValue("is_valid"));
			return false;
		}

		return true;
	}
}
