/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message.ax;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import jp.sourceforge.tsukuyomi.openid.message.MessageException;
import jp.sourceforge.tsukuyomi.openid.message.Parameter;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Implements the extension for Attribute Exchange fetch responses.
 * 
 * @author Marius Scurtescu, Johnny Bufu
 */
public class FetchResponse extends AxMessage {
	private static final Log LOG = LogFactory.getLog(FetchResponse.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	/**
	 * Constructs a Fetch Response with an empty parameter list.
	 */
	protected FetchResponse() {
		parameters.set(new Parameter("mode", "fetch_response"));

		if (DEBUG) {
			LOG.debug("Created empty fetch response.");
		}
	}

	/**
	 * Constructs a Fetch Response with an empty parameter list.
	 */
	public static FetchResponse createFetchResponse() {
		return new FetchResponse();
	}

	/**
	 * Constructs a FetchResponse from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	protected FetchResponse(ParameterList params) {
		parameters = params;
	}

	public static FetchResponse createFetchResponse(ParameterList params)
			throws MessageException {
		FetchResponse resp = new FetchResponse(params);

		if (!resp.isValid()) {
			throw new MessageException(
				"Invalid parameters for a fetch response");
		}

		if (DEBUG) {
			LOG.debug("Created fetch response from parameter list:\n" + params);
		}

		return resp;
	}

	/**
	 * Creates a FetchResponse from a FetchRequest message and the data released
	 * by the user.
	 * 
	 * @param req
	 *            FetchRequest message.
	 * @param userData
	 *            The userData may be a Map<String alias, String value> or a
	 *            Map<String alias, List<String> values>. The attribute values
	 *            are provided by the calling application. If a list of values
	 *            is specified per attribute, at most n will be sent, where n is
	 *            the number of attribute values requested in the FetchRequest.
	 * @return Properly formed FetchResponse.
	 */
	public static FetchResponse createFetchResponse(FetchRequest req,
			Map<String, String> userData) {
		FetchResponse resp = new FetchResponse();

		// go through each requested attribute
		Map<String, Object> attributes = req.getAttributes();

		for (Entry<String, Object> ent : attributes.entrySet()) {
			// find attribute in userData
			String alias = ent.getKey();
			Object value = userData.get(alias);

			// if the value isn't there, skip over it
			if (value == null) {
				continue;
			}

			// if the value is a string, add the single attribute to the
			// response
			if (value instanceof String) {
				resp.addAttribute(
					alias,
					(String) ent.getValue(),
					(String) value);
			} else if (value instanceof List) {
				// if the value is a list (of string) iteratively add each
				// attribute
				// to the response
				Iterator<?> values = ((List<?>) value).iterator();

				// only send up the the maximum requested number
				int max = req.getCount(alias);
				for (int count = 0; count < max && values.hasNext(); count++) {
					// if the value isn't there, skip over it
					String val = (String) values.next();
					if (val == null) {
						count--; // disregard this as a value as we are
						// skipping over it
						continue;
					}
					resp.addAttribute(alias, (String) ent.getValue(), val);
				}
			}
		}

		return resp;
	}

	/**
	 * Adds an attribute to the fetch response.
	 * 
	 * @param alias
	 *            The alias identifier that will be associated with the
	 *            attribute type URI.
	 * @param typeUri
	 *            The attribute type URI.
	 * @param value
	 *            The value of the attribute.
	 */
	public void addAttribute(String alias, String typeUri, String value) {
		int count = getCount(alias);

		String index = "";

		switch (count) {
		case 0:
			parameters.set(new Parameter("type." + alias, typeUri));
			break;

		case 1:
			// rename the existing one
			parameters.set(new Parameter(
				"value." + alias + ".1",
				getParameterValue("value." + alias)));
			parameters.removeParameters("value." + alias);
			index = ".2";
			break;

		default:
			index = "." + Integer.toString(count + 1);
		}

		parameters.set(new Parameter("value." + alias + index, value));
		setCount(alias, ++count);

		if (DEBUG) {
			LOG.debug("Added new attribute to fetch response; type: "
				+ typeUri
				+ " alias: "
				+ alias
				+ " count: "
				+ count);
		}
	}

	/**
	 * Returns a list with the attribute value(s) associated for the specified
	 * attribute alias.
	 * 
	 * @param alias
	 *            The attribute alias.
	 * @return List of attribute values.
	 */
	public List<String> getAttributeValues(String alias) {
		List<String> values = new ArrayList<String>();

		if (!parameters.hasParameter("count." + alias)) {
			values.add(getParameterValue("value." + alias));
		} else {
			for (int i = 1; i <= getCount(alias); i++) {
				values.add(getParameterValue("value."
					+ alias
					+ "."
					+ Integer.toString(i)));
			}
		}

		return values;
	}

	// todo: public String getAttributeValue(String alias)

	/**
	 * Gets a list of attribute aliases.
	 */
	public List<String> getAttributeAliases() {
		List<String> aliases = new ArrayList<String>();

		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (paramName.startsWith("value.")) {
				String alias;
				if (paramName.endsWith(".")) {
					alias = paramName.substring(6, paramName.length() - 1);
				} else {
					alias = paramName.substring(6);
				}

				if (!aliases.contains(alias)) {
					aliases.add(alias);
				}
			}
		}

		return aliases;
	}

	/**
	 * Gets a map with attribute aliases -> list of values.
	 */
	public Map<String, Object> getAttributes() {
		Map<String, Object> attributes = new HashMap<String, Object>();

		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (paramName.startsWith("value.")) {
				String alias;
				if (paramName.endsWith(".")) {
					alias = paramName.substring(6, paramName.length() - 1);
				} else {
					alias = paramName.substring(6);
				}

				if (!attributes.containsKey(alias)) {
					attributes.put(alias, getAttributeValues(alias));
				}
			}
		}

		return attributes;
	}

	/**
	 * Gets the number of values provided in the fetch response for the
	 * specified attribute alias.
	 * 
	 * @param alias
	 *            The attribute alias.
	 */
	public int getCount(String alias) {
		if (parameters.hasParameter("count." + alias)) {
			return Integer.parseInt(parameters.getParameterValue("count."
				+ alias));
		} else if (parameters.hasParameter("value." + alias)) {
			return 1;
		} else {
			return 0;
		}
	}

	/**
	 * Sets the number of values provided in the fetch response for the
	 * specified attribute alias.
	 * 
	 * @param alias
	 *            The attribute alias.
	 * @param count
	 *            The number of values.
	 */
	private void setCount(String alias, int count) {
		// make sure that count.< alias >.1 is removed
		parameters.removeParameters("count." + alias);

		if (count > 1) {
			parameters.set(new Parameter("count." + alias, Integer
				.toString(count)));
		}
	}

	/**
	 * Sets the optional 'update_url' parameter where the OP can later re-post
	 * fetch-response updates for the values of the requested attributes.
	 * 
	 * @param updateUrl
	 *            The URL where the RP accepts later updates for the requested
	 *            attributes.
	 */
	public void setUpdateUrl(String updateUrl) throws MessageException {
		try {
			new URL(updateUrl);
		} catch (MalformedURLException e) {
			throw new MessageException("Invalid update_url: " + updateUrl);
		}

		if (DEBUG) {
			LOG.debug("Setting fetch response update_url: " + updateUrl);
		}

		parameters.set(new Parameter("update_url", updateUrl));
	}

	/**
	 * Gets the optional 'update_url' parameter if available, or null otherwise.
	 */
	public String getUpdateUrl() {
		return parameters.hasParameter("update_url") ? parameters
			.getParameterValue("update_url") : null;
	}

	/**
	 * Checks the validity of the extension.
	 * <p>
	 * Used when constructing a extension from a parameter list.
	 * 
	 * @return True if the extension is valid, false otherwise.
	 */
	private boolean isValid() {
		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (!paramName.equals("mode")
				&& !paramName.startsWith("type.")
				&& !paramName.startsWith("count.")
				&& !paramName.startsWith("value.")
				&& !paramName.equals("update_url")) {
				LOG.warn("Invalid parameter name in fetch response: "
					+ paramName);
				return false;
			}
		}

		return checkAttributes();
	}

	private boolean checkAttributes() {
		for (String alias : getAttributeAliases()) {

			if (!parameters.hasParameter("type." + alias)) {
				LOG.warn("Type missing for attribute alias: " + alias);
				return false;
			}

			if (!parameters.hasParameter("count." + alias)) {
				if (!parameters.hasParameter("value." + alias)) {
					LOG.warn("Value missing for attribute alias: " + alias);
					return false;
				}
			} else // count.alias present
			{
				if (parameters.hasParameter("value." + alias)) {
					LOG.warn("Count parameter present for alias: "
						+ alias
						+ "; should use "
						+ alias
						+ ".[index] format");
					return false;
				}

				int count = getCount(alias);

				for (int i = 1; i <= count; i++) {
					if (!parameters.hasParameter("value."
						+ alias
						+ "."
						+ Integer.toString(i))) {
						LOG.warn("Value missing for alias: "
							+ alias
							+ "."
							+ Integer.toString(i));
						return false;
					}
				}
			}
		}

		return true;
	}
}
