/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message.ax;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sourceforge.tsukuyomi.openid.message.MessageException;
import jp.sourceforge.tsukuyomi.openid.message.Parameter;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Implements the extension for Attribute Exchange store requests.
 * 
 * @author Marius Scurtescu, Johnny Bufu
 */
public class StoreRequest extends AxMessage {
	private static final Log LOG = LogFactory.getLog(StoreRequest.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	/**
	 * Constructs a Store Request with an empty parameter list.
	 */
	protected StoreRequest() {
		parameters.set(new Parameter("mode", "store_request"));

		if (DEBUG) {
			LOG.debug("Created empty store request.");
		}
	}

	/**
	 * Constructs a Store Request with an empty parameter list.
	 */
	public static StoreRequest createStoreRequest() {
		return new StoreRequest();
	}

	/**
	 * Constructs a StoreRequest from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	protected StoreRequest(ParameterList params) {
		parameters = params;
	}

	/**
	 * Constructs a StoreRequest from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	public static StoreRequest createStoreRequest(ParameterList params)
			throws MessageException {
		StoreRequest req = new StoreRequest(params);

		if (!req.isValid()) {
			throw new MessageException("Invalid parameters for a store request");
		}

		if (DEBUG) {
			LOG.debug("Created store request from parameter list:\n" + params);
		}

		return req;
	}

	/**
	 * Adds an attribute to the store request.
	 * 
	 * @param alias
	 *            The identifier that will be associated with the attribute name
	 *            URI
	 * @param typeUri
	 *            The attribute name URI
	 * @param value
	 *            The value of the attribute
	 */
	public void addAttribute(String alias, String typeUri, String value) {
		int count = getCount(alias);

		String index = "";

		switch (count) {
		case 0:
			parameters.set(new Parameter("type." + alias, typeUri));
			break;

		case 1:
			// rename the existing one
			parameters.set(new Parameter(
				"value." + alias + ".1",
				getParameterValue("value." + alias)));
			parameters.removeParameters("value." + alias);
			index = ".2";
			break;

		default:
			index = "." + Integer.toString(count + 1);
		}

		parameters.set(new Parameter("value." + alias + index, value));
		setCount(alias, ++count);

		if (DEBUG) {
			LOG.debug("Added new attribute to store request; type: "
				+ typeUri
				+ " alias: "
				+ alias
				+ " count: "
				+ count);
		}
	}

	/**
	 * Returns a list with the attribute value(s) associated with the specified
	 * alias.
	 * 
	 * @param alias
	 *            Attribute alias.
	 * @return List of attribute values.
	 */
	public List<String> getAttributeValues(String alias) {
		List<String> values = new ArrayList<String>();

		if (!parameters.hasParameter("count." + alias)) {
			values.add(getParameterValue("value." + alias));
		} else {
			for (int i = 1; i <= getCount(alias); i++) {
				values.add(getParameterValue("value."
					+ alias
					+ "."
					+ Integer.toString(i)));
			}
		}

		return values;
	}

	/**
	 * Gets a list of attribute aliases.
	 */
	public List<String> getAttributeAliases() {
		List<String> aliases = new ArrayList<String>();

		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (paramName.startsWith("value.")) {
				String alias;
				if (paramName.endsWith(".")) {
					alias = paramName.substring(6, paramName.length() - 1);
				} else {
					alias = paramName.substring(6);
				}

				if (!aliases.contains(alias)) {
					aliases.add(alias);
				}
			}
		}

		return aliases;
	}

	/**
	 * Gets a map with attribute aliases -> list of values.
	 */
	public Map<String, Object> getAttributes() {
		Map<String, Object> attributes = new HashMap<String, Object>();

		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (paramName.startsWith("value.")) {
				String alias;
				if (paramName.endsWith(".")) {
					alias = paramName.substring(6, paramName.length() - 1);
				} else {
					alias = paramName.substring(6);
				}

				if (!attributes.containsKey(alias)) {
					attributes.put(alias, getAttributeValues(alias));
				}
			}
		}

		return attributes;
	}

	/**
	 * Gets the number of values provided in the fetch response for the
	 * specified attribute alias.
	 * 
	 * @param alias
	 *            The attribute alias.
	 */
	public int getCount(String alias) {
		if (parameters.hasParameter("count." + alias)) {
			return Integer.parseInt(parameters.getParameterValue("count."
				+ alias));
		} else if (parameters.hasParameter("value." + alias)) {
			return 1;
		} else {
			return 0;
		}
	}

	/**
	 * Sets the number of values provided in the fetch response for the
	 * specified attribute alias.
	 * 
	 * @param alias
	 *            The attribute alias.
	 * @param count
	 *            The number of values.
	 */
	private void setCount(String alias, int count) {
		// make sure that count.< alias >.1 is removed
		parameters.removeParameters("count." + alias);

		if (count > 1) {
			parameters.set(new Parameter("count." + alias, Integer
				.toString(count)));
		}
	}

	/**
	 * Checks the validity of the extension.
	 * <p>
	 * Used when constructing a extension from a parameter list.
	 * 
	 * @return True if the extension is valid, false otherwise.
	 */
	public boolean isValid() {
		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (!paramName.equals("mode")
				&& !paramName.startsWith("type.")
				&& !paramName.startsWith("value.")
				&& !paramName.startsWith("count.")) {
				LOG.warn("Invalid parameter name in store request: "
					+ paramName);
				// return false;
			}
		}

		return checkAttributes();
	}

	private boolean checkAttributes() {
		for (String alias : getAttributeAliases()) {
			if (!parameters.hasParameter("type." + alias)) {
				LOG.warn("Type missing for attribute alias: " + alias);
				return false;
			}

			if (!parameters.hasParameter("count." + alias)) {
				if (!parameters.hasParameter("value." + alias)) {
					LOG.warn("Value missing for attribute alias: " + alias);
					return false;
				}
			} else // count.alias present
			{
				if (parameters.hasParameter("value." + alias)) {
					LOG.warn("Count parameter present for alias: "
						+ alias
						+ "; should use "
						+ alias
						+ ".[index] format");
					return false;
				}

				int count = getCount(alias);

				for (int i = 1; i <= count; i++) {
					if (!parameters.hasParameter("value."
						+ alias
						+ "."
						+ Integer.toString(i))) {
						LOG.warn("Value missing for alias: "
							+ alias
							+ "."
							+ Integer.toString(i));
						return false;
					}
				}
			}
		}

		return true;
	}
}
