/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message.ax;

import jp.sourceforge.tsukuyomi.openid.message.MessageException;
import jp.sourceforge.tsukuyomi.openid.message.Parameter;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Implements the extension for Attribute Exchange store responses.
 * 
 * @author Marius Scurtescu, Johnny Bufu
 */
public class StoreResponse extends AxMessage {
	private static final Log LOG = LogFactory.getLog(StoreResponse.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	/**
	 * Constructs a Store Response with an empty parameter list.
	 */
	protected StoreResponse() {
		parameters.set(new Parameter("mode", "store_response"));

		if (DEBUG) {
			LOG.debug("Created empty store request.");
		}
	}

	/**
	 * Constructs a Store Response with an empty parameter list.
	 */
	public static StoreResponse createStoreResponse() {
		return new StoreResponse();
	}

	/**
	 * Constructs a StoreResponse from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	protected StoreResponse(ParameterList params) {
		super(params);
	}

	/**
	 * Constructs a StoreResponse from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	public static StoreResponse createStoreResponse(ParameterList params)
			throws MessageException {
		StoreResponse resp = new StoreResponse(params);

		if (!resp.isValid()) {
			throw new MessageException(
				"Invalid parameters for a store response");
		}

		if (DEBUG) {
			LOG.debug("Created store response from parameter list:\n" + params);
		}

		return resp;
	}

	/**
	 * Marks the Store Response as a failure, by setting the appropirate
	 * parameters.
	 * 
	 * @param description
	 *            Describes the error condition leading to the failure response
	 */
	protected void setFailure(String description) {
		parameters.set(new Parameter("status", "failure"));

		if (description != null) {
			parameters.set(new Parameter("status.description", description));
		}
	}

	/**
	 * Returns true if the Store Response is a failure message, true if it is a
	 * success response.
	 */
	public boolean hasFailed() {
		return parameters.hasParameter("status")
			&& "failure".equals(parameters.getParameterValue("status"));
	}

	/**
	 * Gets the status of the Store Response if the 'status' parameter is part
	 * of the response, or null otherwise.
	 */
	public String getStatus() {
		return parameters.hasParameter("status") ? parameters
			.getParameterValue("status") : null;
	}

	/**
	 * Gets the status of the Store Response if the 'status.description'
	 * parameter is part of the response, or null otherwise.
	 */
	public String getStatusDescription() {
		return parameters.hasParameter("status.description") ? parameters
			.getParameterValue("status.description") : null;
	}

	/**
	 * Checks the validity of the extension.
	 * <p>
	 * Used when constructing a extension from a parameter list.
	 * 
	 * @return True if the extension is valid, false otherwise.
	 */
	private boolean isValid() {
		if (getStatusDescription() != null && getStatus() == null) {
			return false;
		}

		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (!paramName.equals("mode")
				&& !paramName.equals("status")
				&& !paramName.equals("status.description")) {
				LOG.warn("Invalid parameter name in store response: "
					+ paramName);
				return false;
			}

			if (paramName.equals("status")
				&& !"failure".equals(param.getValue())) {
				LOG.warn("Invalid status value: " + param.getValue());
				return false;
			}
		}

		return true;
	}

}
