/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.message.sreg;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sourceforge.tsukuyomi.openid.message.MessageException;
import jp.sourceforge.tsukuyomi.openid.message.Parameter;
import jp.sourceforge.tsukuyomi.openid.message.ParameterList;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Implements the extension for Simple Registration fetch responses.
 * 
 * @author Marius Scurtescu, Johnny Bufu
 */
public class SRegResponse extends SRegMessage {
	private static final Log LOG = LogFactory.getLog(SRegResponse.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	protected final static List<String> SREG_FIELDS =
		Arrays.asList(new String[] {
			"nickname",
			"email",
			"fullname",
			"dob",
			"gender",
			"postcode",
			"country",
			"language",
			"timezone" });

	/**
	 * Constructs a SReg Response with an empty parameter list.
	 */
	protected SRegResponse() {
		if (DEBUG) {
			LOG.debug("Created empty fetch response.");
		}
	}

	/**
	 * Constructs a SReg Response with an empty parameter list.
	 */
	public static SRegResponse createFetchResponse() {
		return new SRegResponse();
	}

	/**
	 * Constructs a SReg Response from a parameter list.
	 * <p>
	 * The parameter list can be extracted from a received message with the
	 * getExtensionParams method of the Message class, and MUST NOT contain the
	 * "openid.<extension_alias>." prefix.
	 */
	protected SRegResponse(ParameterList params) {
		parameters = params;
	}

	public static SRegResponse createSRegResponse(ParameterList params)
			throws MessageException {
		SRegResponse resp = new SRegResponse(params);

		if (!resp.isValid()) {
			throw new MessageException("Invalid parameters for a SReg response");
		}

		if (DEBUG) {
			LOG.debug("Created SReg response from parameter list:\n" + params);
		}

		return resp;
	}

	/**
	 * Creates a SRegResponse from a SRegRequest message and the data released
	 * by the user.
	 * 
	 * @param req
	 *            SRegRequest message.
	 * @param userData
	 *            Map<String attributeName, String attributeValue> with the
	 *            data released by the user.
	 * @return Properly formed SRegResponse.
	 * @throws MessageException
	 *             if any attribute-name in the userData map does not correspond
	 *             to an SREG field-name.
	 */
	public static SRegResponse createSRegResponse(SRegRequest req,
			Map<String, String> userData) throws MessageException {
		SRegResponse resp = new SRegResponse();

		List<String> attributes = req.getAttributes();
		for (String attr : attributes) {
			String value = userData.get(attr);
			if (value != null) {
				resp.addAttribute(attr, value);
			}
		}

		return resp;
	}

	/**
	 * Adds an attribute to the SReg response. The allowed attribute names are
	 * the ones defined in the SReg specification: nickname, email, fullname,
	 * dob, gender, postcode, country, language, timezone.
	 * 
	 * @param attr
	 *            An attribute name.
	 * @param value
	 *            The value of the attribute.
	 */
	public void addAttribute(String attr, String value) throws MessageException {
		parameters.set(new Parameter(attr, value));

		if (!SREG_FIELDS.contains(attr)) {
			throw new MessageException("Invalid attribute for SReg: " + attr);
		}

		if (DEBUG) {
			LOG.debug("Added new attribute to SReg response: "
				+ attr
				+ " value: "
				+ value);
		}
	}

	/**
	 * Returns the value of an attribute.
	 * 
	 * @param attr
	 *            The attribute name.
	 * @return The attribute value.
	 */
	public String getAttributeValue(String attr) {
		return getParameterValue(attr);
	}

	/**
	 * Gets a list of attribute names in the SReg response.
	 */
	public List<String> getAttributeNames() {
		List<String> attributes = new ArrayList<String>();

		for (Parameter param : parameters.getParameters()) {
			attributes.add(param.getKey());
		}

		return attributes;
	}

	/**
	 * Gets a map with attribute names -> values.
	 */
	public Map<String, String> getAttributes() {
		Map<String, String> attributes = new HashMap<String, String>();

		for (Parameter param : parameters.getParameters()) {
			String attr = param.getKey();
			attributes.put(attr, getAttributeValue(attr));
		}

		return attributes;
	}

	/**
	 * Checks the validity of the extension.
	 * <p>
	 * Used when constructing a extension from a parameter list.
	 * 
	 * @return True if the extension is valid, false otherwise.
	 */
	private boolean isValid() {
		for (Parameter param : parameters.getParameters()) {
			String paramName = param.getKey();

			if (!SREG_FIELDS.contains(paramName)) {
				LOG.warn("Invalid parameter name in SReg response: "
					+ paramName);
				return false;
			}
		}

		return true;
	}
}
