/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.op;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public class RealmVerifier {
	private static final Log LOG = LogFactory.getLog(RealmVerifier.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	public static final int OK = 0;
	public static final int DENIED_REALM = 1;
	public static final int MALFORMED_REALM = 2;
	public static final int MALFORMED_RETURN_TO_URL = 3;
	public static final int FRAGMENT_NOT_ALLOWED = 4;
	public static final int PROTOCOL_MISMATCH = 5;
	public static final int PORT_MISMATCH = 6;
	public static final int PATH_MISMATCH = 7;
	public static final int DOMAIN_MISMATCH = 8;

	private List<String> deniedRealmDomains;
	private List<Pattern> deniedRealmRegExps;

	public RealmVerifier() {
		deniedRealmDomains = new ArrayList<String>();

		addDeniedRealmDomain("\\*\\.[^\\.]+");
		addDeniedRealmDomain("\\*\\.[a-z]{2}\\.[a-z]{2}");
	}

	public void addDeniedRealmDomain(String deniedRealmDomain) {
		deniedRealmDomains.add(deniedRealmDomain);

		compileDeniedRealms();
	}

	public List<String> getDeniedRealmDomains() {
		return deniedRealmDomains;
	}

	public void setDeniedRealmDomains(List<String> deniedRealmDomains) {
		this.deniedRealmDomains = deniedRealmDomains;

		compileDeniedRealms();
	}

	private void compileDeniedRealms() {
		deniedRealmRegExps = new ArrayList<Pattern>(deniedRealmDomains.size());

		for (String deniedRealm : deniedRealmDomains) {
			Pattern deniedRealmPattern =
				Pattern.compile(deniedRealm, Pattern.CASE_INSENSITIVE);

			deniedRealmRegExps.add(deniedRealmPattern);
		}
	}

	public int match(String realm, String returnTo) {
		if (DEBUG) {
			LOG.debug("Verifying realm: "
				+ realm
				+ " on return URL: "
				+ returnTo);
		}

		URL realmUrl;
		try {
			realmUrl = new URL(realm);
		} catch (MalformedURLException e) {
			LOG.error("Invalid realm URL: " + realm, e);
			return MALFORMED_REALM;
		}

		String realmDomain = realmUrl.getHost();

		if (isDeniedRealmDomain(realmDomain)) {
			LOG.warn("Blacklisted realm domain: " + realmDomain);
			return DENIED_REALM;
		}

		URL returnToUrl;
		try {
			returnToUrl = new URL(returnTo);
		} catch (MalformedURLException e) {
			LOG.error("Invalid return URL: " + returnTo);
			return MALFORMED_RETURN_TO_URL;
		}

		if (realmUrl.getRef() != null) {
			if (DEBUG) {
				LOG.debug("Realm verification failed: "
					+ "URL fragments are not allowed.");
			}
			return FRAGMENT_NOT_ALLOWED;
		}

		if (!realmUrl.getProtocol().equalsIgnoreCase(returnToUrl.getProtocol())) {
			if (DEBUG) {
				LOG.debug("Realm verification failed: " + "protocol mismatch.");
			}
			return PROTOCOL_MISMATCH;
		}

		if (!domainMatch(realmDomain, returnToUrl.getHost())) {
			if (DEBUG) {
				LOG.debug("Realm verification failed: " + "domain mismatch.");
			}
			return DOMAIN_MISMATCH;
		}

		if (!portMatch(realmUrl, returnToUrl)) {
			if (DEBUG) {
				LOG.debug("Realm verification failed: " + "port mismatch.");
			}
			return PORT_MISMATCH;
		}

		if (!pathMatch(realmUrl, returnToUrl)) {
			if (DEBUG) {
				LOG.debug("Realm verification failed: " + "path mismatch.");
			}
			return PATH_MISMATCH;
		}

		LOG.info("Realm verified: " + realm);

		return OK;
	}

	private boolean isDeniedRealmDomain(String realmDomain) {
		for (int i = 0; i < deniedRealmRegExps.size(); i++) {
			Pattern realmPattern = deniedRealmRegExps.get(i);

			if (realmPattern.matcher(realmDomain).matches()) {
				return true;
			}
		}

		return false;
	}

	private boolean portMatch(URL realmUrl, URL returnToUrl) {
		int realmPort = realmUrl.getPort();
		int returnToPort = returnToUrl.getPort();

		if (realmPort == -1) {
			realmPort = realmUrl.getDefaultPort();
		}

		if (returnToPort == -1) {
			returnToPort = returnToUrl.getDefaultPort();
		}

		return realmPort == returnToPort;
	}

	/**
	 * Does the URL's path equal to or a sub-directory of the realm's path.
	 * 
	 * @param realmUrl
	 * @param returnToUrl
	 * @return If equals or a sub-direcotory return true.
	 */
	private boolean pathMatch(URL realmUrl, URL returnToUrl) {
		String realmPath = realmUrl.getPath();
		String returnToPath = returnToUrl.getPath();

		if (!realmPath.endsWith("/")) {
			realmPath += "/";
		}

		if (!returnToPath.endsWith("/")) {
			returnToPath += "/";
		}

		// return realmPath.startsWith(returnToPath);
		return returnToPath.startsWith(realmPath);
	}

	private boolean domainMatch(String realmDomain, String returnToDomain) {
		if (realmDomain.startsWith("*.")) {
			realmDomain = realmDomain.substring(1).toLowerCase();
			returnToDomain = "." + returnToDomain.toLowerCase();

			return returnToDomain.endsWith(realmDomain);
		} else {
			return realmDomain.equalsIgnoreCase(returnToDomain);
		}
	}
}
