/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.rp;

import java.text.ParseException;
import java.util.Date;

import jp.sourceforge.tsukuyomi.openid.util.InternetDateFormat;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public abstract class AbstractNonceVerifier implements NonceVerifier {
	private static final Log LOG =
		LogFactory.getLog(AbstractNonceVerifier.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	protected int maxAge;

	/**
	 * @param maxAge
	 *            maximum token age in seconds
	 */
	protected AbstractNonceVerifier(int maxAge) {
		this.maxAge = maxAge;
	}

	public int getMaxAge() {
		return maxAge / 1000;
	}

	/**
	 * Checks if nonce date is valid and if it is in the max age boudary. Other
	 * checks are delegated to {@link #seen(java.util.Date, String, String)}
	 */
	public int seen(String idpUrl, String nonce) {
		if (DEBUG) {
			LOG.debug("Verifying nonce: " + nonce);
		}

		Date now = new Date();

		try {
			InternetDateFormat dateFormat = new InternetDateFormat();

			Date nonceDate = dateFormat.parse(nonce);

			if (isTooOld(now, nonceDate)) {
				LOG.warn("Nonce is too old: " + nonce);
				return TOO_OLD;
			}

			return seen(now, idpUrl, nonce);
		} catch (ParseException e) {
			LOG.error("Error verifying the nonce: " + nonce, e);
			return INVALID_TIMESTAMP;
		}
	}

	/**
	 * Subclasses should implement this method and check if the nonce was seen
	 * before. The nonce timestamp was verified at this point, it is valid and
	 * it is in the max age boudary.
	 * 
	 * @param now
	 *            The timestamp used to check the max age boudary.
	 */
	protected abstract int seen(Date now, String idpUrl, String nonce);

	protected boolean isTooOld(Date now, Date nonce) {
		long age = now.getTime() - nonce.getTime();

		return age > maxAge * 1000;
	}
}
