/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.rp.impl;

import java.text.ParseException;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import jp.sourceforge.tsukuyomi.openid.rp.AbstractNonceVerifier;
import jp.sourceforge.tsukuyomi.openid.util.InternetDateFormat;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Marius Scurtescu, Johnny Bufu
 */
public class InMemoryNonceVerifier extends AbstractNonceVerifier {
	private static final Log LOG =
		LogFactory.getLog(InMemoryNonceVerifier.class);
	private static final boolean DEBUG = LOG.isDebugEnabled();

	private Map<String, Set<String>> idpMap =
		new HashMap<String, Set<String>>();

	public InMemoryNonceVerifier(int maxAge) {
		super(maxAge);
	}

	public InMemoryNonceVerifier() {
		super(60);
	}

	@Override
	protected synchronized int seen(Date now, String idpUrl, String nonce) {
		removeAged(now);

		Set<String> seenSet = idpMap.get(idpUrl);

		if (seenSet == null) {
			seenSet = new HashSet<String>();

			idpMap.put(idpUrl, seenSet);
		}

		if (seenSet.contains(nonce)) {
			LOG.error("Possible replay attack! Already seen nonce: " + nonce);
			return SEEN;
		}

		seenSet.add(nonce);

		if (DEBUG) {
			LOG.debug("Nonce verified: " + nonce);
		}

		return OK;
	}

	private synchronized void removeAged(Date now) {
		InternetDateFormat dateFormat = new InternetDateFormat();
		Set<String> idpToRemove = new HashSet<String>();
		for (Entry<String, Set<String>> ent : idpMap.entrySet()) {
			String idpUrl = ent.getKey();

			Set<String> seenSet = ent.getValue();
			Set<String> nonceToRemove = new HashSet<String>();

			for (String nonce : seenSet) {
				try {
					Date nonceDate = dateFormat.parse(nonce);

					if (isTooOld(now, nonceDate)) {
						nonceToRemove.add(nonce);
					}
				} catch (ParseException e) {
					nonceToRemove.add(nonce);
				}
			}

			for (String nonce : nonceToRemove) {
				if (DEBUG) {
					LOG.debug("Removing nonce: "
						+ nonce
						+ " from OP: "
						+ idpUrl);
				}
				seenSet.remove(nonce);
			}

			if (seenSet.size() == 0) {
				idpToRemove.add(idpUrl);
			}
		}

		for (String idpUrl : idpToRemove) {
			if (DEBUG) {
				LOG.debug("Removed all nonces from OP: " + idpUrl);
			}

			idpMap.remove(idpUrl);
		}
	}

	public synchronized int size() {
		int total = 0;

		for (Set<String> seenSet : idpMap.values()) {
			total += seenSet.size();
		}

		return total;
	}
}
