/*
 * Copyright 2008-2009 the Project Tsukuyomi and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.tsukuyomi.openid.yadis;

import java.net.MalformedURLException;
import java.net.URL;

import org.openxri.XRI;

/**
 * Wrapper class for various identifiers that are resolvable to URLs and can be
 * used as YadisURLs with the Yadis protocol.
 * 
 * @author Marius Scurtescu, Johnny Bufu
 */
public class YadisUrl {
	/**
	 * A YadisURL is a regular URL, with a couple restrictions.
	 */
	private URL _yadisUrl;

	/**
	 * Constructs a URL object from a string; needed by the YadisURL(String)
	 * constructor
	 * 
	 * @param urlString
	 *            URL-type identifier in string format
	 * @return URL object
	 * @throws YadisException
	 *             if the provided string is not a valid URL
	 */
	private static URL urlFromString(String urlString) throws YadisException {
		URL url;

		try {
			url = new URL(urlString);
		} catch (MalformedURLException e) {
			throw new YadisException(
				"Invalid URL: " + urlString,
				YadisResult.INVALID_URL,
				e);
		}

		return url;
	}

	/**
	 * Contructs a YadisURL from a string; assumes the string to be a URL-type
	 * identifier
	 * 
	 * @param urlString
	 *            URL-type identifier in string format
	 * @throws YadisException
	 *             if the provided string cannot be a YadisUrl
	 */
	public YadisUrl(String urlString) throws YadisException {
		this(urlFromString(urlString));

		if (!isValid(this._yadisUrl)) {
			throw new YadisException("The scheme name of a Yadis URL "
				+ "must be 'http' or 'https'", YadisResult.INVALID_SCHEME);
		}

	}

	/**
	 * Constructs a YadisURL from a URL object; insures the schema is HTTP or
	 * HTTPS
	 * 
	 * @param urlId
	 *            URL identifier
	 * @throws YadisException
	 *             tf the URL identifier is not a valid YadisURL
	 */
	public YadisUrl(URL urlId) throws YadisException {
		if (isValid(urlId)) {
			_yadisUrl = urlId;
		} else {
			throw new YadisException("The scheme name of a Yadis URL "
				+ "must be 'http' or 'https'", YadisResult.INVALID_SCHEME);
		}
	}

	/**
	 * Validates a URL against the requirements for a YadisUrl.
	 * <p>
	 * The URL must be absolute (the schema must be specified), and the schema
	 * must be HTTP or HTTPS.
	 * 
	 * @param url
	 *            the URL to be validated
	 * @return true if the URL is a valid YadisUrl, or false otherwise
	 */
	private boolean isValid(URL url) {
		return url.getProtocol().equalsIgnoreCase("http")
			|| url.getProtocol().equalsIgnoreCase("https");
	}

	/**
	 * Constructs a YadisURL from an XRI identifier.
	 * 
	 * @param xriId
	 *            The XRI identifier
	 */
	public YadisUrl(XRI xriId) throws YadisException {
		this(urlFromString(xriId.toURINormalForm()));
	}

	/**
	 * Gets the URL to be used in Yadis transactions.
	 */
	public URL getUrl() {
		return _yadisUrl;
	}

	/**
	 * Gets a string representation of the YadisURL.
	 */
	@Override
	public String toString() {
		return _yadisUrl.toString();
	}
}
