/*
 *                 Sun Public License Notice
 * 
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 * 
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2001 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.examples.modules.minicomposer;

import java.awt.Dialog;
import java.awt.datatransfer.*;
import java.io.*;
import java.util.*;
import javax.swing.*;

import org.openide.*;
import org.openide.util.HelpCtx;
import org.openide.util.NbBundle;
import org.openide.util.datatransfer.*;

/** Utilities dealing with data transfer operations on notes.
 */
public final class NoteTransfer implements ExClipboard.Convertor {
    public static final DataFlavor SCORE_FLAVOR = new DataFlavor(Score.class, NbBundle.getMessage(NoteTransfer.class, "LBL_note_flavor"));
    public NoteTransfer() {}
    /** Convert a transferable.
     * If it had just a text selection, make a corresponding score by parsing it.
     * If just a score, make a corresponding text selection by writing it out.
     * If it had a multiple selection all of which had scores, make a single score
     * of them all concatenated, as welll a text selection with them all concatenated.
     * Otherwise leave it alone.
     */
    public Transferable convert(final Transferable t) {
        boolean supportsString = t.isDataFlavorSupported(DataFlavor.stringFlavor);
        boolean supportsScore = t.isDataFlavorSupported(SCORE_FLAVOR);
        if (supportsString && !supportsScore) {
            ExTransferable t2 = ExTransferable.create(t);
            t2.put(new ExTransferable.Single(SCORE_FLAVOR) {
                protected Object getData() throws IOException, UnsupportedFlavorException {
                    String text = (String)t.getTransferData(DataFlavor.stringFlavor);
                    return Score.parse(new StringReader(text));
                }
            });
            return t2;
        } else if (!supportsString && supportsScore) {
            ExTransferable t2 = ExTransferable.create(t);
            t2.put(new ExTransferable.Single(DataFlavor.stringFlavor) {
                protected Object getData() throws IOException, UnsupportedFlavorException {
                    Score s = (Score)t.getTransferData(SCORE_FLAVOR);
                    StringWriter wr = new StringWriter();
                    Score.generate(s, wr);
                    return wr.toString();
                }
            });
            return t2;
        } else if (t.isDataFlavorSupported(ExTransferable.multiFlavor)) {
            try {
                final MultiTransferObject mto = (MultiTransferObject)t.getTransferData(ExTransferable.multiFlavor);
                boolean allSupportScore = true;
                for (int i = 0; i < mto.getCount(); i++) {
                    if (!mto.isDataFlavorSupported(i, SCORE_FLAVOR)) {
                        allSupportScore = false;
                        break;
                    }
                }
                if (allSupportScore) {
                    ExTransferable t2 = ExTransferable.create(t);
                    if (!supportsString) {
                        t2.put(new ExTransferable.Single(DataFlavor.stringFlavor) {
                            protected Object getData() throws IOException, UnsupportedFlavorException {
                                StringWriter wr = new StringWriter();
                                for (int i = 0; i < mto.getCount(); i++) {
                                    Score s = (Score)mto.getTransferData(i, SCORE_FLAVOR);
                                    Score.generate(s, wr);
                                }
                                return wr.toString();
                            }
                        });
                    }
                    if (!supportsScore) {
                        t2.put(new ExTransferable.Single(SCORE_FLAVOR) {
                            protected Object getData() throws IOException, UnsupportedFlavorException {
                                List notes = new ArrayList();
                                for (int i = 0; i < mto.getCount(); i++) {
                                    Score s = (Score)mto.getTransferData(i, SCORE_FLAVOR);
                                    int size = s.getSize();
                                    for (int j = 0; j < size; j++) {
                                        notes.add(s.getNote(j));
                                    }
                                }
                                return new Score(notes);
                            }
                        });
                    }
                    return t2;
                }
            } catch (Exception e) {
                // Should not happen: IOException, UnsupportedFlavorException
                TopManager.getDefault().getErrorManager().notify(ErrorManager.INFORMATIONAL, e);
            }
        }
        return t;
    }
    /** Create a paste type from a transferable.
     * For access from ScoreDataNode as well as NoteNode.
     * @param t the transferable to check
     * @param after if not null, a note to start pasting from
     * @return an appropriate paste type, or null if not appropriate
     */
    public static PasteType createNotePasteType(Transferable t, ScoreCookie cookie, Score.Note after) {
        if (t.isDataFlavorSupported(SCORE_FLAVOR)) {
            return new NotePaste(t, cookie, after);
        } else {
            return null;
        }
    }
    private static final class NotePaste extends PasteType {
        private final Transferable t;
        private final ScoreCookie cookie;
        private final Score.Note after;
        public NotePaste(Transferable t, ScoreCookie cookie, Score.Note after) {
            this.t = t;
            this.cookie = cookie;
            this.after = after;
        }
        public String getName() {
            if (after != null) {
                return NbBundle.getMessage(NoteTransfer.class, "LBL_note_paste_after");
            } else {
                return NbBundle.getMessage(NoteTransfer.class, "LBL_note_paste");
            }
        }
        public HelpCtx getHelpCtx() {
            return new HelpCtx("org.netbeans.examples.modules.minicomposer.explorer");
        }
        public Transferable paste() throws IOException {
            Score s = cookie.getScore();
            int c = s.getSize();
            List notes = new ArrayList(c + 100);
            boolean pasted = false;
            if (after == null) {
                // Paste at the front.
                append(notes);
                pasted = true;
            }
            for (int i = 0; i < c; i++) {
                Score.Note n = s.getNote(i);
                notes.add(n);
                if (!pasted && after == n) {
                    append(notes);
                    pasted = true;
                }
            }
            cookie.setScore(new Score(notes));
            return null;
        }
        private void append(List notes) throws IOException {
            try {
                Score pasted = (Score)t.getTransferData(SCORE_FLAVOR);
                int size = pasted.getSize();
                for (int i = 0; i < size; i++) {
                    notes.add(pasted.getNote(i).cloneNote());
                }
            } catch (UnsupportedFlavorException ufe) {
                // Should not happen.
                IOException ioe = new IOException(ufe.toString());
                TopManager.getDefault().getErrorManager().annotate(ioe, ufe);
                throw ioe;
            }
        }
    }
    public static class NewNote extends NewType {
        private final ScoreCookie cookie;
        public NewNote(ScoreCookie cookie) {
            this.cookie = cookie;
        }
        public String getName() {
            return NbBundle.getMessage(NoteTransfer.class, "LBL_new_note");
        }
        public HelpCtx getHelpCtx() {
            return new HelpCtx("org.netbeans.examples.modules.minicomposer.explorer");
        }
        public void create() throws IOException {
            Score s = cookie.getScore();
            Score.Note n = requestNote();
            if (n == null) return;
            int c = s.getSize();
            List l = new ArrayList(c + 1);
            for (int i = 0; i < c; i++) {
                l.add(s.getNote(i));
            }
            l.add(n);
            cookie.setScore(new Score(l));
        }
        private Score.Note requestNote() {
            JPanel panel = new JPanel();
            JLabel label = new JLabel(NbBundle.getMessage(NoteTransfer.class, "LBL_choose_tone"));
            panel.add(label);
            JComboBox tone = new JComboBox(Score.TONES_LONG);
            tone.setSelectedIndex(Score.DEFAULT_TONE);
            tone.setEditable(false);
            label.setLabelFor(tone);
            panel.add(tone);
            label = new JLabel(NbBundle.getMessage(NoteTransfer.class, "LBL_choose_octave"));
            panel.add(label);
            JComboBox octave = new JComboBox(Score.OCTAVES_LONG);
            octave.setSelectedIndex(Score.DEFAULT_OCTAVE);
            octave.setEditable(false);
            label.setLabelFor(octave);
            panel.add(octave);
            label = new JLabel(NbBundle.getMessage(NoteTransfer.class, "LBL_choose_duration"));
            panel.add(label);
            JComboBox duration = new JComboBox(Score.DURATIONS_LONG);
            duration.setSelectedIndex(Score.DEFAULT_DURATION);
            duration.setEditable(false);
            label.setLabelFor(duration);
            panel.add(duration);
            DialogDescriptor d = new DialogDescriptor(panel, NbBundle.getMessage(NoteTransfer.class, "TITLE_add_note"));
            d.setModal(true);
            d.setHelpCtx(getHelpCtx());
            d.setMessageType(NotifyDescriptor.QUESTION_MESSAGE);
            d.setOptionType(NotifyDescriptor.OK_CANCEL_OPTION);
            Dialog dlg = TopManager.getDefault().createDialog(d);
            dlg.pack();
            dlg.show();
            if (d.getValue() == NotifyDescriptor.OK_OPTION) {
                return new Score.Note(tone.getSelectedIndex(), octave.getSelectedIndex(), duration.getSelectedIndex());
            } else {
                // Closed dialog or pressed Cancel
                return null;
            }
        }
    }
}
