/*
 *                 Sun Public License Notice
 * 
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 * 
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2001 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.examples.modules.minicomposer;

import java.awt.Image;
import java.awt.datatransfer.Transferable;
import java.beans.*;
import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import javax.swing.event.*;

import org.openide.ErrorManager;
import org.openide.TopManager;
import org.openide.actions.OpenAction;
import org.openide.compiler.Compiler;
import org.openide.compiler.CompilerJob;
import org.openide.cookies.CompilerCookie;
import org.openide.cookies.EditorCookie;
import org.openide.filesystems.*;
import org.openide.loaders.*;
import org.openide.nodes.*;
import org.openide.text.CloneableEditorSupport;
import org.openide.util.*;
import org.openide.util.actions.SystemAction;
import org.openide.util.datatransfer.*;

public class ScoreDataNode extends DataNode implements ChangeListener, PropertyChangeListener, FileChangeListener {
    private static final ErrorManager err =
        TopManager.getDefault().getErrorManager().getInstance("org.netbeans.examples.modules.minicomposer.ScoreDataNode"); // NOI18N
    /** Whether this score has actually been opened in an editor.
     * If not, we will not bother to ask for a parse yet.
     */
    private boolean inUse = false;
    public ScoreDataNode(ScoreDataObject obj) {
        this(obj, (ScoreCookie)obj.getCookie(ScoreCookie.class));
    }
    private ScoreDataNode(ScoreDataObject obj, ScoreCookie score) {
        this(obj, score, new ScoreChildren(score));
    }
    private ScoreDataNode(ScoreDataObject obj, ScoreCookie score, ScoreChildren children) {
        super(obj, children);
        getCookieSet().add(children.getIndex());
        score.addChangeListener(WeakListener.change(this, score));
        EditorCookie ed = (EditorCookie)getCookie(EditorCookie.class);
        if (ed != null && (ed instanceof CloneableEditorSupport)) {
            ((CloneableEditorSupport)ed).addChangeListener(WeakListener.change(this, ed));
        } else {
            err.log(ErrorManager.WARNING, "Warning: no CloneableEditorSupport found on " + getDisplayName());
        }
        obj.addPropertyChangeListener(WeakListener.propertyChange(this, obj));
        updateFileListeners();
    }
    protected Sheet createSheet() {
        Sheet sheet = super.createSheet();
        Sheet.Set properties = sheet.get(Sheet.PROPERTIES);
        properties.put(new ClipLengthProperty((ScoreDataObject)getDataObject(), this));
        ExecSupport support = (ExecSupport)getCookie(ExecSupport.class);
        Sheet.Set set = new Sheet.Set();
        set.setName("execution");
        set.setDisplayName(NbBundle.getMessage(ScoreDataNode.class, "LBL_Execution"));
        set.setShortDescription(NbBundle.getMessage(ScoreDataNode.class, "HINT_Execution"));
        support.addProperties(set);
        set.remove(ExecSupport.PROP_DEBUGGER_TYPE);
        set.remove(ExecSupport.PROP_FILE_PARAMS);
        sheet.put(set);
        return sheet;
    }
    public Image getIcon(int type) {
        if (type == BeanInfo.ICON_COLOR_16x16 || type == BeanInfo.ICON_MONO_16x16) {
            Image icon = Utilities.loadImage("org/netbeans/examples/modules/minicomposer/ScoreDataIcon.gif");
            err.log("getIcon: " + getDisplayName());
            ScoreCookie cookie = (ScoreCookie)getCookie(ScoreCookie.class);
            if (cookie != null) {
                if (inUse) {
                    cookie.prepare();
                }
                if (!cookie.isValid()) {
                    err.log("getIcon: error");
                    Image badge = Utilities.loadImage("org/netbeans/examples/modules/minicomposer/error-badge.gif");
                    icon = Utilities.mergeImages(icon, badge, 8, 8);
                }
            }
            DataObject obj = getDataObject();
            err.log("modified=" + obj.isModified());
            err.log("compiler says IUD=" + new SampledAudioCompiler(obj.getPrimaryFile(), false).isUpToDate());
            if (obj.isModified() || outOfDate(obj)) {
                err.log("getIcon: out-of-date");
                Image badge = Utilities.loadImage("org/netbeans/examples/modules/minicomposer/out-of-date-badge.gif");
                icon = Utilities.mergeImages(icon, badge, 16, 0);
            }
            try {
                icon = obj.getPrimaryFile().getFileSystem().getStatus().annotateIcon(icon, type, obj.files());
            } catch (FileStateInvalidException fsie) {
                // Forget about it.
                err.notify(ErrorManager.INFORMATIONAL, fsie);
            }
            return icon;
        } else {
            return null;
        }
    }
    public Image getOpenedIcon(int type) {
        return getIcon(type);
    }
    private static boolean outOfDate(DataObject obj) {
        CompilerCookie.Compile cookie = (CompilerCookie.Compile)obj.getCookie(CompilerCookie.Compile.class);
        if (cookie == null) return true;
        CompilerJob job = new CompilerJob(Compiler.DEPTH_ZERO);
        cookie.addToJob(job, Compiler.DEPTH_ZERO);
        return !job.isUpToDate();
    }
    public void stateChanged(ChangeEvent ev) {
        if (!getDataObject().isValid()) return;
        Object src = ev.getSource();
        err.log("state change: source " + src);
        if (!inUse && (src instanceof EditorCookie)) {
            err.log("editor change: " + getDisplayName());
            inUse = true;
            fireIconChange();
        }
        if (src instanceof ScoreCookie) {
            err.log("icon change: state: " + getDisplayName());
            fireIconChange();
        }
    }
    public void propertyChange(PropertyChangeEvent ev) {
        if (!getDataObject().isValid()) return;
        String prop = ev.getPropertyName();
        err.log("prop change: " + prop + " on: " + getDisplayName());
        if (prop == null || prop.equals(DataObject.PROP_FILES) || prop.equals(DataObject.PROP_MODIFIED)) {
            err.log("icon change: files: " + getDisplayName());
            updateFileListeners();
            fireIconChange();
        }
    }
    private void updateFileListeners() {
        //err.log("adding listeners from the primary file " + getDataObject().getPrimaryFile());
        Iterator it = getDataObject().files().iterator();
        while (it.hasNext()) {
            FileObject fo = (FileObject)it.next();
            //err.log("adding listener to " + fo + " as " + getDisplayName());
            fo.addFileChangeListener(WeakListener.fileChange(this, fo));
        }
    }
    public void fileRenamed(FileRenameEvent fe) {
        err.log("icon change: rename: " + fe.getFile());
        fireIconChange();
    }
    public void fileChanged(FileEvent fe) {
        err.log("icon change: change: " + fe.getFile());
        fireIconChange();
    }
    public void fileDeleted(FileEvent fe) {
        if (!getDataObject().isValid()) return;
        err.log("icon change: deleted: " + fe.getFile());
        fireIconChange();
    }
    public void fileAttributeChanged(FileAttributeEvent fe) {
        // ignore
    }
    public void fileFolderCreated(FileEvent fe) {
        // ignore
    }
    public void fileDataCreated(FileEvent fe) {
        // ignore
    }
    // Access from ClipLengthProperty:
    void fireClipLengthChange() {
        firePropertyChange("clipLength", null, null);
    }
    public Transferable clipboardCopy() throws IOException {
        Transferable deflt = super.clipboardCopy();
        ExTransferable enriched = ExTransferable.create(deflt);
        enriched.put(new ExTransferable.Single(NoteTransfer.SCORE_FLAVOR) {
            protected Object getData() throws IOException {
                return ((ScoreCookie)getCookie(ScoreCookie.class)).getScore();
            }
        });
        return enriched;
    }
    protected void createPasteTypes(Transferable t, List s) {
        super.createPasteTypes(t, s);
        PasteType p = NoteTransfer.createNotePasteType(t, (ScoreCookie)getCookie(ScoreCookie.class), null);
        if (p != null) {
            s.add(p);
        }
    }
    public NewType[] getNewTypes() {
        return new NewType[] {new NoteTransfer.NewNote((ScoreCookie)getCookie(ScoreCookie.class))};
    }
}
