import threading,subprocess,os,signal,itertools,datetime,functools
from Queue import Queue,Empty
import shoutbrowser,config
from favoritelistdb import FavoriteListDataBase
import gettext
import gc

gettext.install('tuxshout','locale',unicode=True)
try:
    import urwid, urwid.raw_display
except:
    import sys
    print _('urwid module could not import!')
    print _('if urwid is not installed,please install!')
    if not config.is_module_installed('easy_install'):
        print _('you can not use "easy_install" ?')
        print _('please install them,and...')
    print _('type below')
    print _('easy_install urwid')
    sys.exit(-1)

version ='0.0.3c'

class ListBoxItem(urwid.Text):
    def __init__(self, label, user_data):
        self.__super = super(self.__class__,self)
        self.__super.__init__(label)
        self.user_data = user_data
        self.__is_active = False
        
    def set_focus(self,is_focused):
        if self.__is_active:return
        if is_focused:
            self.__set_color(SharedObject.palette.ITEM_FOCUS)
        else:
            self.__set_color(None)

    def leave_focus(self):
        if self.__is_active:
            self.__set_color(SharedObject.palette.ITEM_LEAVE_REV)
        else:
            self.__set_color(SharedObject.palette.ITEM_LEAVE)
    def enter_focus(self):
        if self.__is_active:
            self.__set_color(SharedObject.palette.ITEM_REVERSE)
        else:
            self.__set_color(SharedObject.palette.ITEM_FOCUS)

    def __set_color(self,color):
        if color:
            self.set_text(
                (color,
                 self.get_text()[0])
                )
        else:
            self.set_text(self.get_text()[0])

    @property
    def label(self):
        return self.get_text()[0]

    @property
    def is_active(self):
        return self.__is_active

    def activate(self,is_active=True):
        self.__is_active = is_active
        if is_active:
            self.__set_color(SharedObject.palette.ITEM_REVERSE)
        else:
            self.__is_already_focused = False
            self.__set_color(None)

    def selectable(self):
        return True

    def keypress(self,size,key):return key

class ListBoxItemWalker(urwid.SimpleListWalker):
    def __init__(self,contents,focus_callback):
        self._super = super(self.__class__,self)
        self._super.__init__(contents)
        self.focus_callback = focus_callback

    def _focus_item(self,widget,is_active):
        if widget:
            widget.set_focus(is_active)

    def set_focus(self,position):
        focus_out_item,cur_pos = self.get_focus()

        #Only different item can focus out
        if cur_pos != position:
            self._focus_item(focus_out_item,False)
            self._super.set_focus(position)

        focus_in_item = self.get_focus()[0]
        self._focus_item(focus_in_item,True)
        self.focus_callback(focus_in_item)
    
class MouseDrivenListBox(urwid.ListBox):
    def __init__(self,items,key_event_handler,on_focus_callback):
        self.__super = super(self.__class__,self)
        self.__super.__init__(ListBoxItemWalker(items,self.__on_focus))
        self.__on_focus_callback = on_focus_callback
        self.__key_event_handler = key_event_handler
        self.__active_item = None
        self.__prev_focused = None
        self.__focused_timedelta = None
        self.__double_click_timedelta = datetime.timedelta(
            milliseconds=config.double_click_speed)

    def keypress(self, size, key):
        handler = self.__key_event_handler.get(key,None)
        if handler:
            focused_item = self.get_focus()[0]
            focused_item.activate()
            self._toggle_active_item(focused_item)
            handler(focused_item)
        else:
            return self.__super.keypress(size,key)

    def _toggle_active_item(self,item):
        if item and item.is_active and \
                self.__active_item != item:
            if self.__active_item:
                self.__active_item.activate(False)
            self.__active_item = item

    def __is_double_click(self,item):
        is_double_click = False
        if self.__prev_focused == item:
            if self.__focused_timedelta:
                timedelta = datetime.datetime.now() - self.__focused_timedelta
                if timedelta < self.__double_click_timedelta:
                    is_double_click = True
        else:
            self.__prev_focused = item
        self.__focused_timedelta = datetime.datetime.now()
        return is_double_click

    def leave_focus(self):
        item = self.get_focus()[0]
        if item:
            item.leave_focus()

    def enter_focus(self):
        item = self.get_focus()[0]
        if item:
            item.enter_focus()

    def __on_focus(self,item):
        if item:
            is_double_click = self.__is_double_click(item)
            if is_double_click:
                item.activate()
                self._toggle_active_item(item)
            self.__on_focus_callback(item,is_double_click)

    def mouse_event(self, size, event, button, col, row, focus):
        is_handled = False
        if button == config.scroll_back:
            widget,pos = self.get_focus()
            if widget and pos > 0:
                self.set_focus(pos -1)
                is_handled = True
        elif button == config.scroll_next:
            widget,pos = self.get_focus()
            if widget and pos < len(self.body) -1:
                self.set_focus(pos + 1)
                is_handled = True
        else:
            is_handled = self.__super.mouse_event(size,
                                                  event,
                                                  button,
                                                  col,
                                                  row,
                                                  focus)
        return is_handled

class ListBoxedFrame(urwid.Frame):
    def __init__(self,listbox,header_text):
        self.listbox = listbox
        self.__super.__init__(urwid.LineBox( self.listbox ),
                              urwid.Text(header_text))
    def set_body(self,listbox):
        self.listbox = listbox
        self.__super.set_body(urwid.LineBox( self.listbox ))

    @property
    def has_item(self):
        return len(self.listbox.body) > 0

    def leave_focus(self):
        self.listbox.leave_focus()

    def enter_focus(self):
        self.listbox.enter_focus()

class StationListBox(ListBoxedFrame):
    def __init__(self):
        self._station_cache =[]
        self._station_info = {}
        self.__thread = None
        self.__need_swip_cache = False
        self.__prev_focused = None
        super(StationListBox,self).__init__(None,_('Stations'))

    def add_station(self,name,id,info):
        if self.__need_swip_cache:self.__swip_cache()
        self._station_cache.append((name,id))
        self._station_info[id]=info

    def __swip_cache(self):
        del self._station_cache[:]
        self._station_info.clear()
        self.__need_swip_cache = False

    def _on_focus(self,item,is_double_click):
        if SharedObject.notify and item:
            SharedObject.notify.print_info(
                shoutbrowser.translate_info(
                    self._station_info[item.user_data])
                )

            if item.is_active and is_double_click:
                self.on_player_activate(item)

    def on_player_activate(self,item):
        PlayerDialog().build(item.label,
                             item.user_data)

    def build(self,event_handler):
        if self.__need_swip_cache:
            self.__swip_cache()

        listbox =  MouseDrivenListBox(
            [ListBoxItem(name,id)                             
             for name ,id in self._station_cache]
            ,event_handler,self._on_focus)

        if self.body:
            if isinstance(self.body.w,urwid.Pile):
                del self.body.w.widget_list[:]
            else:
                del self.body.w.body.contents[:]

        self.set_body(listbox)
        SharedObject.notify.clean()
        listbox.set_focus(0)
        self.leave_focus()
        self.__need_swip_cache = True

class FavoriteStationListBox(StationListBox):
    def __init__(self,remove_event_callback):
        self.__super = super(self.__class__,self)
        self.__remove_event_callback = remove_event_callback
        self.__super.__init__()

    def on_remove_favorite(self,item):
        SharedObject.notify.print_info(_('Remove from Favorite: %s') % item.label) 
        FavoriteListDataBase().remove_station(item.user_data)
        self.__remove_event_callback()

    def build(self):
        event_handler ={'enter':self.on_player_activate,
                        'p':self.on_player_activate,
                        'delete':self.on_remove_favorite,
                        'r':self.on_remove_favorite}
        return self.__super.build(event_handler)

class SearchedStationListBox(StationListBox):
    def __init__(self):
        self.__super = super(self.__class__,self)
        self.__super.__init__()

    def on_add_favorite(self,item):
        SharedObject.notify.print_info(_('Add to Favorite: %s') % item.label)
        codec,bitrate,genre = self._station_info[item.user_data][0:3]
        FavoriteListDataBase().add_station(item.user_data,
                                           item.label,codec,bitrate,genre)

    def build(self):
        event_handler ={'enter':self.on_player_activate,
                        'p':self.on_player_activate,
                        'a':self.on_add_favorite}
        return self.__super.build(event_handler)

class GenreListBox(ListBoxedFrame):
    def __init__(self,callback):
        self.__active_item = None
        self.build(callback)
        
    def __on_focus(self,item,is_double_click):
        if item.is_active and is_double_click:
            item.user_data(item)

    def build(self,callback):
        event_handler ={'enter':callback}
        listbox =  MouseDrivenListBox(
            [ListBoxItem(name,callback)           
             for name in config.major_genre]
            ,event_handler,self.__on_focus)

        listbox.set_focus(0)
        self.__super = super(self.__class__,self)
        self.__super.__init__(listbox,
                              _('Genres'))

class GenreBrowser(urwid.Columns):
    __thread = None
    def __init__(self):
        self.__genrebox = GenreListBox(self.genre_callback)
        self.__super = super(self.__class__,self)
        self.__super.__init__(self.get_content())
        
    def terminate(self):
        if self.__thread:
            self.__thread.terminate()

    def get_content(self,genre=''):
        self.__stationbox = SearchedStationListBox()
        self.__stationbox.build()
        return [ ('fixed',25,
                 self.__genrebox),
                 self.__stationbox
                 ]

    def mouse_event(self, size, event, button, col, row, focus):
        if not self.get_cursor_coords(size):
            cur_index = self.get_focus_column()
            self.move_cursor_to_coords(size,col,row)
            focused = self.get_focus_column()
            self.__super.set_focus_column(cur_index)            
            self.set_focus_column(focused)
        return self.__super.mouse_event(size, event, button, col, row, focus)

    def set_focus_column(self,num):
        if self.widget_list[num].has_item:
            self.get_focus().leave_focus()
            self.__super.set_focus_column(num)
            self.get_focus().enter_focus()

    def genre_callback(self,item):
        genre = item.label
        if not self.__thread:
            self.__thread = ThreadQueueDispatcher()
        request = functools.partial(shoutbrowser.query_get_stations,genre)
        self.__thread.regist(RequestCommand(self.__stationbox,
                                            request)
                             )
        SharedObject.notify.print_warn(
            _('Station-list of %s Downloading...') % genre)

class FavoriteStation(urwid.Frame):
    def terminate(self):
        del self.body.w.widget_list[:]

    def __init__(self):
        station_frame = self.get_content()
        super(self.__class__,self).__init__(station_frame.get_body(),
                                            header=station_frame.get_header())

    def __update_content(self):
        SharedObject.view.set_body(self.get_content())

    def get_content(self):
        stations = FavoriteStationListBox(self.__update_content)

        FavoriteListDataBase().get_stations(stations.add_station)
        stations.build()
        return stations

class StationSearch(urwid.Columns):
    def __init__(self):
        self.__thread = None
        self.__keyword = ''
        self.__entry = config.request_limit
        self.__bitrate = config.request_bitrate
        self.__codec = self._translate_codec(
            filter(lambda item:item[1] == config.request_codec,
                   shoutbrowser.mime_type.iteritems())[0][0])
        self.__super = super(self.__class__,self)
        self.__super.__init__(self.get_content())
        self.__super.set_focus_column(0)

    def terminate(self):
        if self.__thread:
            self.__thread.terminate()

    def _on_bitrate_toggled(self,radio,state,rate):
        if state:
            self.__bitrate = rate

    def _on_codec_toggled(self,radio,state,codec):
        if state:
            self.__codec = codec

    def _verify_search_form(self):
        is_valid = False
        if not self.__keyword:
            SharedObject.notify.print_error(_('Please set keyword at least'))
        elif not self.__entry:
            SharedObject.notify.print_error(_('Please set entry more than 0'))
        else:
            is_valid = True
        return is_valid

    def _on_search(self,button):
        self.__entry = self.__entry_box.value()
        self.__keyword = self.__keyword_box.get_text()[0]
        if not self._verify_search_form():
            return
        
        self.__update_content()

    def _create_field_label(self,text):
        return urwid.Text(text+':')

    def _create_radio(self,group,label,value,callback,state=False):
        return urwid.RadioButton(group,
                                 label,
                                 state,
                                 callback,
                                 value)

    def _build_keyword_box(self):
        self.__keyword_box = urwid.Edit(edit_text=self.__keyword)
        yield self._create_field_label(_('keyword'))
        yield self._decorate_textbase_box(self.__keyword_box)

    def _build_limit_box(self):
        self.__entry_box = urwid.IntEdit(default=self.__entry)
        yield self._create_field_label(_('entry'))
        yield self._decorate_textbase_box(self.__entry_box)

    def _decorate_textbase_box(self,widget):
        return urwid.AttrWrap(widget,
                              SharedObject.palette.TEXTBOX)

    def _toggle_state(self,group,value):
        for radio in group:
            if radio.user_data == value:
                radio.set_state(True,False)
            else:
                radio.set_state(False,False)

    def _build_bitrate_group(self):
        bitrates = config.request_bitrate_range
        group = []
        event = self._on_bitrate_toggled
        yield self._create_radio(group, _('All'), 'All',event)
        for rate in bitrates:
            yield self._create_radio(group, rate+'kb', rate,event)
        self._toggle_state(group,self.__bitrate)

    def _build_bitrate_box(self):
        yield self._create_field_label(_('bitrate'))
        yield urwid.GridFlow(
            [x for x in self._build_bitrate_group()]
            ,9,1,1,'left')

    def _translate_codec(self,codec):
        return codec.split('/')[1]

    def _build_codec_group(self):
        group = []
        event = self._on_codec_toggled
        for codec,label in shoutbrowser.mime_type.iteritems():
            codec = self._translate_codec(codec)
            yield self._create_radio(group,label,codec,event)
        self._toggle_state(group,self.__codec)

    def _build_codec_box(self):
        yield self._create_field_label(_('format'))
        yield urwid.GridFlow(
            [x for x in self._build_codec_group()]
            ,9,1,1,'left')

    def _build_search_button(self):
        yield urwid.Padding(
            urwid.Button(_('Search'),self._on_search),
            'center',10)

    def _build_search_form(self):
        body = urwid.ListBox([
                widget for widget in itertools.chain(
                    self._build_keyword_box(),
                    self._build_bitrate_box(),
                    self._build_codec_box(),
                    self._build_limit_box(),
                    self._build_search_button()
                    )])

        return urwid.Frame(urwid.LineBox(body),
                           header=urwid.Text(_('Search Form')))

    def get_content(self,*args):
        self.__stationbox = SearchedStationListBox()
        self.__stationbox.build()
        return [ ('fixed',25,
                 self._build_search_form()),
                 self.__stationbox
                 ]

    def __update_content(self):
        if not self.__thread:
            self.__thread = ThreadQueueDispatcher()

        request = functools.partial(shoutbrowser.query_search_stations,
                                    self.__keyword,
                                    self.__bitrate,
                                    self.__codec,
                                    self.__entry_box.value()
                                    )
        self.__thread.regist(RequestCommand(self.__stationbox,
                                            request)
                             )
        SharedObject.notify.print_warn(_('Stations Searching...'))

    def mouse_event(self, size, event, button, col, row, focus):
        if not self.get_cursor_coords(size):
            cur = self.get_focus_column()
            self.move_cursor_to_coords(size,col,row)
            focused = self.get_focus_column()
            self.__super.set_focus_column(cur)
            self.set_focus_column(focused)
        return self.__super.mouse_event(size, event, button, col, row, focus)

    def set_focus_column(self,num):
        cur =self.get_focus()
        focused = self.widget_list[num]
        if cur == focused:return
        if isinstance(focused,SearchedStationListBox):
            if focused.has_item:
                focused.enter_focus()
                self.__super.set_focus_column(num)
        else:
            cur.leave_focus()
            self.__super.set_focus_column(num)

class TerminatableThread(threading.Thread):
    _need_terminate = False
    def __init__(self,target=None,args=()):
        super(TerminatableThread,self).__init__(target=target,args=args)

    def terminate(self):
        self._need_terminate = True

class RequestCommand(TerminatableThread):
    def __init__(self,stationbox,func):
        self.__box = stationbox
        self.__request = func
        super(self.__class__,self).__init__()

    def run(self):
        resource = None
        try:
            resource = self.__request()

        except:
            SharedObject.notify.print_warn(_('Requests are failed!'))
            return

        if not self._need_terminate:
            handler = shoutbrowser.StationListHandler(self.__box.add_station)
            shoutbrowser.saxing_data(resource,handler)
            self.__box.build()
        elif resource:
            resource.close()

class ThreadQueueDispatcher(TerminatableThread):
    __q = None
    __worker = None
    def __init__(self):
        super(self.__class__,self).__init__()
        self.__q = Queue()
        self.start()

    def regist(self,thread_entry):
        self.__q.put(thread_entry)

    def run(self):
        while not self._need_terminate:
            #single request can run at the same time.
            #queued request have priority over the running.
            try:
                entry = self.__q.get(timeout=0.1)
            except Empty:
                continue
            if self.__worker and self.__worker.isAlive():
                self.__worker.terminate()
            self.__worker = entry
            self.__worker.start()

        if self.__worker and self.__worker.isAlive():
            self.__worker.terminate()

class DialogBroker(object):
    def __init__(self):
        self.__dialog = None
        self.__prev_size = None
        self.__top = None

    def connect(self,widget,size):
        if self.__top:
            raise Exception
        self.__top = widget,size

    def disconnect(self):
        del self.__top ; del self.__dialog
        self.__top = None ; self.__dialog = None

    def propagate_key(self,size,key):
        if self.__dialog:
            if urwid.is_mouse_event(key):
                event,button,col,row = key
                self.__dialog.mouse_event(size, event, button, col, row, False)
            else:
                self.__dialog.keypress(size,key)

    @property
    def has_connection(self):
        return None != self.__dialog

    def __create_dialog(self,size):
        widget,widget_size = self.__top

        #calcurate position to locate top-widget as center
        left,top = [ (size[i] - widget_size[i])/2 for i in range(2)]
        wrapped_widget = urwid.Filler(
            urwid.AttrWrap(widget,
                           SharedObject.palette.DIALOG)
            )

        self.__dialog = urwid.Overlay(wrapped_widget,
                                      SharedObject.view,
                                      ('fixed left',left),
                                      widget_size[0],
                                      ('fixed top',top),
                                      widget_size[1])
    def get_canvas(self,size):
        canvas = None
        if self.__top:
            if self.__prev_size != size:
                self.__create_dialog(size)
            canvas = self.__dialog.render(size,True)
        return canvas

class AbstractDialog(object):
    def build(self,content,size):
        SharedObject.broker.connect(content,size)

    def close(self):
        SharedObject.broker.disconnect()

class ShoutRadioPlayer(TerminatableThread):
    def __init__(self,id):
        self.__station_id = id
        self.__proc = None
        self.__temp = None
        self.__super = super(self.__class__,self)
        self.__super.__init__()

    def run(self):
        location = self._get_playlist()
        if self._need_terminate:return

        command = [param if param !='%s' else location 
                   for param in config.player_command]

        SharedObject.notify.print_warn(_("Execute Player now!"))
        self.__proc = subprocess.Popen(command,
                                       stdout=subprocess.PIPE,
                                       stderr=subprocess.PIPE)
        self.__proc.wait()
        err = self.__proc.stderr.read()
        if err:
            SharedObject.notify.print_error(err)

    def _get_playlist(self):
        playlist = shoutbrowser.get_playlist(self.__station_id)
        if self._need_terminate:raise SystemExit()

        if not config.is_player_support_playlist:
            parser = shoutbrowser.PlayListParser(playlist)
            if self._need_terminate:raise SystemExit()

            if not config.is_player_support_activity_check:
                SharedObject.notify.print_warn(_("Station's Activity Asking..."))
                return parser.get_active_item()
            else:
                return parser.iteritems().next()
        else:

            self._write_playlist_to_tempfile(playlist)
            return self.__temp.name

    def _write_playlist_to_tempfile(self,playlist):
        import tempfile
        self.__temp = open(
            os.path.join( tempfile.gettempdir(),
                          '%s.%s' % (tempfile.gettempprefix(),id)),
            'w')
        self.__temp.write(playlist.read())
        if not self.__temp.closed:
            self.__temp.close()

    def terminate(self):
        self.__super.terminate()
        if self.__proc and self.__proc.returncode == None:
            os.kill(self.__proc.pid,signal.SIGTERM)
        self._cleanup_tempfile()

    def _cleanup_tempfile(self):
        if self.__temp:
            if not self.__temp.closed:
                self.__temp.close()
            os.remove(self.__temp.name)

class PlayerDialog(AbstractDialog):
    def __init__(self):
        self.__super = super(self.__class__,self)
        self.__super.__init__()
        self.__thread = None

    def build(self,name,id):
        terminate_button = urwid.Padding(
            urwid.Button(_('Terminate'),self._close),
            'center',('relative',22)
            )
        content = urwid.Pile([urwid.Text(_('Player calling...')),
                              urwid.Divider('-'),
                              urwid.Text(name),
                              terminate_button
                              ])
        self.__super.build(content,(60,6))
        self._run_worker(id)

    def _run_worker(self,id):
        SharedObject.notify.print_warn(_('Playlist Downloading...'))
        self.__thread = ShoutRadioPlayer(id)
        self.__thread.start()

    def _cleanup_worker(self):
        if self.__thread:
            self.__thread.terminate()
            self.__thread.join()

    def _close(self,button):
        SharedObject.notify.print_warn(_('Player Terminating...'))
        self._cleanup_worker()
        SharedObject.notify.print_warn(_('Player Terminated!'))
        self.__super.close()

class HelpDialog(AbstractDialog):
    __about_message =\
'''tuxshout version %s
(c)2009 s.yamada
Release under the LGPL
tuxshout is SHOUTcast Radio tuner''' % version

    __help_message =\
_('''To run below method, type shortcut-key (i.e. B-key for Browser)
Help     : display this messages
Browser  : supplys genre-based searching
Favorite : supplys listing of registered stations
Search   : supplys searching with detail infomations
Exit     : exit program''')

    __method_message =\
_('''if station selected,below shortcut-key available
some shortcut are available at below mode
Browser=B Search=S Favorite=F
p [B|S|F]: play the station
enter    : same as p-key
a [B|S]  : add the station to favorite list
r [F]    : remove the station from favorite list
delete   : same as r-key''')

    __config_message =\
_('''configurable parameter is store in the config.py
please edit it to adapt to you environment''')

    def __init__(self):
        self.__super = super(self.__class__,self)
        self.__super.__init__()

    def build(self):
        content = urwid.Pile([urwid.Text(self.__about_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__help_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__method_message),
                              urwid.Divider('-'),
                              urwid.Text(self.__config_message),
                              urwid.Padding(
                    urwid.Button(_('Close'),self._close),
                    'center',
                    ('relative',16)
                    )
                              ])
        self.__super.build(content,(65,24))

    def _close(self,button):
        self.__super.close()

class AbstractMenuItem(urwid.AttrWrap):
    def __init__(self,key,label,func,need_toggle=True):
        self._shortcut = key
        readable_key = key.upper()
        base_text = '%s: %s' % (readable_key,label)
        super(AbstractMenuItem,self).__init__(
            ListBoxItem(base_text,None),
            SharedObject.palette.MENU)
        self.need_toggle = need_toggle
        self.func = func
    
    def activate(self,is_active=True):
        if is_active:
            self.func()
            if self.need_toggle:
                self.set_attr(SharedObject.palette.MENU_REVERSE)
        else:
            self.set_attr(SharedObject.palette.MENU)
        
    def keypress(self, size, key):
        if self._shortcut == key:
            self.activate()
        else:
            return key

class ApplicationMenuItem(AbstractMenuItem):
    def __init__(self,key,label,application):
        self._super = super(self.__class__,self)
        self._super.__init__(key,
                             label,
                             self._switch_application)
        self._application = application

    def _switch_application(self):
        body = SharedObject.view.get_body()
        if body:
            body.terminate()
            del body
        SharedObject.view.set_body(
            self._application()
            )
        SharedObject.notify.clean()
            
class MenuBox(urwid.Columns):
    def __init__(self):
        self.__super = super(self.__class__,self)
        self.__super.__init__(self.__create_menu())
        self.__active_item = self.widget_list[0]
        self.__active_item.activate()

    def __create_menu(self):
        return [
            ApplicationMenuItem('b',_('Browse'),GenreBrowser),
            ApplicationMenuItem('f',_('Favorite'),FavoriteStation),
            ApplicationMenuItem('s',_('Search'),StationSearch),
            AbstractMenuItem('?',_('Help'),self.__show_help,False),
            AbstractMenuItem('x',_('Exit'),self.__exit_program,False)
            ]

    def set_focus(self,item):
        item.activate()
        if item.need_toggle:
            self.__active_item.activate(False)
            self.__active_item = item

    def keypress(self,size,key):
        is_propagated = False
        for cell in self.widget_list:
            if not cell.keypress(size,key):
                is_propagated = True
                if cell.need_toggle:
                    self.__active_item.activate(False)
                    self.__active_item = cell
                break
        if not is_propagated:return key

    def __show_help(self):
        HelpDialog().build()

    def __exit_program(self):
        import sys
        sys.exit(0)

class TuxShout:
    __worker_thread = None
    def __init__(self):
        SharedObject.regist_main_frame(urwid.Frame( None ))
        SharedObject.view.set_header(
            urwid.AttrWrap(MenuBox(),
                           SharedObject.palette.MENU))

    def set_status(self,text,state):
        info = urwid.AttrWrap(
            urwid.Text(text),state)
        SharedObject.view.set_footer(info)

    def main(self):
        space = (65,24)
        if space > SharedObject.screen.get_cols_rows():
            print _(\
'''tuxshout can not run at this nallow screen!
please allocate more wider screen than 
width :%s
height:%s''') % space
            return
        SharedObject.screen.run_wrapper( self.run )

    def run(self):
        SharedObject.screen.set_mouse_tracking()
        size = SharedObject.screen.get_cols_rows()
        broker = SharedObject.broker
        while 1:
            self.draw_screen(size,
                             broker.get_canvas(size))
            keys = SharedObject.screen.get_input()
            for k in keys:
                if k == "window resize":
                    size = SharedObject.screen.get_cols_rows()
                    continue

                #key-action of dialog prioir than other widgets
                if broker.has_connection:
                    broker.propagate_key(size,k)
                    continue

                #mouse event-handler
                if urwid.is_mouse_event(k):
                    event,button,col,row = k
                    SharedObject.view.mouse_event(size,event,button,
                                                  col,row,True)
                    continue

                #key-action of any other widget prioir than menu
                if not SharedObject.view.keypress( size, k ):
                    continue
                else:
                    SharedObject.view.header.keypress(size,k)

    def draw_screen(self, size,canvas=None):
        if not canvas:
            canvas = SharedObject.view.render( size, focus=True )
        SharedObject.screen.draw_screen( size, canvas )

class Notification:
    def print_info(self,text):
        self._set_status(text,SharedObject.palette.MSG_INFO)

    def print_warn(self,text):
        self._set_status(text,SharedObject.palette.MSG_WARN)

    def print_error(self,text):
        self._set_status(text,SharedObject.palette.MSG_ERR)

    def _set_status(self,text,state):
        info = SharedObject.view.footer
        if not info:
            info =urwid.AttrWrap(
                urwid.Text(text),state)
            SharedObject.view.set_footer(info)
        else:
            info.w.set_text(text)
            info.set_attr(state)

    def clean(self):
        SharedObject.view.set_footer(None)

class ColorPalette(object):
    def __init__(self,screen):
        screen.register_palette([
                (self.DEFAULT, 'white', 'black', 'standout'),
                (self.DIALOG, 'black', 'light gray', 'standout'),
                (self.MENU, 'black', 'dark cyan', 'standout'),
                (self.MENU_REVERSE, 'white', 'dark blue',('bold','underline')),
                (self.ITEM_LEAVE, 'black', 'dark cyan', 'bold'),
                (self.ITEM_LEAVE_REV, 'black', 'dark magenta', 'bold'),
                (self.ITEM_FOCUS, 'white', 'dark blue', 'bold'),
                (self.ITEM_REVERSE, 'white', 'dark red', 'bold'),
                (self.TEXTBOX,   'white', 'dark green','underline'),
                (self.MSG_INFO, 'light green', 'black', 'underline'),
                (self.MSG_WARN,  'yellow', 'black','bold'),
                (self.MSG_ERR,   'light red', 'black',('bold','underline'))
                ])
    @property
    def MSG_INFO(self):
        return 'info'

    @property
    def MSG_WARN(self):
        return 'warn'

    @property
    def MSG_ERR(self):
        return 'error'

    @property
    def MENU(self):
        return 'menu'

    @property
    def MENU_REVERSE(self):
        return 'menu_reverse'

    @property
    def ITEM_LEAVE(self):
        return 'leave'

    @property
    def ITEM_LEAVE_REV(self):
        return 'leave_rev'

    @property
    def ITEM_FOCUS(self):
        return 'focus'

    @property
    def ITEM_REVERSE(self):
        return 'reverse'

    @property
    def TEXTBOX(self):
        return 'tbox'

    @property
    def DEFAULT(self):
        return 'default'

    @property
    def DIALOG(self):
        return 'dialog'


class SharedObjectManager:
    def __init__(self):
        self.screen = urwid.raw_display.Screen()
        self.palette = ColorPalette(self.screen)
        self.notify = Notification()

    def regist_main_frame(self,frame):
        self.view = frame
        self.broker = DialogBroker()

SharedObject = SharedObjectManager()

if __name__ == '__main__':
    gc.enable()
    gc.set_threshold(5)
    TuxShout().main()
