﻿'Copyright (C) 2010 pepetaro, All rights reserved.
'This library is free software; you can redistribute it and/or
'modify it under the terms of the GNU Lesser General Public
'License as published by the Free Software Foundation; either
'version 3 of the License, or (at your option) any later version.

'This library is distributed in the hope that it will be useful,
'but WITHOUT ANY WARRANTY; without even the implied warranty of
'MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
'Lesser General Public License for more details.

'You should have received a copy of the GNU Lesser General Public
'License along with this library; if not, write to the Free Software
'Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

Namespace TwitterServices
    Public NotInheritable Class ListsService
        <DebuggerBrowsable(DebuggerBrowsableState.Never)> Private tw As Twitter
        Friend Sub New(ByVal tw As Twitter)
            Me.tw = tw
        End Sub
        Private ReadOnly Property Self As TwitterUser
            Get
                If tw.SelfInfo Is Nothing Then
                    If Not tw.VerifyCredentials() Then
                        Throw New InvalidOperationException("認証に失敗しました。コンシューマー鍵かアクセストークンが正しくない可能性があります。")
                    End If
                End If
                Return tw.SelfInfo
            End Get
        End Property

        ''' <summary>リストを作成します。</summary>
        ''' <param name="name">リストの名前</param>
        ''' <param name="mode">公開か非公開のどちらか</param>
        ''' <returns>作成したリスト</returns>
        Public Overloads Function Create(ByVal name As String, ByVal mode As ListMode) As TwitterList
            Dim param As New Dictionary(Of String, String)
            param.Add("name", name)
            param.Add("mode", mode.ToString.ToLower)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists.xml", Self.ScreenName)), "POST", param, Nothing)
            Dim xd = XElement.Load(stream)
            stream.Close()
            Return TwitterList.FromXml(xd)
        End Function
        ''' <summary>リストを作成します。</summary>
        ''' <param name="name">リストの名前</param>
        ''' <param name="mode">公開モード</param>
        ''' <param name="description">説明</param>
        ''' <returns>作成したリスト</returns>
        Public Overloads Function Create(ByVal name As String, ByVal mode As ListMode, ByVal description As String) As TwitterList
            Dim param As New Dictionary(Of String, String)
            param.Add("name", name)
            param.Add("mode", mode.ToString.ToLower)
            param.Add("description", description)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists.xml", Self.ScreenName)), "POST", param, Nothing)
            Dim xd = XElement.Load(stream)
            stream.Close()
            Return TwitterList.FromXml(xd)
        End Function

        ''' <summary>リストの設定を変更します。</summary>
        ''' <param name="id">更新するリストの名前</param>
        ''' <param name="param">更新する項目リスト</param>
        ''' <returns>更新したリスト</returns>
        Public Function Update(ByVal id As String, ByVal param As Dictionary(Of ListUpdateItems, Object)) As TwitterList
            Dim uploadparam As New Dictionary(Of String, String)
            For Each o In param
                Dim str As String
                If o.Value Is GetType(String) Then
                    str = DirectCast(o.Value, String)
                Else
                    str = o.Value.ToString.ToLower
                End If
                uploadparam.Add(o.Key.ToString.ToLower, str)
            Next
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists/{1}.xml", Self.ScreenName, id)), "POST", uploadparam, Nothing)
            Dim xd = XElement.Load(stream)
            stream.Close()
            Return TwitterList.FromXml(xd)
        End Function

        Friend Shared Function GetLists(ByVal stream As Stream, ByVal close As Boolean, <Out()> ByRef resultCursor As CursorResult) As IEnumerable(Of TwitterList)
            Dim xd = XElement.Load(stream)
            If close Then stream.Close()
            Dim re = xd...<list>.select(Function(o) TwitterList.FromXml(o))
            resultCursor.NextCursor = Long.Parse(xd...<next_cursor>.Value)
            resultCursor.PreviousCursor = Long.Parse(xd...<previous_cursor>.Value)
            Return re
        End Function

        ''' <summary>指定したユーザーのリストを取得します。</summary>
        ''' <param name="user">取得元のユーザーscreen_name</param>
        ''' <param name="cursor">カーソル位置</param>
        ''' <param name="resultCursor">Twitterから返されたCursor情報</param>
        Public Function GetLists(ByVal user As String, ByVal cursor As Long, <Out()> ByRef resultCursor As CursorResult) As IEnumerable(Of TwitterList)
            Dim param As New Dictionary(Of String, String)
            param.Add("cursor", cursor.ToString)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists.xml", Self.ScreenName)), "GET", param, Nothing)
            Return GetLists(stream, True, resultCursor)
        End Function

        ''' <summary>リストを取得します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="id">リストの名前</param>
        Public Function GetList(ByVal user As String, ByVal id As String) As TwitterList
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists/{1}.xml", user, id)), "GET", Nothing, Nothing)
            Dim xd = XElement.Load(stream)
            stream.Close()
            Return TwitterList.FromXml(xd)
        End Function

        ''' <summary>リストを削除します。</summary>
        ''' <param name="id">削除するリストの名前</param>
        Public Sub Destroy(ByVal id As String)
            tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists/{1}.xml", Self.ScreenName, id)), "DELETE", Nothing, Nothing).Close()
        End Sub

        ''' <summary>指定したリストのタイムラインを取得します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        Public Overloads Function GetStatuses(ByVal user As String, ByVal list_id As String) As IEnumerable(Of TwitterStatus)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists/{1}/statuses.xml", user, list_id)), "GET", Nothing, Nothing)
            Return TimelinesServices.GetTweets(stream, True)
        End Function
        ''' <summary>指定したリストのタイムラインを取得します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        ''' <param name="per_page">1ページに返される個数 1～200</param>
        ''' <param name="page">ページ番号 1～</param>
        Public Overloads Function GetStatuses(ByVal user As String, ByVal list_id As String, ByVal per_page As Integer, ByVal page As Integer) As IEnumerable(Of TwitterStatus)
            Dim param As New Dictionary(Of String, String)
            param.Add("per_page", per_page.ToString)
            param.Add("page", page.ToString)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists/{1}/statuses.xml", user, list_id)), "GET", param, Nothing)
            Return TimelinesServices.GetTweets(stream, True)
        End Function

        ''' <summary>指定したユーザーが登録されているリストを取得します。</summary>
        ''' <param name="user">確認するユーザーのscreen_name</param>
        ''' <param name="cursor">カーソル位置</param>
        ''' <param name="resultCursor">Twitterから返されたCursor情報</param>
        Public Function GetMemberships(ByVal user As String, ByVal cursor As Long, <Out()> ByRef resultCursor As CursorResult) As IEnumerable(Of TwitterList)
            Dim param As New Dictionary(Of String, String)
            param.Add("cursor", cursor.ToString)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists/memberships.xml", user)), "GET", param, Nothing)
            Return GetLists(stream, True, resultCursor)
        End Function

        ''' <summary>指定したユーザーが購読（フォロー）しているリストを取得します。</summary>
        ''' <param name="user">確認するユーザーのscreen_name</param>
        ''' <param name="cursor">カーソル位置</param>
        ''' <param name="resultCursor">Twitterから返されたCursor情報</param>
        Public Function GetSubscriptions(ByVal user As String, ByVal cursor As Long, <Out()> ByRef resultCursor As CursorResult) As IEnumerable(Of TwitterList)
            Dim param As New Dictionary(Of String, String)
            param.Add("cursor", cursor.ToString)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/lists/subscriptions.xml", user)), "GET", param, Nothing)
            Return GetLists(stream, True, resultCursor)
        End Function

#Region "登録系"
        ''' <summary>指定したリストがフォローしているユーザーを取得します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        ''' <param name="cursor">カーソル位置</param>
        ''' <param name="resultCursor">Twitterから返されたCursor情報</param>
        Public Function GetMembers(ByVal user As String, ByVal list_id As String, ByVal cursor As Long, <Out()> ByRef resultCursor As CursorResult) As IEnumerable(Of TwitterUser)
            Dim param As New Dictionary(Of String, String)
            param.Add("cursor", cursor.ToString)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/members.xml", user, list_id)), "GET", param, Nothing)
            Return UsersService.GetUsers(stream, True, resultCursor)
        End Function

        ''' <summary>リストにユーザーを追加（フォロー）します。</summary>
        ''' <param name="list_id">リストの名前</param>
        ''' <param name="id">追加するユーザーのscreen_nameかid</param>
        Public Sub AddMember(ByVal list_id As String, ByVal id As String)
            Dim param As New Dictionary(Of String, String)
            param.Add("id", id)
            tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/members.xml", Self.ScreenName, list_id)), "POST", param, Nothing).Close()
        End Sub

        ''' <summary>リストからユーザーを削除します。</summary>
        ''' <param name="list_id">リストの名前</param>
        ''' <param name="id">削除するユーザーのscreen_nameかid</param>
        Public Sub RemoveMember(ByVal list_id As String, ByVal id As String)
            Dim param As New Dictionary(Of String, String)
            param.Add("id", id)
            tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/members.xml", Self.ScreenName, list_id)), "DELETE", param, Nothing).Close()
        End Sub

        ''' <summary>リストに指定したユーザーが登録されているかを確認します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        ''' <param name="id">確認するユーザーのscreen_nameまたはid</param>
        Public Function ExistsMember(ByVal user As String, ByVal list_id As String, ByVal id As String) As Boolean
            Try
                tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/members/{2}.xml", user, list_id, id)), "GET", Nothing, Nothing).Close()
                Return True
            Catch
                Return False
            End Try
        End Function
#End Region
#Region "購読系"
        ''' <summary>指定したリストをフォローしているユーザーを取得します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        ''' <param name="cursor">カーソル位置</param>
        ''' <param name="resultCursor">Twitterから返されたCursor情報</param>
        Public Function GetSubscribers(ByVal user As String, ByVal list_id As String, ByVal cursor As Long, <Out()> ByRef resultCursor As CursorResult) As IEnumerable(Of TwitterUser)
            Dim param As New Dictionary(Of String, String)
            param.Add("cursor", Cursor.ToString)
            Dim stream = tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/subscribers.xml", user, list_id)), "GET", param, Nothing)
            Return UsersService.GetUsers(stream, True, resultCursor)
        End Function

        ''' <summary>指定したリストをフォローします。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        Public Sub AddSubscribers(ByVal user As String, ByVal list_id As String)
            tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/subscribers.xml", user, list_id)), "POST", Nothing, Nothing).Close()
        End Sub

        ''' <summary>指定したリストのフォローを解除します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        Public Sub RemoveSubscribers(ByVal user As String, ByVal list_id As String)
            tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/subscribers.xml", user, list_id)), "DELETE", Nothing, Nothing).Close()
        End Sub

        ''' <summary>リストに指定したユーザーが登録されているかを確認します。</summary>
        ''' <param name="user">リストの作者のscreen_name</param>
        ''' <param name="list_id">リストの名前</param>
        ''' <param name="id">確認するユーザーのscreen_nameまたはid</param>
        Public Function ExistsSubscribers(ByVal user As String, ByVal list_id As String, ByVal id As String) As Boolean
            Try
                tw.GetStream(Twitter.CreateUri(1, String.Format("{0}/{1}/subscribers/{2}.xml", user, list_id, id)), "GET", Nothing, Nothing).Close()
                Return True
            Catch
                Return False
            End Try
        End Function
#End Region
    End Class
End Namespace