/**********************************************************************
 * cipher_suite_t.h                                         August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/

#ifndef _CIPHER_SUITE_T_H
#define _CIPHER_SUITE_T_H

#include "types/base_t.h"

#ifdef __KERNEL__
#include <linux/string.h>
#else
#include <string.h>
#endif

typedef struct {
	u8 cs[2];
} cipher_suite_t;

#define CIPHER_SUITE_NLEN 2

#define TLS_NULL_WITH_NULL_NULL                   {{ 0x00, 0x00 }}
#define TLS_NULL_WITH_NULL_NULL_I                 0x0000
#define TLS_NULL_WITH_NULL_NULL_S                 "TLS_NULL_WITH_NULL_NULL"

#define TLS_RSA_WITH_NULL_MD5                     {{ 0x00, 0x01 }}
#define TLS_RSA_WITH_NULL_MD5_I                   0x0001
#define TLS_RSA_WITH_NULL_MD5_S                   "TLS_RSA_WITH_NULL_MD5"
#define TLS_RSA_WITH_NULL_SHA                     {{ 0x00, 0x02 }}
#define TLS_RSA_WITH_NULL_SHA_I                   0x0002
#define TLS_RSA_WITH_NULL_SHA_S                   "TLS_RSA_WITH_NULL_SHA"
#define TLS_RSA_EXPORT_WITH_RC4_40_MD5            {{ 0x00, 0x03 }}
#define TLS_RSA_EXPORT_WITH_RC4_40_MD5_I          0x0003
#define TLS_RSA_EXPORT_WITH_RC4_40_MD5_S          "TLS_RSA_EXPORT_WITH_RC4_40_MD5"
#define TLS_RSA_WITH_RC4_128_MD5                  {{ 0x00, 0x04 }}
#define TLS_RSA_WITH_RC4_128_MD5_I                0x0004
#define TLS_RSA_WITH_RC4_128_MD5_S                "TLS_RSA_WITH_RC4_128_MD5"
#define TLS_RSA_WITH_RC4_128_SHA                  {{ 0x00, 0x05 }}
#define TLS_RSA_WITH_RC4_128_SHA_I                0x0005
#define TLS_RSA_WITH_RC4_128_SHA_S                "TLS_RSA_WITH_RC4_128_SHA"
#define TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5        {{ 0x00, 0x06 }}
#define TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5_I      0x0006
#define TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5_S      "TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5"
#define TLS_RSA_WITH_IDEA_CBC_SHA                 {{ 0x00, 0x07 }}
#define TLS_RSA_WITH_IDEA_CBC_SHA_I               0x0007
#define TLS_RSA_WITH_IDEA_CBC_SHA_S               "TLS_RSA_WITH_IDEA_CBC_SHA"
#define TLS_RSA_EXPORT_WITH_DES40_CBC_SHA         {{ 0x00, 0x08 }}
#define TLS_RSA_EXPORT_WITH_DES40_CBC_SHA_I       0x0008
#define TLS_RSA_EXPORT_WITH_DES40_CBC_SHA_S       "TLS_RSA_EXPORT_WITH_DES40_CBC_SHA"
#define TLS_RSA_WITH_DES_CBC_SHA                  {{ 0x00, 0x09 }}
#define TLS_RSA_WITH_DES_CBC_SHA_I                0x0009
#define TLS_RSA_WITH_DES_CBC_SHA_S                "TLS_RSA_WITH_DES_CBC_SHA"
#define TLS_RSA_WITH_3DES_EDE_CBC_SHA             {{ 0x00, 0x0A }}
#define TLS_RSA_WITH_3DES_EDE_CBC_SHA_I           0x000A
#define TLS_RSA_WITH_3DES_EDE_CBC_SHA_S           "TLS_RSA_WITH_3DES_EDE_CBC_SHA"

#define TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA      {{ 0x00, 0x0B }}
#define TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA_I    0x000B
#define TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA_S    "TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA"
#define TLS_DH_DSS_WITH_DES_CBC_SHA               {{ 0x00, 0x0C }}
#define TLS_DH_DSS_WITH_DES_CBC_SHA_I             0x000C
#define TLS_DH_DSS_WITH_DES_CBC_SHA_S             "TLS_DH_DSS_WITH_DES_CBC_SHA"
#define TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA          {{ 0x00, 0x0D }}
#define TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA_I        0x000D
#define TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA_S        "TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA"
#define TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA      {{ 0x00, 0x0E }}
#define TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA_I    0x000E
#define TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA_S    "TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA"
#define TLS_DH_RSA_WITH_DES_CBC_SHA               {{ 0x00, 0x0F }}
#define TLS_DH_RSA_WITH_DES_CBC_SHA_I             0x000F
#define TLS_DH_RSA_WITH_DES_CBC_SHA_S             "TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA"
#define TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA          {{ 0x00, 0x10 }}
#define TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA_I        0x0010
#define TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA_S        "TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA"
#define TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA     {{ 0x00, 0x11 }}
#define TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA_I   0x0011
#define TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA_S   "TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA"
#define TLS_DHE_DSS_WITH_DES_CBC_SHA              {{ 0x00, 0x12 }}
#define TLS_DHE_DSS_WITH_DES_CBC_SHA_I            0x0012
#define TLS_DHE_DSS_WITH_DES_CBC_SHA_S            "TLS_DHE_DSS_WITH_DES_CBC_SHA"
#define TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA         {{ 0x00, 0x13 }}
#define TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA_I       0x0013
#define TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA_S       "TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA"
#define TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA     {{ 0x00, 0x14 }}
#define TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA_I   0x0014
#define TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA_S   "TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA"
#define TLS_DHE_RSA_WITH_DES_CBC_SHA              {{ 0x00, 0x15 }}
#define TLS_DHE_RSA_WITH_DES_CBC_SHA_I            0x0015
#define TLS_DHE_RSA_WITH_DES_CBC_SHA_S            "TLS_DHE_RSA_WITH_DES_CBC_SHA"
#define TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA         {{ 0x00, 0x16 }}
#define TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA_I       0x0016
#define TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA_S       "TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA"

#define TLS_DH_anon_EXPORT_WITH_RC4_40_MD5        {{ 0x00, 0x17 }}
#define TLS_DH_anon_EXPORT_WITH_RC4_40_MD5_I      0x0017
#define TLS_DH_anon_EXPORT_WITH_RC4_40_MD5_S      "TLS_DH_anon_EXPORT_WITH_RC4_40_MD5"
#define TLS_DH_anon_WITH_RC4_128_MD5              {{ 0x00, 0x18 }}
#define TLS_DH_anon_WITH_RC4_128_MD5_I            0x0018
#define TLS_DH_anon_WITH_RC4_128_MD5_S            "TLS_DH_anon_WITH_RC4_128_MD5"
#define TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA     {{ 0x00, 0x19 }}
#define TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA_I   0x0019
#define TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA_S   "TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA"
#define TLS_DH_anon_WITH_DES_CBC_SHA              {{ 0x00, 0x1A }}
#define TLS_DH_anon_WITH_DES_CBC_SHA_I            0x001A
#define TLS_DH_anon_WITH_DES_CBC_SHA_S            "TLS_DH_anon_WITH_DES_CBC_SHA"
#define TLS_DH_anon_WITH_3DES_EDE_CBC_SHA         {{ 0x00, 0x1B }}
#define TLS_DH_anon_WITH_3DES_EDE_CBC_SHA_I       0x001B
#define TLS_DH_anon_WITH_3DES_EDE_CBC_SHA_S       "TLS_DH_anon_WITH_3DES_EDE_CBC_SHA"

/* 56-bit Export Cipher Suites For TLS
 * http://www.ietf.org/proceedings/02mar/I-D/draft-ietf-tls-56-bit-ciphersuites-01.txt
 */
#define TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA       {{ 0x00, 0x62 }}
#define TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA_I     0x0062
#define TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA_S     "TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA"
#define TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA   {{ 0x00, 0x63 }}
#define TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA_I 0x0063
#define TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA_S "TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA"
#define TLS_RSA_EXPORT1024_WITH_RC4_56_SHA        {{ 0x00, 0x64 }}
#define TLS_RSA_EXPORT1024_WITH_RC4_56_SHA_I      0x0064
#define TLS_RSA_EXPORT1024_WITH_RC4_56_SHA_S      "TLS_RSA_EXPORT1024_WITH_RC4_56_SHA"
#define TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA    {{ 0x00, 0x65 }}
#define TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA_I  0x0065
#define TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA_S  "TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA"
#define TLS_DHE_DSS_WITH_RC4_128_SHA              {{ 0x00, 0x66 }}
#define TLS_DHE_DSS_WITH_RC4_128_SHA_I            0x0066
#define TLS_DHE_DSS_WITH_RC4_128_SHA_S            "TLS_DHE_DSS_WITH_RC4_128_SHA"

/* AES Cipher Suites for TLS
 * http://www.ietf.org/rfc/rfc3268.txt */
#define TLS_RSA_WITH_AES_128_CBC_SHA              {{ 0x00, 0x2F }}
#define TLS_RSA_WITH_AES_128_CBC_SHA_I            0x002F
#define TLS_RSA_WITH_AES_128_CBC_SHA_S            "TLS_RSA_WITH_AES_128_CBC_SHA"
#define TLS_DH_DSS_WITH_AES_128_CBC_SHA           {{ 0x00, 0x30 }}
#define TLS_DH_DSS_WITH_AES_128_CBC_SHA_I         0x0030
#define TLS_DH_DSS_WITH_AES_128_CBC_SHA_S         "TLS_DH_DSS_WITH_AES_128_CBC_SHA"
#define TLS_DH_RSA_WITH_AES_128_CBC_SHA           {{ 0x00, 0x31 }}
#define TLS_DH_RSA_WITH_AES_128_CBC_SHA_I         0x0031
#define TLS_DH_RSA_WITH_AES_128_CBC_SHA_S         "TLS_DH_RSA_WITH_AES_128_CBC_SHA"
#define TLS_DHE_DSS_WITH_AES_128_CBC_SHA          {{ 0x00, 0x32 }}
#define TLS_DHE_DSS_WITH_AES_128_CBC_SHA_I        0x0032
#define TLS_DHE_DSS_WITH_AES_128_CBC_SHA_S        "TLS_DHE_DSS_WITH_AES_128_CBC_SHA"
#define TLS_DHE_RSA_WITH_AES_128_CBC_SHA          {{ 0x00, 0x33 }}
#define TLS_DHE_RSA_WITH_AES_128_CBC_SHA_I        0x0033
#define TLS_DHE_RSA_WITH_AES_128_CBC_SHA_S        "TLS_DHE_RSA_WITH_AES_128_CBC_SHA"
#define TLS_DH_anon_WITH_AES_128_CBC_SHA          {{ 0x00, 0x34 }}
#define TLS_DH_anon_WITH_AES_128_CBC_SHA_I        0x0034
#define TLS_DH_anon_WITH_AES_128_CBC_SHA_S        "TLS_DH_anon_WITH_AES_128_CBC_SHA"
#define TLS_RSA_WITH_AES_256_CBC_SHA              {{ 0x00, 0x35 }}
#define TLS_RSA_WITH_AES_256_CBC_SHA_I            0x0035
#define TLS_RSA_WITH_AES_256_CBC_SHA_S            "TLS_RSA_WITH_AES_256_CBC_SHA"
#define TLS_DH_DSS_WITH_AES_256_CBC_SHA           {{ 0x00, 0x36 }}
#define TLS_DH_DSS_WITH_AES_256_CBC_SHA_I         0x0036
#define TLS_DH_DSS_WITH_AES_256_CBC_SHA_S         "TLS_DH_DSS_WITH_AES_256_CBC_SHA"
#define TLS_DH_RSA_WITH_AES_256_CBC_SHA           {{ 0x00, 0x37 }}
#define TLS_DH_RSA_WITH_AES_256_CBC_SHA_I         0x0037
#define TLS_DH_RSA_WITH_AES_256_CBC_SHA_S         "TLS_DH_RSA_WITH_AES_256_CBC_SHA"
#define TLS_DHE_DSS_WITH_AES_256_CBC_SHA          {{ 0x00, 0x38 }}
#define TLS_DHE_DSS_WITH_AES_256_CBC_SHA_I        0x0038
#define TLS_DHE_DSS_WITH_AES_256_CBC_SHA_S        "TLS_DHE_DSS_WITH_AES_256_CBC_SHA"
#define TLS_DHE_RSA_WITH_AES_256_CBC_SHA          {{ 0x00, 0x39 }}
#define TLS_DHE_RSA_WITH_AES_256_CBC_SHA_I        0x0039
#define TLS_DHE_RSA_WITH_AES_256_CBC_SHA_S        "TLS_DHE_RSA_WITH_AES_256_CBC_SHA"
#define TLS_DH_anon_WITH_AES_256_CBC_SHA          {{ 0x00, 0x3A }}
#define TLS_DH_anon_WITH_AES_256_CBC_SHA_I        0x003A
#define TLS_DH_anon_WITH_AES_256_CBC_SHA_S        "TLS_DH_anon_WITH_AES_256_CBC_SHA"

#define CIPHER_SUITE_END_MARKER                   {{ 0xff, 0xff }}
#define CIPHER_SUITE_END_MARKER_I                 0xffff

#define cipher_suite_cmp(a, b) (memcmp((a), (b), CIPHER_SUITE_NLEN))
#define cipher_suite_cpy(dst, src) (memcpy((dst), (src), CIPHER_SUITE_NLEN))

#define cipher_suite_toi(a) ((a)->cs[0] << 8 | (a)->cs[1])


static inline size_t
cipher_suite_ncount(const cipher_suite_t *list, size_t bytes)
{
	size_t i = 0;
	cipher_suite_t cs_end = CIPHER_SUITE_END_MARKER;

	if (!list)
		return 0;

	while (bytes >= sizeof(cipher_suite_t)) {
		if (!cipher_suite_cmp(&cs_end, list++))
			break;
		bytes =- sizeof(cipher_suite_t);
		i += sizeof(cipher_suite_t);
	}

	return i;
}


static inline size_t
cipher_suite_count(const cipher_suite_t *list)
{
	size_t i = 0;
	cipher_suite_t cs_end = CIPHER_SUITE_END_MARKER;

	if (!list)
		return 0;

	while (1) {
		if (!cipher_suite_cmp(&cs_end, list++))
			break;
		i++;
	}

	return i * sizeof(cipher_suite_t);
}


static inline cipher_suite_t *
cipher_suite_nfind(cipher_suite_t *haystack, const cipher_suite_t *needle, 
		size_t bytes)
{
	cipher_suite_t cs_end = CIPHER_SUITE_END_MARKER;

	while (bytes >= sizeof(cipher_suite_t)) {
		if (!cipher_suite_cmp(&cs_end, haystack))
			break;
		if (!cipher_suite_cmp(needle, haystack))
			return haystack;
		haystack++;
		bytes -= sizeof(cipher_suite_t);
	}

	return NULL;
}


static inline cipher_suite_t *
cipher_suite_find(cipher_suite_t *haystack, const cipher_suite_t *needle)
{
	return cipher_suite_nfind(haystack, needle, 
			cipher_suite_count(haystack));
}

#endif /* _CIPHER_SUITE_T_H */
