/**********************************************************************
 * alert.c                                                  August 2005
 *
 * KSSLD: An implementation of SSL/TLS in the Linux Kernel
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This file based in part on code from LVS www.linuxvirtualserver.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 **********************************************************************/


#include <linux/kernel.h>
#include <linux/net.h>
#include <linux/version.h>

#include <net/tcp.h>

#include "record.h"
#include "alert.h"
#include "message.h"
#include "conn.h"
#include "log.h"


static inline int kssl_alert_write(kssl_record_t *cr, void *data)
{
	u8 *buf;
	alert_t *alert;

	alert = (alert_t *)data;

	KSSL_DEBUG(12, "kssl_alert_write: level=%d description=%d\n",
			alert->level, alert->description);

	if (alert->level == al_fatal)
		kssl_session_invalidate(cr->conn->sess_state.id);

	if (alert->level == al_fatal || alert->description == ad_close) {
		/* N.B. This does not actually close the connection.
		 * It just queues it to be closed. Which will occur
		 * after an attempt to send the alert has been made. */
		KSSL_DEBUG(2, "Sending Alert, closing connection: %p\n", 
				cr->conn);
		kssl_conn_close(cr->conn, KSSL_CONN_SSL_CLOSE);
	}

	kssl_record_head_set(cr, ct_alert, ALERT_NLEN);

	buf = cr->iov->iov_base;
	tls_head_to_buf(&(cr->record.head), buf);
	alert_to_buf(alert, buf + TLS_HEAD_NLEN);

	/* Fill out alert structure */
	if (cr->msg)
		kssl_message_destroy(cr->msg);
	cr->msg = kssl_message_create(ct_alert);
	if (!cr->msg) {
		KSSL_DEBUG(6, "kssl_alert_write:kssl_message_create fail\n");
		return -ENOMEM;
	}
	memcpy(&(cr->msg->data.alert), alert, sizeof(alert_t));

	return 0;
}


int kssl_alert_send(kssl_record_t *cr, alert_t *alert, int reuse)
{
	KSSL_DEBUG(12, "kssl_alert_send: level=%d description=%d\n",
			alert->level, alert->description);

	/* Do nothing if the level is 0  
	 * or the connectino is marked to be closed*/
	if (!alert->level || (cr->conn->flag & KSSL_CONN_SSL_CLOSE)) {
		KSSL_DEBUG(9, "kssl_alert_send: ignored\n");
		if (reuse)
			kssl_record_destroy(cr);
		return 0;
	}

	/* If a version has not been set for the connection yet then
	 * use the version in the incoming message. However,
	 * ALerts can only be sent in response to records in
	 * SSL3/TLS1 format -> we don't know how to format
	 * alerts for other record versions */
	if(!cr->conn->conn_state.version.major) {
		if(kssl_version_verify(&(cr->record.head.version))) {
			KSSL_DEBUG(6, "kssl_alert_send: "
					"kssl_version_verify fail\n");
			return(-EINVAL);
		}
		memcpy(&(cr->conn->conn_state.version), 
				&(cr->record.head.version), 
				sizeof(protocol_version_t));
	}

	return(kssl_record_build_send(cr, kssl_alert_write,
				ALERT_NLEN + TLS_HEAD_NLEN, alert, reuse));
}


int kssl_alert_process(kssl_record_t *cr, alert_t *alert) 
{
	/* If it is an alert, close */
	KSSL_DEBUG(12, "kssl_alert_process: level=%d description=%d: "
			"closing connection\n",
			alert->level, alert->description);

	if (alert->level == al_fatal)
		kssl_session_invalidate(cr->conn->sess_state.id);

	kssl_conn_close(cr->conn, KSSL_CONN_SSL_CLOSE);
	kssl_record_destroy(cr);
	return(0);
}

