/*
 * @file  conn.c
 * @brief the connection management component
 * @brief transmit data between client and realserver
 * @brief and calls protocol module to decide realserver
 * @brief also modify realserver data
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <assert.h>
#include <glib.h>
#include <limits.h>
#include "vanessa_logger.h"
#include "l7vs_config.h"
#include "l7vs_iomuxlist.h"
#include "l7vs_lsock.h"
#include "l7vs_conn.h"
#include "l7vs_service.h"
#include "l7vs.h"
#include "l7vs_module.h"
#include "l7vs_sched.h"

static int l7vs_conn_cl_callback(struct l7vs_iomux*);
static int l7vs_conn_client_receiving(struct l7vs_iomux*);
static int l7vs_conn_is_rs_connected(struct l7vs_iomux*, struct l7vs_conn*, struct l7vs_service**, struct l7vs_dest**);
static int l7vs_conn_connection_establish(struct l7vs_iomux*, struct l7vs_conn*, struct l7vs_service*, struct l7vs_dest*);
static int l7vs_conn_block(struct l7vs_iomux*, struct l7vs_iomux*);
static int l7vs_conn_busy(struct l7vs_iomux*);
static int l7vs_conn_sending(struct l7vs_iomux*, struct l7vs_iomux*);
static int l7vs_conn_sent_next_state(struct l7vs_iomux*, struct l7vs_iomux*);
static int l7vs_conn_rs_callback(struct l7vs_iomux* );
static int l7vs_conn_realserver_receiving(struct l7vs_iomux*);
static int l7vs_conn_realserver_modify_data(struct l7vs_iomux*);
static int l7vs_conn_recv(struct l7vs_iomux*, int);
static int l7vs_conn_preread(struct l7vs_iomux*, int);
static int l7vs_conn_send(struct l7vs_iomux*, int);
static int l7vs_conn_change_connect_rs(struct l7vs_conn*, struct l7vs_dest*);
static int l7vs_conn_QoS_control( struct l7vs_conn *conn );
static void l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize );

#define L7VS_CONN_READ_BUFSIZE         102400		//! conn read buffer size

/*!
 * connection create
 *
 * @param[in]	lfd	listening socket fd
 * @param[in]	lsock	listening socket
 * @return	connection
 */
struct l7vs_conn *
l7vs_conn_create(int lfd, struct l7vs_lsock *lsock)
{
        socklen_t len = 0;
        struct l7vs_conn *conn = NULL;
        int mss = 0;
        int ret = 0;
	int flags = 0;

	if (-1 == lfd) {
		VANESSA_LOGGER_ERR_UNSAFE("error / lsock fd is not specified at:%s:%d", __FILE__, __LINE__);
		goto create_error;
	}

	if (!lsock) {
		VANESSA_LOGGER_ERR_UNSAFE("error / lsock is null at:%s:%d", __FILE__, __LINE__);
		goto create_error;
	}
       
        conn = (struct l7vs_conn*)malloc(sizeof(struct l7vs_conn));
        if (!conn) {
		VANESSA_LOGGER_ERR_UNSAFE("error / can not allocate memory at:%s:%d", __FILE__, __LINE__);
		goto create_error;
	}
	memset(conn, 0, sizeof(struct l7vs_conn));

	conn->ciom = l7vs_iomux_get_from_avail_list();
	if (!conn->ciom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / can not get client iom at:%s:%d", __FILE__, __LINE__);
		goto create_error;
	}

	conn->riom = l7vs_iomux_get_from_avail_list();
	if (!conn->riom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / can not get realserver iom at:%s:%d", __FILE__, __LINE__);
		goto create_error;
	}

        conn->cldata_len = 0;
        conn->cldata_bufsize = L7VS_CONN_READ_BUFSIZE;
        conn->cldata = (char *)malloc(conn->cldata_bufsize);
        if (!conn->cldata) {
		VANESSA_LOGGER_ERR_UNSAFE("error / can not allocate memory for buffer at:%s:%d", __FILE__, __LINE__);
		goto create_error;
        }
	memset(conn->cldata, 0, conn->cldata_bufsize);

        len = sizeof(struct sockaddr_in);
        conn->ciom->fd = accept(lfd, (struct sockaddr *)&conn->caddr, &len);
        if (conn->ciom->fd < 0) {
		VANESSA_LOGGER_ERR_UNSAFE("error / accept error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
		goto create_error;
        }

	// get current flag
	flags = fcntl(conn->ciom->fd, F_GETFL, 0);
	if (0 > flags) {
		VANESSA_LOGGER_ERR_UNSAFE("error / fcntl(F_GETFL) error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
		goto create_error;
	}

	// set nonblocking
	flags = fcntl(conn->ciom->fd, F_SETFL, flags | O_NONBLOCK);
	if (0 > flags) {
		VANESSA_LOGGER_ERR_UNSAFE("error / fcntl(F_SETFL) error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
		goto create_error;
	}

	// get client mss
        len = sizeof(int);
        ret = getsockopt(conn->ciom->fd, IPPROTO_TCP, TCP_MAXSEG, &mss, &len);
        if (0 > ret) {
		VANESSA_LOGGER_ERR_UNSAFE("error / getsockopt TCP_MAXSEG error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
		goto create_error;
        }
        conn->cmss = mss;
        VANESSA_LOGGER_DEBUG_UNSAFE("client %s mss %d",
                                    inet_ntoa(conn->caddr.sin_addr), mss);

        conn->lsock = lsock;
        conn->proto = lsock->proto;
	conn->ciom->status = iomux_conn_receiving;
	conn->ciom->callback = l7vs_conn_cl_callback;
        conn->ciom->data = conn;
	conn->sorry_conn_flag = 0;
	conn->old_dest = NULL;
	l7vs_iomux_add(conn->ciom, iom_read);

        return conn;

create_error:
	if (conn) {
		if (conn->ciom) {
			if (-1 != conn->ciom->fd) {
				close(conn->ciom->fd);
			}
			l7vs_iomux_put_to_avail_list(conn->ciom);
		}
		if (conn->riom) {
			l7vs_iomux_put_to_avail_list(conn->riom);
		}
		if (conn->cldata) {
			free(conn->cldata);
		}
		free(conn);
	}
	return NULL;
}

/*!
 * connection destroy
 *
 * @param[in]	conn	connection
 * @return	void
 */
void
l7vs_conn_destroy(struct l7vs_conn *conn)
{
	if (!conn) {
		return;
	}

	if (conn->srv) {
		l7vs_service_remove_conn(conn->srv, conn);
		conn->srv = NULL;
	}

	if (conn->ciom) {
		if (-1 != conn->ciom->fd) {
			l7vs_conn_close_csock(conn);
		}
		l7vs_iomux_put_to_avail_list(conn->ciom);
		conn->ciom = NULL;
	}

	if (conn->riom) {
		if (-1 != conn->riom->fd) {
			l7vs_conn_close_rsock(conn);
		}
		l7vs_iomux_put_to_avail_list(conn->riom);
		conn->riom = NULL;
	}

	if (conn->cldata) {
		free(conn->cldata);
		conn->cldata = NULL;
	}

	free(conn);	
}

/*!
 * client connection callback
 *
 * @param[in]	iom	connection iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_cl_callback(struct l7vs_iomux *iom )
{
	struct l7vs_conn* conn = NULL;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	conn = (struct l7vs_conn *)iom->data;

	// invalid status check
	if (iomux_conn_receiving != iom->status &&
	    iomux_conn_block != iom->status &&
	    iomux_conn_busy != iom->status && 
	    iomux_conn_sending != iom->status && 
	    iomux_conn_sending_busy != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		l7vs_conn_destroy(conn);
		return 0;
	}

	// switch by current status
	switch (iom->status) {
	case iomux_conn_receiving:
		if (!l7vs_conn_client_receiving(iom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / cl_callback/client receiving failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		if (iomux_conn_disconnected == iom->status) {
			VANESSA_LOGGER_DEBUG("client connection disconnected");
			l7vs_conn_destroy(conn);
		}
		break;
	case iomux_conn_block:
		if (!l7vs_conn_block(iom, conn->riom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / cl_callback/block failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		break;
	case iomux_conn_busy:
		if (!l7vs_conn_busy(iom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / cl_callback/busy failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;	
		}
		break;		
	case iomux_conn_sending:
	  	if (!l7vs_conn_sending(iom, conn->riom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / cl_callback/sending failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		break;
	case iomux_conn_sending_busy:
		if (!l7vs_conn_sending(iom, conn->riom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / cl_callback/sending_busy failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		break;
	default:
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		l7vs_conn_destroy(conn);
		return 0;
	}

	return 1;
}

/*!
 * client receiving
 *
 * @param[in]	iom	connection iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_client_receiving(struct l7vs_iomux *iom)
{
	struct l7vs_conn *conn = NULL;
        struct l7vs_service *srv = NULL;
        struct l7vs_dest *dest = NULL;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_receiving != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	conn = (struct l7vs_conn *)iom->data;

	// when the real server is not set in read-blocked
	if (iomux_conn_block != conn->riom->status && iomux_conn_busy != conn->riom->status) {
		// set the real server in read-blocked
		conn->riom->status = iomux_conn_block;
	}

	// receive client data
	if (!l7vs_conn_recv(iom, iom->fd)) {
		VANESSA_LOGGER_ERR_UNSAFE("error / recv failed at:%s:%d", __FILE__, __LINE__);
		return 0;
	}
	switch (iom->status) {
	case iomux_conn_receiving:
		// data still remain, epoll re-registration for read
		l7vs_iomux_mod(iom, iom_read);
		break;
	case iomux_conn_received:
		// all data is received, service is selected by using the received data
		if (!l7vs_conn_is_rs_connected(iom, conn, &srv, &dest)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / is realserver connected failed at:%s:%d", __FILE__, __LINE__);
			return 0;
		}
		switch (iom->status) {
		case iomux_conn_block:
			//already decided real server
			return 1;
		case iomux_conn_service_selected:
			//service and destination were selected then attempt to connect the real server
			if (!l7vs_conn_connection_establish(iom, conn, srv, dest)) {
				VANESSA_LOGGER_ERR_UNSAFE("error / client connect rs failed at:%s:%d", __FILE__, __LINE__);
				return 0;
			}
			break;
		default:
			//invalid status
			VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
			return 0;
		}
		break;
	case iomux_conn_disconnected:
		// client connection was disconnected
		break;
	default:
		//invalid status
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	return 1;
}

/*!
 * check realserver fd, and if real server not connected, call select_service function
 *
 * @param[in]	iom	connection_iomux
 * @param[in]	conn	connection
 * @param[out]	srv	service
 * @param[out]	dest	destination(real server)
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_is_rs_connected(struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service **srv, struct l7vs_dest **dest)
{
	int ret = 0;
	int temp_tcps = 0;
	struct l7vs_dest *d;	// destination for sorry-server
	int sorry_state;	// sorry state

	struct l7vs_lsock *lsock = NULL;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (!conn) {
		VANESSA_LOGGER_ERR_UNSAFE("error / conn is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_received != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	lsock = conn->lsock;

	//is real server connected?
	if (-1 != conn->riom->fd) {
		// check sorry status of service
		// argment 0 means no connection count check
		sorry_state = l7vs_sched_sorry_check(conn->srv, 0);
		if (sorry_state) {
			// check connect to sorry-server now
			if (!conn->sorry_conn_flag) {
				// get sorry-server destination (argment 0)
				d = l7vs_sched_sorry_dest(conn->srv, conn, 0);
				if (!d) {
					VANESSA_LOGGER_ERR_UNSAFE("error / sorry-server dest NG at:%s:%d", __FILE__, __LINE__);
					iom->status = iomux_conn_rs_connect_error;
					return 0;
				}
				conn->old_dest = conn->dest;
				// change connection to sorry-server
				if (!l7vs_conn_change_connect_rs(conn, d)) {
					VANESSA_LOGGER_ERR_UNSAFE("error / change connection to Sorry NG at:%s:%d", __FILE__, __LINE__);
					iom->status = iomux_conn_rs_connect_error;
					return 0;
				}
				conn->sorry_conn_flag = 1;
			}
		} else {
			// check connect to sorry-server now
			if (conn->sorry_conn_flag) {
				// get old real-server destination (argment 1)
				d = l7vs_sched_sorry_dest(conn->srv, conn, 1);
				if (!d) {
					VANESSA_LOGGER_ERR_UNSAFE("error / old real-server dest NG at:%s:%d", __FILE__, __LINE__);
					iom->status = iomux_conn_rs_connect_error;
					return 0;
				}
				// change connection to real-server
				if (!l7vs_conn_change_connect_rs(conn, d)) {
					VANESSA_LOGGER_ERR_UNSAFE("error / change connection to RS NG at:%s:%d", __FILE__, __LINE__);
					iom->status = iomux_conn_rs_connect_error;
					return 0;
				}
				conn->sorry_conn_flag = 0;
			}
		}
		iom->status = iomux_conn_block;
		if (iomux_conn_busy == conn->riom->status) {
			conn->riom->status = iomux_conn_sending_busy;
		}
		else {
			conn->riom->status = iomux_conn_sending;
		}
		//epoll re-registration for write
		l7vs_iomux_mod(conn->riom, iom_write);
	}
	else {
		//select service
		ret = l7vs_lsock_select_service(lsock, conn, conn->cldata,
						conn->cldata_len, srv, dest, &temp_tcps);
		if (0 >= ret) {
			*srv = NULL;
			*dest = NULL;
			VANESSA_LOGGER_ERR_UNSAFE("error / no matching service found at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_receive_error;
			return 0;
		}
		iom->status = iomux_conn_service_selected;
	}

	return 1;
}

/*!
 * connecction establish
 *
 * @param[in]	iom	connection_iomux
 * @param[in]	conn	connection
 * @param[in]	srv	service
 * @param[in]	dest	destination(real server)
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_connection_establish(struct l7vs_iomux *iom, struct l7vs_conn *conn, struct l7vs_service *srv, struct l7vs_dest *dest)
{
	int ret = 0;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (!conn) {
		VANESSA_LOGGER_ERR_UNSAFE("error / conn is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (!srv) {
		VANESSA_LOGGER_ERR_UNSAFE("error / srv is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (!dest) {
		VANESSA_LOGGER_ERR_UNSAFE("error / dest is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iom->status != iomux_conn_service_selected) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	// connect to real server
	if (!l7vs_conn_connect_rs(conn, dest)) {
		VANESSA_LOGGER_ERR_UNSAFE("error / connect rs failed at:%s:%d", __FILE__, __LINE__);
		iom->status = iomux_conn_rs_connect_error;
		return 0;
	}
	iom->status = iomux_conn_rs_connected;

	// set pointer for cross reference
	conn->srv = srv;
	l7vs_service_register_conn(srv, conn);

	// service establish
	ret = l7vs_service_establish(conn->srv, conn);
	if (0 > ret) {
		VANESSA_LOGGER_ERR_UNSAFE("error / service establish failed at:%s:%d", __FILE__, __LINE__);
		iom->status = iomux_conn_rs_connect_error;
		return 0;
	}
	
	// iom status set 
	iom->status = iomux_conn_block;
	conn->riom->status = iomux_conn_sending;

	return 1;
}

/*!
 * block status:
 * when block status, arrived data turn next epoll reaction
 * and changes state to busy, expect the another iom is already busy
 *
 * @param[in]	iom		connection_iomux
 * @param[in]	another_iom	another side's iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_block(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom)
{
	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (!another_iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / another_iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_block != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	// epoll re-registraion for read
	l7vs_iomux_mod(iom, iom_read);

	// when another iom status is busy, do not change my status
	if (iomux_conn_busy != another_iom->status &&
	    iomux_conn_sending_busy != another_iom->status &&
	    iomux_conn_sent_busy != another_iom->status) {
		iom->status = iomux_conn_busy;
	}

	return 1;
}

/*!
 * busy status:
 * when busy status, arrived data turn next epoll reaction and status keep busy
 *
 * @param[in]	iom	connection_iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_busy(struct l7vs_iomux *iom)
{
	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_busy != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	//epoll re-registration for read
	l7vs_iomux_mod(iom, iom_read);

	return 1;
}

/*!
 * sending status:
 * 
 * @param[in]	iom		connection_iomux
 * @param[in]	another_iom	another side's iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_sending(struct l7vs_iomux *iom, struct l7vs_iomux *another_iom)
{
	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0; 
	}

	if (!another_iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / another_iom is null at:%s:%d", __FILE__, __LINE__);
		return 0; 
	}

	if (iomux_conn_sending != iom->status && iomux_conn_sending_busy != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	// send to handling iom
	if (!l7vs_conn_send(iom, iom->fd)) {
		iom->status = iomux_conn_send_error;
		VANESSA_LOGGER_ERR_UNSAFE("error / send failed at:%s:%d", __FILE__, __LINE__);
		return 0;
	}
	if (iomux_conn_sent == iom->status || iomux_conn_sent_busy == iom->status) {
		// when all data was sent, select the next state
		if (!l7vs_conn_sent_next_state(another_iom, iom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / send next state failed at:%s:%d", __FILE__, __LINE__);
			return 0;
		}
		if (iomux_conn_receiving != iom->status && iomux_conn_block != iom->status) {
			VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
			return 0;
		}
	}
	else if (iomux_conn_sending == iom->status || iomux_conn_sending_busy == iom->status) {
		// data still remain
		// epoll re-registration for write
		l7vs_iomux_mod(iom, iom_write);
	}
	else  {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	return 1;
}

/*!
 * decide sent next state
 *
 * @param[in,out]	src_iom		source connection_iomux
 * @param[in,out]	dest_iom	destionation connection_iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_sent_next_state(struct l7vs_iomux *src_iom, struct l7vs_iomux *dest_iom)
{
	if (!src_iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / src_iom is null at:%s:%d", __FILE__, __LINE__);
		return  0;
	}
	if (!dest_iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / dest_iom is null at:%s:%d", __FILE__, __LINE__);
		return  0;
	}

	// when writing, the dest_status will be "sent" or "sent_busy"
	if (iomux_conn_sent != dest_iom->status && iomux_conn_sent_busy != dest_iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid dest status(%d) at:%s:%d", dest_iom->status, __FILE__, __LINE__);
		return 0;
	}
	// and the src_status will be "block" or "busy"
	if (iomux_conn_block != src_iom->status && iomux_conn_busy != src_iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid src status(%d) at:%s:%d", src_iom->status, __FILE__, __LINE__);
		return 0;
	}

	// decide the next state
	if (iomux_conn_sent_busy == dest_iom->status) {
		// when receiving from source, request come to the destination, then destination has priority
		src_iom->status = iomux_conn_block;
		dest_iom->status = iomux_conn_receiving;
	}
	else if (iomux_conn_busy == src_iom->status) {
		// when sending to destination, request come to the source, then source has priority
		src_iom->status = iomux_conn_receiving;
		dest_iom->status = iomux_conn_block;
	}
	else {
		// request come to neither source nor destnation, ready to read both
		src_iom->status = iomux_conn_receiving;	
		dest_iom->status = iomux_conn_receiving;
	}
	l7vs_iomux_mod(src_iom, iom_read);
	l7vs_iomux_mod(dest_iom, iom_read);

	return 1;
}

/*!
 * realserver connection callback
 *
 * @param[in]	iom	connection iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_rs_callback(struct l7vs_iomux *iom)
{
	struct l7vs_conn* conn = NULL;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	conn = (struct l7vs_conn *)iom->data;

	// invalid status check
	if (iomux_conn_receiving != iom->status &&
	    iomux_conn_block != iom->status &&
	    iomux_conn_busy != iom->status && 
	    iomux_conn_sending != iom->status && 
	    iomux_conn_sending_busy != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		l7vs_conn_destroy(conn);
		return 0;
	}

	// switch by current status
	switch (iom->status) {
	case iomux_conn_receiving:
		if (!l7vs_conn_realserver_receiving(iom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / rs_callback/realserver receiving failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		if (iomux_conn_disconnected == iom->status) {
			VANESSA_LOGGER_DEBUG("realserver connection disconnected");
			l7vs_conn_destroy(conn);
		}
		break;
	case iomux_conn_block:
		if (!l7vs_conn_block(iom, conn->ciom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / rs_callback/block failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		break;
	case iomux_conn_busy:
		if (!l7vs_conn_busy(iom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / rs_callback/busy failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;	
		}
		break;		
	case iomux_conn_sending:
	  	if (!l7vs_conn_sending(iom, conn->ciom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / rs_callback/sending failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		break;
	case iomux_conn_sending_busy:
		if (!l7vs_conn_sending(iom, conn->ciom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / rs_callback/sending_busy failed at:%s:%d", __FILE__, __LINE__);
			l7vs_conn_destroy(conn);
			return 0;
		}
		break;
	default:
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		l7vs_conn_destroy(conn);
		return 0;
	}

	return 1;
}

/*!
 * realserver receiving 
 *
 * @param[in]	iom	connection iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_realserver_receiving(struct l7vs_iomux *iom)
{
	struct l7vs_conn *conn = NULL;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_receiving != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	conn = (struct l7vs_conn *)iom->data;

	// when the client is not set in read-blocked
	if (iomux_conn_block != conn->ciom->status && iomux_conn_busy != conn->ciom->status) {
		// set the real server in read-blocked
		conn->ciom->status = iomux_conn_block;
	}

	// receive realserver data
	if (!l7vs_conn_recv(iom, iom->fd)) {
		VANESSA_LOGGER_ERR_UNSAFE("error / recv failed at:%s:%d", __FILE__, __LINE__);
		return 0;
	}
	switch (iom->status) {
	case iomux_conn_receiving:
		// data still remain, epoll re-registration for read
		l7vs_iomux_mod(iom, iom_read);
		break;
	case iomux_conn_received:
		// all data is received, append the session infomation
		if (!l7vs_conn_realserver_modify_data(iom)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / realserver modify data failed at:%s:%d", __FILE__, __LINE__);
			return 0;
		}
		iom->status = iomux_conn_block;
		if (iomux_conn_busy == conn->ciom->status) {
			conn->ciom->status = iomux_conn_sending_busy;
		} else {
			conn->ciom->status = iomux_conn_sending;
		}
		//epoll re-registration for write
		l7vs_iomux_mod(conn->ciom, iom_write);
	case iomux_conn_disconnected:
		// client connection was disconnected
		break;
	default:
		//invalid status
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	return 1;
}

/*!
 * realserver modify data
 *
 * @param[in]	iom	connection iomux
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
static int
l7vs_conn_realserver_modify_data(struct l7vs_iomux *iom)
{
	struct l7vs_conn *conn = NULL;
        char *newbuf = NULL;
	int (*relayf)(struct l7vs_service *, struct l7vs_conn *, char *, size_t *) = NULL;
	int ret = 0;
	size_t len_ret = 0;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_received != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	conn = (struct l7vs_conn *)iom->data;

	//set the function pointer that modify the real server data
	if (conn->srv) {
		relayf = conn->srv->pm->analyze_rsdata;
	}

	if (relayf) {
		len_ret = conn->cldata_len;
		newbuf = (char *)realloc(conn->cldata, conn->cldata_len + L7VS_PROTOMOD_MAX_ADD_BUFSIZE + 1);
		if (!newbuf) {
			VANESSA_LOGGER_ERR_UNSAFE("error / realloc failed at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_buffer_realloc_error;
			return 0;
		}
		conn->cldata = newbuf;

		ret = (*relayf)(conn->srv, conn, conn->cldata, &len_ret);
		if (0 != ret) {
			VANESSA_LOGGER_ERR_UNSAFE("error / analyze rsdata failed at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_rsdata_modify_error;
			return 0;
		}
		if (L7VS_PROTOMOD_MAX_ADD_BUFSIZE < len_ret - conn->cldata_len) {
			VANESSA_LOGGER_ERR_UNSAFE("error / bufsize too long modified by protomod at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_rsdata_modify_error;
			return 0;
		}
		else {
			conn->cldata_len = len_ret;
		}
	}

	iom->status = iomux_conn_rsdata_modified;
	return 1;
}

/*!
 * receive data
 *
 * @param[in]	iom		connection_iomux
 * @param[in]	source_fd	the filedescriptor to be read
 * @return	status
 * @retval	0	failed 
 * @retval	1	succeed
 */
int
l7vs_conn_recv(struct l7vs_iomux *iom, int source_fd)
{
        int ret = 0;
	struct l7vs_conn *conn = NULL;
	char read_buf[L7VS_CONN_READ_BUFSIZE];
        char *newbuf = NULL;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (-1 == source_fd) {
		VANESSA_LOGGER_ERR_UNSAFE("error / source fd is not specified at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_receiving != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	conn = (struct l7vs_conn *)iom->data;

	//QoS Control
	if( !l7vs_conn_QoS_control( conn ) && (source_fd == conn->ciom->fd) ){
		iom->status = iomux_conn_receiving;
		return 1;
	}
	// read the data
	ret = recv(source_fd, read_buf, L7VS_CONN_READ_BUFSIZE - 1, 0);
	switch (ret) {
	case -1:
		if (EAGAIN != errno) {
			VANESSA_LOGGER_ERR_UNSAFE("error / recv error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
			iom->status = iomux_conn_receive_error;
			return 0;
		}
		return 1;
	case 0:
		VANESSA_LOGGER_DEBUG("peer disconnected");
		iom->status = iomux_conn_disconnected;
		return 1;
	case L7VS_CONN_READ_BUFSIZE - 1:
		// if the buffer is insufficient, realloc the buffer
		conn->cldata_bufsize += L7VS_CONN_READ_BUFSIZE - 1;
		newbuf = (char *)realloc(conn->cldata,
					 conn->cldata_bufsize);
		if (!newbuf) {
			VANESSA_LOGGER_ERR_UNSAFE("error / realloc failed at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_buffer_realloc_error;
			return 0;
		}
		conn->cldata = newbuf;

		// check the next data, and decide the status
		if (!l7vs_conn_preread(iom, source_fd)) {
			VANESSA_LOGGER_ERR_UNSAFE("error / preread failed at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_receive_error;
			return 0;
		}
		if (iomux_conn_disconnected == iom->status) {
			return 1;
		}
		break;
	default:
		// reading data is finished
		iom->status = iomux_conn_received;
		break;
	}
	
	// register reveive information for QoS
	if( source_fd == conn->ciom->fd ){
		l7vs_conn_QoS_recvsize_register( conn, ret );
	}
	// copy to conn_buffer
	memcpy(conn->cldata + conn->cldata_len, read_buf, ret);
	conn->cldata_len += ret;
	conn->cldata[conn->cldata_len] = '\0';

	return 1;
}

/*!
 * pre-read whether next data exists
 *
 * @param[in]	iom		connection iomux
 * @param[in]	source_fd	source file description
 * @return	status
 * @retval	0	failed
 * @retval	1	succeed
 */
int
l7vs_conn_preread(struct l7vs_iomux *iom, int source_fd)
{
	int preread_ret = 0;
	char preread_buf[1] = {0};

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (-1 == source_fd) {
		VANESSA_LOGGER_ERR_UNSAFE("error / source fd is not specified at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_receiving != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	preread_ret = recv(source_fd, preread_buf, 1, MSG_PEEK);
	switch (preread_ret) {
	case -1:
		if (EAGAIN != errno) {
			VANESSA_LOGGER_ERR_UNSAFE("error / recv error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
			iom->status = iomux_conn_receive_error;
			return 0;
		}
		// finished reading all data
		iom->status = iomux_conn_received;
		break;
	case 0:
		VANESSA_LOGGER_DEBUG("peer disconnected");
		iom->status =  iomux_conn_disconnected;
		break;
	case 1:
		// the data that should be read remains
		// status does not change
		break;
	default:
		VANESSA_LOGGER_ERR_UNSAFE("error / preread error at:%s:%d", __FILE__, __LINE__);
		return 0;
	}
	return 1;
}

/*!
 * send data
 *
 * @param[in]	iom	connection_iomux
 * @param[in]	dest_fd	the filedescriptor to be write
 * @return	status
 * @retval	0	failed 
 * @retval	1	succeed
 */
int
l7vs_conn_send(struct l7vs_iomux *iom, int dest_fd)
{
	int ret = 0;
	struct l7vs_conn *conn = NULL;
	char *newbuf = NULL;
	size_t sent_len = 0;

	if (!iom) {
		VANESSA_LOGGER_ERR_UNSAFE("error / iom is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (-1 == dest_fd) {
		VANESSA_LOGGER_ERR_UNSAFE("error / dest fd is not specified at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

	if (iomux_conn_sending != iom->status &&
	    iomux_conn_sending_busy != iom->status) {
		VANESSA_LOGGER_ERR_UNSAFE("error / invalid status(%d) at:%s:%d", iom->status, __FILE__, __LINE__);
		return 0;
	}

	conn = (struct l7vs_conn *)iom->data;

	// write the data
	ret = send(dest_fd, conn->cldata, conn->cldata_len, MSG_NOSIGNAL);
	if (0 > ret) {
		VANESSA_LOGGER_ERR_UNSAFE("error / send error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
		iom->status = iomux_conn_send_error;
		return 0;
	}
	sent_len = ret;
	VANESSA_LOGGER_DEBUG_UNSAFE("sent %d bytes", sent_len);

	// decide the status
	if (conn->cldata_len == sent_len) {
		// finished sending all data
		if (iomux_conn_sending == iom->status) {
			iom->status = iomux_conn_sent;
		} else {
			iom->status = iomux_conn_sent_busy;
		}
		//cldata memory realloc
		conn->cldata_len = 0;
		conn->cldata_bufsize = L7VS_CONN_READ_BUFSIZE;
		newbuf = (char *)malloc(conn->cldata_bufsize);
		if (!newbuf) {	
			VANESSA_LOGGER_ERR_UNSAFE("error / malloc failed at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_send_error;
			return 0;
		}
		free(conn->cldata);
		conn->cldata = newbuf;
		memset(conn->cldata, 0, L7VS_CONN_READ_BUFSIZE);
	}
	else {
		// shift data for next sending
		newbuf = (char *)malloc((conn->cldata_len - sent_len) + 1);
		if (!newbuf) {	
			VANESSA_LOGGER_ERR_UNSAFE("error / malloc failed at:%s:%d", __FILE__, __LINE__);
			iom->status = iomux_conn_send_error;
			return 0;
		}
		memcpy(newbuf, (conn->cldata + sent_len), conn->cldata_len - sent_len);
		free(conn->cldata);
		conn->cldata = newbuf;
		conn->cldata_len -= sent_len;
		conn->cldata[conn->cldata_len] = '\0';

		// the data that should be sent remains, status does not change
	}

	return 1;

}

/*!
 * connect to real server
 *
 * @param[in]	conn	connection
 * @param[in]	dest	destination
 * @return	status
 * @retval	0	failed 
 * @retval	1	succeed
 */
int
l7vs_conn_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest)
{
        int s;
        int stype;
        int flags;
        int ret;
	enum iomaction	action;

	if (!conn) {
		VANESSA_LOGGER_ERR_UNSAFE("error / conn is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}
	if (!dest) {
		VANESSA_LOGGER_ERR_UNSAFE("error / dest is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}

        switch (conn->proto) {
        case IPPROTO_TCP:
                stype = SOCK_STREAM;
                break;
        case IPPROTO_UDP:
                stype = SOCK_DGRAM;
                break;
        default:
		VANESSA_LOGGER_ERR_UNSAFE("error / unknown socket type at:%s:%d", __FILE__, __LINE__);
                return 0;
        }

        s = socket(PF_INET, stype, conn->proto);  
        if (0 > s) {
		VANESSA_LOGGER_ERR_UNSAFE("error / socket error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
                return 0;
        }
        
        flags = fcntl(s, F_GETFL, 0);
        if (0 > flags) {
		VANESSA_LOGGER_ERR_UNSAFE("error / fcntl(F_GETFL) error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
                close(s);
                return 0;
        }

        flags = fcntl(s, F_SETFL, flags | O_NONBLOCK);
        if (0 > flags) {
		VANESSA_LOGGER_ERR_UNSAFE("error / fcntl(F_SETFL) error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
                close(s);
                return 0;
        }

        if (conn->cmss > 0) {
                ret = setsockopt(s, IPPROTO_TCP, TCP_MAXSEG, &conn->cmss,
                                 sizeof(int));
                if (ret < 0) {
			VANESSA_LOGGER_ERR_UNSAFE("warning / setsockopt(TCP_MAXSEG) error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
                }
        }

        ret = connect(s, (struct sockaddr *)&dest->addr,
                      sizeof(struct sockaddr_in)); 
        if (0 > ret) {
                if (errno != EINPROGRESS) {
			VANESSA_LOGGER_ERR_UNSAFE("error / connect error(%s) at:%s:%d", strerror(errno), __FILE__, __LINE__);
                        close(s);
                        return 0;
                }
	}
	VANESSA_LOGGER_DEBUG_UNSAFE("dest %p nactive %d->%d",
				    dest, dest->nactive,
				    dest->nactive + 1);

	action = iom_write;
	dest->nactive++;   
        conn->dest = dest;   
        conn->riom->fd = s;
        conn->riom->callback = l7vs_conn_rs_callback;
        conn->riom->data = conn;
	l7vs_iomux_add(conn->riom, action);

        return 1;
}

/*!
 * close client socket 
 *
 * @param[in]	conn	connection
 * @return	void 
 */
void
l7vs_conn_close_csock(struct l7vs_conn *conn)
{
	if (!conn) {
		return;
	}
	if (!conn->ciom) {
		return;
	}
	if (-1 == conn->ciom->fd) {
		return;
	}

	l7vs_iomux_remove(conn->ciom);
	close(conn->ciom->fd);
	conn->ciom->fd = -1;
}

/*!
 * close real server socket
 *
 * @param[in]	conn	connection
 * @return	void 
 */
void
l7vs_conn_close_rsock(struct l7vs_conn *conn)
{
	if (!conn) {
		return;
	}
	if (!conn->riom) {
		return;
	}
	if (-1 == conn->riom->fd) {
		return;
	}

	if (conn->dest) {
		VANESSA_LOGGER_DEBUG_UNSAFE("dest %p nactive %d->%d",
					    conn->dest, conn->dest->nactive,
					    conn->dest->nactive - 1);
		conn->dest->nactive--;
		conn->dest->ninact++;
		if(conn->dest->ninact == INT_MAX) {
			conn->dest->ninact = 0;
		}
	}

	l7vs_iomux_remove(conn->riom);
	close(conn->riom->fd);
	conn->riom->fd = -1;
}

/*!
 * confirmation of whether conn is closed
 *
 * @param[in]	conn	connection
 * @return	closed or not
 * @retval	0	not closed 
 * @retval	1	closed
 */
int
l7vs_conn_closed(struct l7vs_conn *conn)
{
	if (!conn) {
		return 1;
	}
	if ((conn->ciom && -1 != conn->ciom->fd) || (conn->riom && -1 != conn->riom->fd)) {
		return 0;
	} else {
		return 1;
	}
}

/*!
 * Change connection to specified destination (to real-server or sorry-server).
 *
 * @param[in]	*conn		connection pointer
 * @param[in]	*dest		destination pointer
 * @return	int		change connection result 0=NG, 1=OK
 */
static int l7vs_conn_change_connect_rs(struct l7vs_conn *conn, struct l7vs_dest *dest)
{
	if (!conn) {
		VANESSA_LOGGER_ERR_UNSAFE("error / conn is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}
	if (!dest) {
		VANESSA_LOGGER_ERR_UNSAFE("error / dest is null at:%s:%d", __FILE__, __LINE__);
		return 0;
	}
	// save old destination
	conn->old_dest = conn->dest;
	// close real-server connection
	l7vs_conn_close_rsock(conn);
	// connect to new destination
	return l7vs_conn_connect_rs(conn, dest);
}

/*!
 * Judges from the data size received last time and recieve-interval
 *
 * @param[in]	*conn		connection pointer
 * @return	int		permitte for recv function call 0=NG(not permitted), 1=OK(permitted)
 */
int
l7vs_conn_QoS_control( struct l7vs_conn *conn )
{
	if (!conn) {
		return 1;
	}
	if (!conn->srv) {
		return 1;
	}

	struct  timeval curr_time;

	gettimeofday( &curr_time, NULL );
	unsigned long long	diff_usec;
	unsigned long long	tmp_size;

	//QoS Control check : service
	diff_usec = ((curr_time.tv_sec * 1000000ULL) + curr_time.tv_usec) - ((conn->srv->recvtime.tv_sec * 1000000ULL) + conn->srv->recvtime.tv_usec);
	if( !diff_usec ){
		VANESSA_LOGGER_DEBUG("recv interval time is 0");
		VANESSA_LOGGER_DEBUG("QoS control intervened in service");
		return 0;
	}
	tmp_size = conn->srv->recvsize * 1000000ULL;
	if( !tmp_size ){
		return 1;
	}
	VANESSA_LOGGER_DEBUG_UNSAFE("service_qos_setting:%lld", conn->srv->QoS_threshold_service);
	VANESSA_LOGGER_DEBUG_UNSAFE("persec:%lld", (tmp_size / diff_usec));
	if( ((tmp_size / diff_usec) > conn->srv->QoS_threshold_service) && (conn->srv->QoS_threshold_service) ){
		VANESSA_LOGGER_DEBUG("QoS control intervened in service");
		return 0;
	}

	//QoS Control check : connection
	diff_usec = ((curr_time.tv_sec * 1000000ULL) + curr_time.tv_usec) - ((conn->recvtime.tv_sec * 1000000ULL) + conn->recvtime.tv_usec);
	if( !diff_usec ){
		VANESSA_LOGGER_DEBUG("recv interval time is 0");
		VANESSA_LOGGER_DEBUG("QoS control intervened in conn");
		return 0;
	}
	tmp_size = conn->recvsize * 1000000ULL;
	if( !tmp_size ){
		return 1;
	}
	VANESSA_LOGGER_DEBUG_UNSAFE("conn_qos_setting:%lld", conn->srv->QoS_threshold_conn);
	VANESSA_LOGGER_DEBUG_UNSAFE("persec:%lld", (tmp_size / diff_usec));
	if( ((tmp_size / diff_usec) > conn->srv->QoS_threshold_conn) && (conn->srv->QoS_threshold_conn) ){
		VANESSA_LOGGER_DEBUG("QoS control intervened in conn");
		return 0;
	}
	return 1;
}

/*!
 * Regsiter recive time and recieved data size
 *
 * @param[in]	*conn		connection pointer
 * @param[in]	in_recvsize	recieved size
 * @return	void 
 */
void
l7vs_conn_QoS_recvsize_register( struct l7vs_conn *conn, const size_t in_recvsize )
{
	if (!conn) {
		return;
	}
	if (!conn->srv) {
		return;
	}
	struct	timeval CurrTime;
	gettimeofday( &CurrTime, NULL );
	conn->recvtime	= CurrTime;
	conn->recvsize	= in_recvsize;
	conn->srv->recvtime	= CurrTime;
	conn->srv->recvsize	= in_recvsize;
}

