/*! @file
 *  @brief Sync structure program
 *
 *  @author NTT COMWARE
 *  @date 2007-05-18
 *  @version 0.1
 */
#include <time.h>
#include <netdb.h>
#include <sys/socket.h>
#include "l7vs_sync.h"
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

//! Socket property 
enum sync_sock_prop{
	SYNC_NOT_READY,             //!< Socket is not opened
	SYNC_SEND_READY,            //!< Send socket
	SYNC_RECV_READY             //!< Receive socket
};

//! Latest transfered data structure
static struct sync_send_data latest_data;

//! Internal value structure
static struct sync_internal_val
{
	int file_descriptor;                    //!< Opened socket file descriptor
	struct addrinfo *address_info;          //!< Socket property
	enum sync_sock_prop initialize_flag;    //!< Initialize flag
} internal_val;

/*! Make serial number
 *
 *  @return Serial number
 *  @retval nonzero Serial number
 *  @retval 0 Error
 */
unsigned long long int make_serial(void);

/*! change to big endian (long long int)
 *
 *  @param  serial Serial number (little endian)
 *
 *  @return Serial number (big endian)
 */
unsigned long long int hton_ll(const unsigned long long int serial);

/*! change to little endian (long long int)
 *
 *  @param  serial Serial number (big endian)
 *
 *  @return Serial number (little endian)
 */
unsigned long long int ntoh_ll(const unsigned long long int serial);

/*! Initialize network socket
 *
 *  @param ip_addr Destination IP address (IPv4 or IPv6)
 *  @param servname Destination service name or port number
 *  @param nic Network device (ex. eth0, bond0, hme0, etc.)
 *  @param mode Operation mode (0 : send, 1: receive)
 *  @param fd File descriptor.
 *
 *  @return Initialize results. (SYNC_OK/SYNC_INITIALIZE_ALREADY/SYNC_SOCK_NG/SYNC_BIND_NG)
 *  @retval SYNC_OK Everything is ok
 *  @retval SYNC_INITIALIZE_ALREADY Network socket had already been initialized.
 *  @retval SYNC_SOCK_NG socket() NG
 *  @retval SYNC_BIND_NG bind() NG
 */
enum sync_err_t sync_sock_init(const char *ip_addr, const char *servname, const char *nic,
			enum sync_operation mode, int *fd)
{
	struct addrinfo hints;
	int ret;

	// check by continuous initialize.
	if ( internal_val.initialize_flag != SYNC_NOT_READY ) {
		// initialization has already been done.
		// return SYNC_INITIALIZE_ALREADY.
		return SYNC_INITIALIZE_ALREADY;
	}

	// Argument check (fd is don't care)
	// mode is not SYNC_SEND and SYNC_RECV.
	if ( mode != SYNC_SEND && mode != SYNC_RECV ) {
		return SYNC_UNKNOWN_MODE;
	}
	// ip_adde is NULL. (when mode = SYNC_SEND)
	if ( mode == SYNC_SEND && ip_addr == NULL ) {
		return SYNC_IP_NULL;
	}
	// servname is NULL.
	if ( servname == NULL ) {
		return SYNC_SERVNAME_NULL;
	}
	// servname is zero
	if ( strcmp(servname, "0") == 0 ) {
		return SYNC_PORT_ZERO;
	}
	// NIC is NULL.
	if ( nic == NULL ) {
		return SYNC_NIC_NULL;
	}

	// initialize structure
	memset(&latest_data, 0, sizeof(struct sync_send_data));
	memset(&hints, 0, sizeof(struct addrinfo));

	// set address hints
	hints.ai_family = AF_UNSPEC;     // permit any protocol (IPv4, IPv6, etc.)
	hints.ai_socktype = SOCK_DGRAM;  // UDP

	// get address infomation
	// send mode
	if (mode == SYNC_SEND) {
		ret = getaddrinfo(ip_addr, servname, &hints, &internal_val.address_info);
		if (ret != 0) {
			if ( ret == EAI_SERVICE ) {
				return SYNC_SERVNAME_NG;
			}
			// Temporary failure in name resolution
			if ( ret == EAI_AGAIN ) {
				return SYNC_IP_NG;
			}
			return SYNC_IP_NG;
		}
	}
	else if(mode == SYNC_RECV) {
		hints.ai_flags = AI_PASSIVE; // same as IN_ADDR_ANY(IPv4), IN6ADDR_ANY_INIT(IPv6)
		ret = getaddrinfo(NULL, servname, &hints, &internal_val.address_info);
		if (ret != 0) {
			//  Servname not supported for ai_socktype
			if ( ret == EAI_SERVICE ) {
				return SYNC_SERVNAME_NG;
			}
			return SYNC_IP_NG;
		}
	}
	
	// make a socket
	internal_val.file_descriptor = socket( internal_val.address_info->ai_family,
                                           internal_val.address_info->ai_socktype,
                                           internal_val.address_info->ai_protocol);
	// if cannot make a socket
	if ( internal_val.file_descriptor == -1 ) {
		return SYNC_SOCK_NG;
	}

	// set a initialaize_flag.
	// send mode.
	if ( mode == SYNC_SEND ) {
		internal_val.initialize_flag = SYNC_SEND_READY;
	}
	// receive mode
	else if ( mode == SYNC_RECV ) {
		internal_val.initialize_flag = SYNC_RECV_READY;
	}

	// set a NIC
	if ( mode == SYNC_SEND ) {
		if ( setsockopt(internal_val.file_descriptor, SOL_SOCKET, SO_BINDTODEVICE, nic, strlen(nic) + 1) == -1 ) {
			// cannot set a NIC
			// call sync_sock_fini(end function), and close a socket.
			sync_sock_fini();
			return SYNC_NIC_NG;
		}
	}

	//! assigns a name to an unnamed socket
	if ( mode == SYNC_RECV ) {
		ret = bind(internal_val.file_descriptor, internal_val.address_info->ai_addr, internal_val.address_info->ai_addrlen);
		//! error
		if ( ret != 0 ) {
			//! call sync_sock_fini(end function), and close a socket.
			sync_sock_fini();
			return SYNC_BIND_NG;
		}
	}

	//! return file descriptor
	*fd = internal_val.file_descriptor;

	// initialize complete
	return SYNC_OK;
}

/*! Send transfer data to standby server
 *
 *  @param data Points to input data from external program. This will be send to standby server.
 *
 *  @return Send results.
 *  @retval SYNC_OK Everything is ok.
 *  @retval SYNC_INITIALIZE_YET Network socket hadn't been initialized yet.
 *  @retval SYNC_NOT_SEND_MODE Initialize flag is not send flag.
 *  @retval SYNC_DATA_NULL Points to send data is NULL.
 *  @retval SYNC_SIZE_OVER Data size is over.
 *  @retval SYNC_SEND_NG sendto() NG
 */
enum sync_err_t sync_send_to(const struct sync_data *send_data)
{
	int	send_bytes;
	struct sync_send_data data;
	unsigned long long int ser;

	// check initialize flag
	if ( internal_val.initialize_flag == SYNC_NOT_READY ) {
		// initialization has already been done.
		return SYNC_INITIALIZE_YET;
	}

	// initialize flag is not SYNC_SEND_READY
	if ( internal_val.initialize_flag != SYNC_SEND_READY ) {
		return SYNC_NOT_SEND_MODE;
	}

	// data is NULL.
	if (send_data == NULL) {
		return SYNC_DATA_NULL;
	}

	// data size is over
	if (send_data->size > DATA_SIZE) {
		return SYNC_SIZE_OVER;
	}

	// make a send data
	// change to a big endian.
	ser = make_serial();
	data.serial = hton_ll(ser);

	// part of size.
	data.size = sizeof(struct sync_data);
	data.size = htons(data.size);
	// part of data.
	memcpy((char *)data.word, (char *)send_data, sizeof(struct sync_data));

	// send data
	send_bytes = sendto(internal_val.file_descriptor, &data, SEND_DATA_SIZE, 0,
				 internal_val.address_info->ai_addr, internal_val.address_info->ai_addrlen);

	// send data fails.
	if (send_bytes == -1){
		return SYNC_SEND_NG;
	}

	// send data complete
	return SYNC_OK;
}

/*! Receive transfer data from active server
 *
 *  @param  recv_data Points to output data from external program.
 *
 *  @return Send results.
 *  @retval SYNC_OK Everything is ok.
 *  @retval SYNC_INITIALIZE_YET Network socket hadn't been initialized yet.
 *  @retval SYNC_NOT_RECV_MODE Initialize flag is not receive flag.
 *  @retval SYNC_DATA_NULL Points to receive data is NULL.
 *  @retval SYNC_RECV_NG recvfrom() NG
 *  @retval SYNC_SIZE_NG Size of received data is different from send size.
 */ 
enum sync_err_t sync_recv_from(struct sync_data *recv_data)
{
	int	recv_bytes;
	struct sync_send_data sync_data;
	char data[568];

	// check by continuous initialize.
	// initialization has already been done.
	if ( internal_val.initialize_flag == SYNC_NOT_READY ) {
		return SYNC_INITIALIZE_YET;
	}
	// initialize flag is not SYNC_RECV_READY
	if ( internal_val.initialize_flag != SYNC_RECV_READY ) {
		return SYNC_NOT_RECV_MODE;
	}

	// DATA is NULL
	if ( recv_data == NULL ) {
		return SYNC_DATA_NULL;
	}

	// receive data
//	recv_bytes = recvfrom( internal_val.file_descriptor, data, sizeof(struct sync_send_data), MSG_DONTWAIT, NULL, NULL );
	recv_bytes = recvfrom( internal_val.file_descriptor, data, sizeof(struct sync_send_data), 0, NULL, NULL );

	// receive data fails.
	if ( recv_bytes < 0 ) {
		if ( latest_data.size == 0 ) {
			return SYNC_RECV_NG;
		}
		memcpy(recv_data, latest_data.word, sizeof(struct sync_data));
	} else {
		memcpy((struct sync_send_data *)&sync_data, (struct sync_send_data *)&data, sizeof(struct sync_send_data));
		sync_data.size = ntohs(sync_data.size);
		sync_data.serial = ntoh_ll(sync_data.serial);
	
		// compare size @@check@@ (over size)
		if (sync_data.size >= SEND_DATA_SIZE) {
			if ( latest_data.size == 0 ) {
				return SYNC_RECV_NG;
			}
			memcpy(recv_data, latest_data.word, sizeof(struct sync_data));
		// compare serial (serial is old.)
		} else if ( sync_data.serial <= ntoh_ll(latest_data.serial) ) {
			if ( latest_data.size == 0 ) {
				return SYNC_RECV_NG;
			}
			memcpy(recv_data, latest_data.word, sizeof(struct sync_data));
		} else {
			// extract data.
			memcpy(recv_data, sync_data.word, sizeof(struct sync_data));
			// refresh latest data
			memcpy(&latest_data, data, sizeof(struct sync_send_data));
		}
	}

	// receive data complate
	return SYNC_OK;
}

/*! Finalize network socket
 *
 *  @return Finalize results.
 *  @retval SYNC_OK Everything is ok.
 *  @retval SYNC_INITIALIZE_YET Network socket hadn't been initialized yet.
 *  @retval SYNC_CLOSE_NG close() NG
 */
enum sync_err_t sync_sock_fini(void)
{
	// check by continuous initialize.
	if ( internal_val.initialize_flag == SYNC_NOT_READY ) {
		// initialization has already been done.
		// return SYNC_INITIALIZE_ALREADY.
		return SYNC_INITIALIZE_YET;
	}

	// close a socket.
	if ( close(internal_val.file_descriptor) == -1 ) {
		// failre.
		// return SYNC_CLOSE_NG.
		return SYNC_CLOSE_NG;
	}
	// success.
	// addrinfo free.
	freeaddrinfo(internal_val.address_info);
	// initialize initialize_flag.
	internal_val.initialize_flag = SYNC_NOT_READY;
	// return SYNC_OK.
	return SYNC_OK;
}

/*! Make serial number
 *
 *  @return Serial number
 *  @retval nonzero Serial number
 *  @retval 0 Error
 */
unsigned long long int make_serial(void)
{
	unsigned long long int serial_num;
        struct timespec current_time;

	// get time by clock_gettime
        if ( clock_gettime(CLOCK_REALTIME, &current_time) == -1 ) {
		// failre.
		// return 0.
		serial_num = 0;
	} else {
		// make a serial succeeds.
		// return serial number.
		serial_num = (unsigned long long int) current_time.tv_sec * 1000000 +
		             (unsigned long long int) current_time.tv_nsec / 1000;
	}

	return serial_num;
}

/*! change to big endian (long long int)
 *
 *  @param  serial Serial number (little endian)
 *
 *  @return Serial number (big endian)
 */
unsigned long long int hton_ll(const unsigned long long int serial)
{
        unsigned long long int ret;
        long *p1,*p2;

        p1 = (long *) ((void *) &serial);
        p2 = (long *) ((void *) &serial) + 1;

        *p1 = htonl(*p1);
        *p2 = htonl(*p2);

        *((long *) ((void *) &ret)) = *p2;
        *((long *) ((void *) &ret) + 1) = *p1;

        return ret;
}

/*! change to little endian (long long int)
 *
 *  @param  serial Serial number (big endian)
 *
 *  @return Serial number (little endian)
 */
unsigned long long int ntoh_ll(const unsigned long long int serial)
{
        unsigned long long int ret;
        long *p1,*p2;

        p1 = (long *) ((void *) &serial);
        p2 = (long *) ((void *) &serial) + 1;

        *p1 = ntohl(*p1);
        *p2 = ntohl(*p2);

        *((long *) ((void *) &ret)) = *p2;
        *((long *) ((void *) &ret) + 1) = *p1;

        return ret;
}
