/*
 * @file  sched.c
 * @brief the framework module of scheduer 
 * @brief it proceeds common function of scheduler 
 *
 * L7VSD: Linux Virtual Server for Layer7 Load Balancing
 * Copyright (C) 2005  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include "vanessa_logger.h"
#include "l7vs_sched.h"
#include "l7vs_service.h"
#include "l7vs_module.h"
#include "l7vs_dest.h"

static struct l7vs_scheduler *l7vs_sched_load(char *name);
static void l7vs_sched_unload(struct l7vs_scheduler *sched);
static struct l7vs_scheduler *l7vs_sched_lookup(char *name);
static gint l7vs_sched_cmp(struct l7vs_scheduler *sched, char *name);

static GList *l7vs_sched_list = NULL;

/*!
 * Get scheduler module.
 * @param[in]	*name		scheduler module name
 * @return	struct l7vs_scheduler*	OK=scheduler pointer, NG=NULL
 */
struct l7vs_scheduler *
l7vs_sched_get(char *name)
{
        struct l7vs_scheduler *sched;

	// argment check
	if (!name) {
		VANESSA_LOGGER_ERR("Scheduler module name is NULL");
		return NULL;
	}
        sched = l7vs_sched_lookup(name);
        if (sched == NULL) {
                sched = l7vs_sched_load(name);
                if (sched == NULL) {
                        VANESSA_LOGGER_ERR("Scheduler module not found"
                                           " (maybe module problem)");
                        return sched;
                }
                sched->refcnt = 0;
        }

        sched->refcnt++;
        return sched;
}

/*!
 * Put scheduler module.
 * @param[in]	*sched		scheduler pointer
 * @return	void
 */
void
l7vs_sched_put(struct l7vs_scheduler *sched)
{
	// argument check
	if (sched) {
        	if (--sched->refcnt <= 0) {
        	        l7vs_module_remove(&l7vs_sched_list, sched);
        	        l7vs_sched_unload(sched);
        	}
	}
}

/*!
 * Bind scheduler module to service.
 * @param[in]	*sched		scheduler pointer
 * @param[in]	*svc		service pointer
 * @return	void
 */
void
l7vs_sched_bind(struct l7vs_scheduler *sched,
                struct l7vs_service *svc)
{
	// argument check
	if (sched && svc) {
        	svc->scheduler = sched;
        	if (sched->bind != NULL) {
                	sched->bind(svc);
        	}
	}
}

/*!
 * Unbind scheduler module form service.
 * @param[in]	*sched		scheduler pointer
 * @param[in]	*svc		service pointer
 * @return	void
 */
void
l7vs_sched_unbind(struct l7vs_scheduler *sched,
                  struct l7vs_service *svc)
{
	// argument check
	if (sched && svc) {
        	svc->scheduler = NULL;
        	if (sched->unbind != NULL) {
                	sched->unbind(svc);
        	}
	}
}

/*!
 * Load scheduler module.
 * @param[in]	*name		scheduler module name
 * @return	struct l7vs_scheduler*	OK=scheduler pointer, NG=NULL
 */
static struct l7vs_scheduler *
l7vs_sched_load(char *name)
{
        struct l7vs_scheduler *sched;

	// argment check
	if (!name) {
		VANESSA_LOGGER_ERR("Scheduler module name is NULL");
		return NULL;
	}
        sched = (struct l7vs_scheduler *)l7vs_module_load(name, "sched");
        if (sched != NULL) {
                l7vs_module_register(&l7vs_sched_list, sched);
        }

        return sched;
}

/*!
 * Unload scheduler module.
 * @param[in]	*sched		scheduler pointer
 * @return	void
 */
static void
l7vs_sched_unload(struct l7vs_scheduler *sched)
{
        void *h;

	// argument check
	if (sched) {
	        h = sched->handle;
		if (sched->fini) {
		        sched->fini();
		}
	        l7vs_module_unload(h);
	}
}

/*!
 * Lookup scheduler module in sched_list.
 * @param[in]	*name		scheduler module name
 * @return	struct l7vs_scheduler*	found=scheduler pointer, not found=NULL
 */
static struct l7vs_scheduler *
l7vs_sched_lookup(char *name)
{
	// argment check
	if (!name) {
		VANESSA_LOGGER_ERR("Scheduler module name is NULL");
		return NULL;
	}
        return (struct l7vs_scheduler *)
                l7vs_module_lookup(l7vs_sched_list, name,
                                   (GCompareFunc)l7vs_sched_cmp);
}

/*!
 * Compare scheduler module name.
 * @param[in]	*sched		scheduler pointer
 * @param[in]	*name		scheduler module name
 * @return	gint		compare result match=0, not match=-1 or other
 */
static gint
l7vs_sched_cmp(struct l7vs_scheduler *sched, char *name)
{
	// argument check
	if (sched) {
		if (sched->modname && name) {
		        return strcmp(sched->modname, name);
		}
	}
	return -1;
}

/*!
 * Sorry status of the specified service is checked.
 * @param[in]	*srv		service pointer
 * @param[in]	cc_check_flag	connection count check flag
 * @return	int		check result 0=not sorry or check NG , 1=sorry
 */
int
l7vs_sched_sorry_check(struct l7vs_service *srv, int cc_check_flag)
{
	GList *l;
	struct l7vs_dest *d;
	int dest_cc = 0;

	// argument check
	if (!srv) {
		VANESSA_LOGGER_ERR("Argument srv is NULL pointer");
		return 0;
	}
	if (cc_check_flag != 0 && cc_check_flag != 1) {
		VANESSA_LOGGER_ERR("Invalid cc_check_flag value");
		return 0;
	}

	// check sorry-flag of service
	if (srv->sorry_flag) {
		return 1;
	}

	// check valid destination of service
	if (!srv->dest_list) {
		return 1;
	}

	// not continuous check if cc_check_flag==0
	if (!cc_check_flag) {
		return 0;
	}

	// srv->sorry_cc not set then not sorry
	if (srv->sorry_cc == 0) {
		return 0;
	}
	// check connection count in all real servers of service
	for (l = g_list_first(srv->dest_list); l != NULL; l = g_list_next(l)) {
		d = (struct l7vs_dest *)l->data;
		if (d) {
			dest_cc += d->nactive;
			if (dest_cc + 1 > srv->sorry_cc) {
				return 1;
			}
		}
	}
	return 0;
}

/*!
 * Get sorry-server destination or old real-server destination.
 * @param[in]	*srv		service pointer
 * @param[in]	*conn		connection pointer
 * @param[in]	reverse		specification which get sorry-server or old real-server
 * @return	l7vs_dest*	sorry-server or old real-server destination
 */
struct l7vs_dest *
l7vs_sched_sorry_dest(struct l7vs_service *srv, struct l7vs_conn *conn, int reverse)
{
	// argument check
	if (reverse != 0 && reverse != 1) {
		VANESSA_LOGGER_ERR("Invalid reverse value");
		return NULL;
	}
	if (!reverse) {
		// argument check
		if (!srv) {
			VANESSA_LOGGER_ERR("Argument srv is NULL pointer");
			return NULL;
		}
		// return sorry-server destination
		return srv->sorry_dest;
	} else {
		// argument check
		if (!conn) {
			VANESSA_LOGGER_ERR("Argument conn is NULL pointer");
			return NULL;
		}
		// return old real-server destination
		return conn->old_dest;
	}
}
